#ifndef AhcReadout_HH
#define AhcReadout_HH

#include <iostream>
#include <fstream>

#include "CrcReadout.hh"
#include "CrcLocationData.hh"
#include "AhcVfeStartUpDataFine.hh"
#include "AhcVfeStartUpDataCoarse.hh"


class AhcReadout : public CrcReadout {

public:
  AhcReadout(unsigned b) : CrcReadout(b,0xac) {
  }

  AhcReadout(unsigned b, unsigned char c) : CrcReadout(b,c) {
  }

  virtual ~AhcReadout() {
  }

  virtual bool record(RcdRecord &r) {
    if(doPrint(r.recordType())) {
      std::cout << "AhcReadout::record()" << std::endl;
      r.RcdHeader::print(std::cout," ") << std::endl;
    }

    SubInserter inserter(r);

    UtlPack tid;
    tid.halfWord(1,SubHeader::ahc);
    tid.byte(2,_location.crateNumber());
    DaqTwoTimer *t(inserter.insert<DaqTwoTimer>(true));
    t->timerId(tid.word());


    if(r.recordType()==RcdHeader::startUp) {
      assert(doAhcVfeStartUpDataFine(r,true));
    }


    if(r.recordType()==RcdHeader::configurationStart) {
      assert(doAhcVfeStartUpDataFine(r,false));

      SubAccessor accessor(r);

      std::vector<const CrcReadoutConfigurationData*>
	bc(accessor.access< CrcReadoutConfigurationData>());
      
      for(unsigned j(0);j<bc.size();j++) {
	if(bc[j]->crateNumber()==_location.crateNumber()) {
	  _config=*(bc[j]);
	  if(doPrint(r.recordType(),1)) _config.print(std::cout," ") << std::endl;
	}
      }

      // First find 12-HAB values
      std::vector<const CrcLocationData<AhcVfeConfigurationDataFine>*>
	v12(accessor.access<CrcLocationData<AhcVfeConfigurationDataFine> >());
      
      // Copy to local array to save values
      _v12Config.clear();
      for(unsigned i(0);i<v12.size();i++) {
	_v12Config.push_back(*(v12[i]));
      }
      
      if(doPrint(r.recordType(),2)) {
	std::cout << " Number of AhcVfeConfigurationDataFine subrecords found = "
		  << _v12Config.size() << std::endl;

	for(unsigned i(0);i<_v12Config.size();i++) {
	  _v12Config[i].print(std::cout," ") << std::endl;
	}
      }

      // Now find 8-HAB values
      std::vector<const CrcLocationData<AhcVfeConfigurationDataCoarse>*> 
	v08(accessor.access<CrcLocationData<AhcVfeConfigurationDataCoarse> >());

      // Copy to local array to save values
      _v08Config.clear();
      for(unsigned i(0);i<v08.size();i++) {
	_v08Config.push_back(*(v08[i]));
      }
      
      if(doPrint(r.recordType(),2)) {
	std::cout << " Number of AhcVfeConfigurationDataCoarse subrecords found = "
		  << _v08Config.size() << std::endl;
	
	for(unsigned i(0);i<_v08Config.size();i++) {
	  _v08Config[i].print(std::cout," ") << std::endl;
	}
      }
      
      assert(doAhcVfeConfigurationDataFine(_v12Config,_v08Config,r));
    }

    if(r.recordType()==RcdHeader::runEnd) {
      assert(doAhcVfeStartUpDataFine(r,false));

      // Always ensure there is a broadcast default for 12-HABs
      if(_v12Config.size()==0) {
	CrcLocationData<AhcVfeConfigurationDataFine> defaultData(_location);
	defaultData.slotBroadcast(true);
	defaultData.crcComponent(CrcLocation::feBroadcast);
	defaultData.label(1);
	_v12Config.push_back(defaultData);
      }

      /*
      CrcLocationData<AhcVfeConfigurationDataCoarse> defaultDataCoarse(_location);
      defaultDataCoarse.slotBroadcast(true);
      defaultDataCoarse.crcComponent(CrcLocation::feBroadcast);

      std::vector<const CrcLocationData<AhcVfeConfigurationDataFine>*> v;
      v.push_back(&defaultData);
      */

      if(doPrint(r.recordType(),2)) {
	std::cout << " Number of AhcVfeConfigurationDataFine default subrecords set = "
		  << _v12Config.size() << std::endl;
	for(unsigned i(0);i<_v12Config.size();i++) {
	  _v12Config[i].print(std::cout," ") << std::endl;
	}
      }

      if(doPrint(r.recordType(),2)) {
	std::cout << " Number of AhcVfeConfigurationDataCoarse default subrecords set = "
		  << _v08Config.size() << std::endl;

	for(unsigned i(0);i<_v08Config.size();i++) {
	  _v08Config[i].print(std::cout," ") << std::endl;
	}
      }
      
      //assert(doAhcVfeConfigurationDataFine(v,r));

      /*
      std::vector<const CrcLocationData<AhcVfeConfigurationDataCoarse>*> v08;
      v08.push_back(&defaultDataCoarse);

      for(unsigned i(0);i<v08.size();i++) {
	if(doPrint(r.recordType(),2))
	  v08[i]->print(std::cout," ") << std::endl;
      }
      */

      assert(doAhcVfeConfigurationDataFine(_v12Config,_v08Config,r));
   }

    if(r.recordType()==RcdHeader::shutDown) {
      assert(doAhcVfeStartUpDataFine(r,false));
    }
 
    bool reply(CrcReadout::record(r));

    // Close off overall timer                                                                                                                
    t->setEndTime();
    if(doPrint(r.recordType())) t->print(std::cout," ") << std::endl;

    return reply;
  }


  bool doAhcVfeStartUpDataFine(RcdRecord &r, bool reset) {
    SubAccessor accessor(r);
    SubInserter inserter(r);

    AhcVfeConfigurationDataFine defaultData,dummyData;
    if(doPrint(r.recordType(),2)) defaultData.print(std::cout," ") << std::endl;

    std::vector<const CrcLocationData<AhcVfeStartUpDataFine>*>
      v(accessor.access<CrcLocationData<AhcVfeStartUpDataFine> >());
    
    CrcLocationData<AhcVfeStartUpDataFine> readData;

    for(unsigned i(0);i<v.size();i++) {      
      assert(v[i]->label()==1);
      assert(v[i]->crateNumber()==_crateNumber);
      assert(!v[i]->slotBroadcast());
      assert(v[i]->crcComponent()<=CrcLocation::fe7);

      unsigned s(v[i]->slotNumber());
      assert(s>=0 && s<=21);

      if(_device[s]!=0) {
	assert(_device[s]->writeAhcVfeStartUpData(v[i]->crcComponent(),
						  *(v[i]->data()),
						  *(readData.data()),reset));

	// Repeat at StartUp so as to get actual values in read
	if(reset)
	  assert(_device[s]->writeAhcVfeStartUpData(v[i]->crcComponent(),
						    *(v[i]->data()),
						    *(readData.data()),false));
	readData.location(v[i]->location());
	readData.label(2);
	inserter.insert(readData);
	
	if(doPrint(r.recordType(),2)) readData.print(std::cout," ") << std::endl;

	// Load known configuration value on startUp
	if(reset) {
	  assert(_device[s]->writeAhcVfeConfigurationData(v[i]->crcComponent(),
							  defaultData,dummyData));
	}
      }
    }

    // Now do the 8 HAB version for the back layers
    AhcVfeConfigurationDataCoarse defaultDataCoarse,dummyDataCoarse;
    if(doPrint(r.recordType(),2)) defaultDataCoarse.print(std::cout," ") << std::endl;

    std::vector<const CrcLocationData<AhcVfeStartUpDataCoarse>*>
      w(accessor.access<CrcLocationData<AhcVfeStartUpDataCoarse> >());
    
    CrcLocationData<AhcVfeStartUpDataCoarse> readDataCoarse;

    for(unsigned i(0);i<w.size();i++) {      
      assert(w[i]->label()==1);
      assert(w[i]->crateNumber()==_crateNumber);
      assert(!w[i]->slotBroadcast());
      assert(w[i]->crcComponent()<=CrcLocation::fe7);

      unsigned s(w[i]->slotNumber());
      assert(s>=0 && s<=21);

      if(_device[s]!=0) {
	assert(_device[s]->writeAhcVfeStartUpData(w[i]->crcComponent(),
						  *(w[i]->data()),
						  *(readDataCoarse.data()),reset));
	
	// Repeat at StartUp so as to get actual values in read
	if(reset)
	  assert(_device[s]->writeAhcVfeStartUpData(w[i]->crcComponent(),
						    *(w[i]->data()),
						    *(readDataCoarse.data()),false));
	readDataCoarse.location(w[i]->location());
	readDataCoarse.label(2);
	inserter.insert(readDataCoarse);
	
	if(doPrint(r.recordType(),2)) readDataCoarse.print(std::cout," ") << std::endl;

	// Load known configuration value on startUp
	if(reset) {
	  assert(_device[s]->writeAhcVfeConfigurationData(w[i]->crcComponent(),
							  defaultDataCoarse,dummyDataCoarse));
	}
      }
    }

    return true;
  }

  /*
  bool doAhcVfeConfigurationDataFine(std::vector<const CrcLocationData<AhcVfeConfigurationDataFine>*> &v, RcdRecord &r) {

    // Cannot write multiple times as lose previous values
    // Find single value for each slot/FE
    AhcVfeConfigurationDataFine writeData[22][8];

    // First look for slot and FE broadcast
    for(unsigned i(0);i<v.size();i++) {      
      if(v[i]->label()==1) {
	assert(v[i]->crateNumber()==_crateNumber);
	if(v[i]->slotBroadcast() && v[i]->crcComponent()==CrcLocation::feBroadcast) {
	  for(unsigned s(0);s<=21;s++) {
	    for(unsigned f(0);f<8;f++) {
	      writeData[s][f]=*(v[i]->data());
	    }
	  }
	}
      }
    }

    // Look for slot broadcast and not FE broadcast
    for(unsigned i(0);i<v.size();i++) {      
      if(v[i]->label()==1) {
	assert(v[i]->crateNumber()==_crateNumber);
	if(v[i]->slotBroadcast() && v[i]->crcComponent()<=CrcLocation::fe7) {
	  for(unsigned s(0);s<=21;s++) {
	    writeData[s][v[i]->crcComponent()]=*(v[i]->data());
	  }
	}
      }
    }
	  
    // Look for not slot broadcast and FE broadcast
    for(unsigned i(0);i<v.size();i++) {      
      if(v[i]->label()==1) {
	assert(v[i]->crateNumber()==_crateNumber);
	if(!v[i]->slotBroadcast() && v[i]->crcComponent()==CrcLocation::feBroadcast) {
	  unsigned s(v[i]->slotNumber());
	  assert(s>=0 && s<=21);

	  for(unsigned f(0);f<8;f++) {
	    writeData[s][f]=*(v[i]->data());
	  }
	}
      }
    }

    // Look for not slot broadcast and not FE broadcast
    for(unsigned i(0);i<v.size();i++) {
      if(v[i]->label()==1) {
	assert(v[i]->crateNumber()==_crateNumber);
	if(!v[i]->slotBroadcast() && v[i]->crcComponent()<=CrcLocation::fe7) {
	  unsigned s(v[i]->slotNumber());
	  assert(s>=0 && s<=21);

	  writeData[s][v[i]->crcComponent()]=*(v[i]->data());
	}
      }
    }

    // Now load
    SubInserter inserter(r);

    CrcLocationData<AhcVfeConfigurationDataFine> readData;
    readData.crateNumber(_crateNumber);

    if(doPrint(r.recordType(),1)) 
      _config.print(std::cout," ") << std::endl;

    for(unsigned s(2);s<=21;s++) {
      if(_device[s]!=0 && _config.slotEnable(s)) {
	for(unsigned f(0);f<8;f++) {
	  if(_config.slotFeEnable(s,f)) {
	    if(doPrint(r.recordType(),2)) writeData[s][f].print(std::cout," ") << std::endl;

	    assert(_device[s]->writeAhcVfeConfigurationData((CrcLocation::CrcComponent)f,
	                                                    writeData[s][f],
							    *(readData.data())));
	    readData.slotNumber(s);
	    readData.crcComponent((CrcLocation::CrcComponent)f);
	    readData.label(2);
	    inserter.insert(readData);
	    
	    if(doPrint(r.recordType(),2)) readData.print(std::cout," ") << std::endl;
	  }
	}
      }
    }

    return true;
  }
  */

  bool doAhcVfeConfigurationDataFine(const std::vector< CrcLocationData<AhcVfeConfigurationDataFine>   > &v12, 
				     const std::vector< CrcLocationData<AhcVfeConfigurationDataCoarse> > &v08, 
				     RcdRecord &r) {

    // Cannot write multiple times as lose previous values
    // Find single value for each slot/FE
    bool write8[22][8];
    AhcVfeConfigurationDataFine  writeData12[22][8];
    AhcVfeConfigurationDataCoarse writeData08[22][8];

    // Default to standard 12 HABs, not 8 HABs
    for(unsigned i(0);i<22;i++) {      
      for(unsigned j(0);j<8;j++) {      
	write8[i][j]=false;
      }
    }

    // Do 12 HABs

    // First look for slot and FE broadcast
    for(unsigned i(0);i<v12.size();i++) {      
      if(v12[i].label()==1) {
	assert(v12[i].crateNumber()==_crateNumber);
	if(v12[i].slotBroadcast() && v12[i].crcComponent()==CrcLocation::feBroadcast) {
	  for(unsigned s(0);s<=21;s++) {
	    for(unsigned f(0);f<8;f++) {
	      writeData12[s][f]=*(v12[i].data());
	    }
	  }
	}
      }
    }

    // Look for slot broadcast and not FE broadcast
    for(unsigned i(0);i<v12.size();i++) {      
      if(v12[i].label()==1) {
	assert(v12[i].crateNumber()==_crateNumber);
	if(v12[i].slotBroadcast() && v12[i].crcComponent()<=CrcLocation::fe7) {
	  for(unsigned s(0);s<=21;s++) {
	    writeData12[s][v12[i].crcComponent()]=*(v12[i].data());
	  }
	}
      }
    }
	  
    // Look for not slot broadcast and FE broadcast
    for(unsigned i(0);i<v12.size();i++) {      
      if(v12[i].label()==1) {
	assert(v12[i].crateNumber()==_crateNumber);
	if(!v12[i].slotBroadcast() && v12[i].crcComponent()==CrcLocation::feBroadcast) {
	  unsigned s(v12[i].slotNumber());
	  assert(s>=0 && s<=21);

	  for(unsigned f(0);f<8;f++) {
	    writeData12[s][f]=*(v12[i].data());
	  }
	}
      }
    }

    // Look for not slot broadcast and not FE broadcast
    for(unsigned i(0);i<v12.size();i++) {
      if(v12[i].label()==1) {
	assert(v12[i].crateNumber()==_crateNumber);
	if(!v12[i].slotBroadcast() && v12[i].crcComponent()<=CrcLocation::fe7) {
	  unsigned s(v12[i].slotNumber());
	  assert(s>=0 && s<=21);

	  writeData12[s][v12[i].crcComponent()]=*(v12[i].data());
	}
      }
    }

    // Now do 8 HABs with no broadcasts
    for(unsigned i(0);i<v08.size();i++) {
      if(v08[i].label()==1) {
	assert(v08[i].crateNumber()==_crateNumber);
	assert(!v08[i].slotBroadcast() && v08[i].crcComponent()<=CrcLocation::fe7);
	unsigned s(v08[i].slotNumber());
	assert(s>=0 && s<=21);

	write8[s][v08[i].crcComponent()]=true;
	writeData08[s][v08[i].crcComponent()]=*(v08[i].data());
      }
    }


    // Now load
    SubInserter inserter(r);

    if(doPrint(r.recordType(),1)) {
      std::cout << " AhcReadout::doAhcVfeConfigurationDataFine() Writing data" << std::endl;
      _config.print(std::cout," ") << std::endl;
    }

    CrcLocationData<AhcVfeConfigurationDataFine> readData;
    readData.crateNumber(_crateNumber);

    for(unsigned s(2);s<=21;s++) {
      if(_device[s]!=0 && _config.slotEnable(s)) {
	for(unsigned f(0);f<8;f++) {
	  if(_config.slotFeEnable(s,f)) {
	    if(!write8[s][f]) {
	      if(doPrint(r.recordType(),2)) writeData12[s][f].print(std::cout," ") << std::endl;

	      assert(_device[s]->writeAhcVfeConfigurationData((CrcLocation::CrcComponent)f,
							      writeData12[s][f],
							      *(readData.data())));
	      readData.slotNumber(s);
	      readData.crcComponent((CrcLocation::CrcComponent)f);
	      readData.label(2);
	      inserter.insert(readData);
	      
	      if(doPrint(r.recordType(),2)) readData.print(std::cout," ") << std::endl;
	    }
	  }
	}
      }
    }

    CrcLocationData<AhcVfeConfigurationDataCoarse> readDataCoarse;
    readDataCoarse.crateNumber(_crateNumber);

    for(unsigned s(2);s<=21;s++) {
      if(_device[s]!=0 && _config.slotEnable(s)) {
	for(unsigned f(0);f<8;f++) {
	  if(_config.slotFeEnable(s,f)) {
	    if(write8[s][f]) {
	      if(doPrint(r.recordType(),2)) writeData08[s][f].print(std::cout," ") << std::endl;

	      assert(_device[s]->writeAhcVfeConfigurationData((CrcLocation::CrcComponent)f,
							      writeData08[s][f],
							      *(readDataCoarse.data())));
	      readDataCoarse.slotNumber(s);
	      readDataCoarse.crcComponent((CrcLocation::CrcComponent)f);
	      readDataCoarse.label(2);
	      inserter.insert(readDataCoarse);
	      
	      if(doPrint(r.recordType(),2)) readDataCoarse.print(std::cout," ") << std::endl;
	    }
	  }
	}
      }
    }

    return true;
  }


private:
  std::vector< CrcLocationData<AhcVfeConfigurationDataFine>   > _v12Config;
  std::vector< CrcLocationData<AhcVfeConfigurationDataCoarse> > _v08Config;
};

#endif
