#ifndef BmlHodUsbDevice_HH
#define BmlHodUsbDevice_HH

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <cstdio>

#include <iostream>
#include <iomanip>
#include <fstream>
#include <string>

#include "TermIOs.hh"

#ifndef PRINTOUT
#define PRINTOUT false
#endif

class BmlHodUsbDevice {

public:
  BmlHodUsbDevice(std::string d, bool s=true, std::string l="") 
    : _deviceName(d), _fileDescriptor(-1), _logFile(0) {

    if(l!="") {
      _writeNumber=0;
      _readNumber=0;
      _logFile=new ofstream(l.c_str());
      *_logFile << "Log file for BmlHodUsbDevice" << std::endl;
    }

    _fileDescriptor=open(_deviceName.c_str(),O_RDWR);
    if(_fileDescriptor < 0) {
      perror((std::string("BmlHodUsbDevice::setup  error opening device ")
	      +_deviceName).c_str());
    } else {
      if(s) setup();
    }
  }

  ~BmlHodUsbDevice() {
    close(_fileDescriptor);
    if(_logFile!=0) _logFile->close();
  }

  bool setup() {
    if(PRINTOUT) std::cout << "BmlHodUsbDevice::setup  for device " 
			   << _deviceName << std::endl;
    if(_fileDescriptor < 0) return false;

    TermIOs t;
    tcgetattr(_fileDescriptor,&t);

     t.c_cflag = 0;
    cfsetispeed(&t,B115200);
    cfsetospeed(&t,B115200);
    t.c_cflag |= CS8;
    t.c_cflag |= CREAD;
    t.c_cflag |= CLOCAL;
    //t.c_cflag |= CRTSCTS;
    
    t.c_lflag = 0x0;
    //t.c_lflag |= ICANON;
    //t.c_lflag |= ECHO;
    //t.c_lflag |= ECHOE;
    t.c_lflag &= ~ICANON;
    t.c_lflag &= ~ECHO;
    t.c_lflag &= ~ECHOE;
    
    t.c_iflag = 0;
    
    t.c_oflag = 0;
    //t.c_oflag |= OPOST;
    //t.c_oflag |= ONLCR;
     
    t.print(std::cout);

    tcsetattr(_fileDescriptor,TCSANOW,&t);

    if(PRINTOUT) std::cout << cfgetispeed(&t) << " " 
			   << cfgetospeed(&t) << std::endl;
    
    int err(999);
    
    err=tcflush(_fileDescriptor,TCIOFLUSH);
    if(err<0) {
      perror((std::string("BmlHodUsbDevice::setup  error flushing device ")
	      +_deviceName).c_str());
      return false;
    }
    
    err=tcflow(_fileDescriptor,TCOON);
    if(err<0) {
      perror((std::string("BmlHodUsbDevice::setup  error flowing device ")
	      +_deviceName).c_str());
      return false;
    }

    return true;
  }


  bool write(unsigned n, const unsigned char *c) {
    if(_fileDescriptor<0) return false;

    if(PRINTOUT) std::cout << std::endl << "BmlHodUsbDevice::write" << std::endl;

    int err=::write(_fileDescriptor,c,n);
    if(PRINTOUT) std::cout << "BmlHodUsbDevice::write  write sent " 
			   << err << " bytes" << std::endl;

    if(err!=n) {
      perror("BmlHodUsbDevice::write  error writing");
      if(tcflush(_fileDescriptor,TCIOFLUSH)<0) 
	perror("BmlHodUsbDevice::write  error flushing");
      return false;
    }

    if(_logFile!=0) {
      *_logFile << "Write " << _writeNumber << ": " << hex;
      for(unsigned i(0);i<n;i++) *_logFile << " " << std::setfill('0') 
					   << std::setw(2) << (unsigned)c[i];
      *_logFile << dec << std::endl;
    }
    _writeNumber++;

    return true;
  }
    
  bool read(unsigned n, unsigned char *c) {
    int err=::read(_fileDescriptor,c,n);
    if(PRINTOUT) std::cout << "BmlHodUsbDevice::read  read received " 
			   << err << " bytes" << std::endl;

    if(err!=n) {
      perror("BmlHodUsbDevice::read  error reading");
      if(tcflush(_fileDescriptor,TCIOFLUSH)<0) 
	perror("BmlHodUsbDevice::read  error flushing");
      return false;
    }

    if(_logFile!=0) {
      *_logFile << "Read  " << _readNumber << ": " << hex;
      for(unsigned i(0);i<n;i++) *_logFile << " " << std::setfill('0') 
					   << std::setw(2) << (unsigned)c[i];
      *_logFile << dec << std::endl;
    }
    _readNumber++;

    return true;
  }

  unsigned read(unsigned char *c) {
    int err=::read(_fileDescriptor,c,1024);
    if(PRINTOUT) std::cout << "BmlHodUsbDevice::read  read received " 
			   << err << " bytes" << std::endl;

    if(err<0) {
      perror("BmlHodUsbDevice::read  error reading");
      if(tcflush(_fileDescriptor,TCIOFLUSH)<0) 
	perror("BmlHodUsbDevice::read  error flushing");
      return false;
    }

    if(_logFile!=0) {
      *_logFile << "Read  " << _readNumber << ": " << hex;
      for(unsigned i(0);i<err;i++) *_logFile << " " << std::setfill('0') 
					   << std::setw(2) << (unsigned)c[i];
      *_logFile << dec << std::endl;
    }
    _readNumber++;

    return err;
  }

  std::ostream& print(std::ostream &o, std::string s="") const {
    o << s << "BmlHodUsbDevice::print()  Device " << _deviceName << std::endl;
    o << s << "  File descriptor = " << _fileDescriptor << std::endl;
    return o;
  }

private:
  std::string _deviceName;
  int _fileDescriptor;
  ofstream *_logFile;
  unsigned _writeNumber;
  unsigned _readNumber;
};

#endif
