#ifndef BmlLc1176Readout_HH
#define BmlLc1176Readout_HH

#include <iostream>
#include <fstream>

#include "RcdHeader.hh"
#include "RcdUserRW.hh"
#include "SubInserter.hh"
#include "SubAccessor.hh"

#include "DaqBusAdapter.hh"
#include "BmlLc1176VmeDevice.hh"


class BmlLc1176Readout : public RcdUserRW {

public:
  BmlLc1176Readout(unsigned b, unsigned char a=0x25) : 
    _busAdapter(0),
    _addressTableReader("online/hal/Lc1176VmeAddress.hal"),
    _addressTable("LC1176 VME Address Table",_addressTableReader),
    _device(0) {
    
    // Catch exceptions locally here in case PCI card not installed
    try {
      _busAdapter=new DaqBusAdapter(b);
    } catch ( std::exception e ) {
      _busAdapter=0;
    }
    
    if(_busAdapter!=0) {
      _device=new BmlLc1176VmeDevice(_addressTable,*_busAdapter,a);
      if(_device->alive()) {
	std::cout << "BmlLc1176Readout::ctor()  PCI card " << b
		  << ", VME base address 0x" << std::hex << (unsigned)a << std::dec 
		  << "0000 found alive" << std::endl;
      } else {
	delete _device;
	_device=0;
      }
    }

    /*
    if(_device==0) 
      std::cout << "BmlLc1176Readout::ctor()  PCI card " << b
		<< ", VME base address 0x00" << std::hex << (unsigned)a << std::dec 
		<< "000000 not found alive" << std::endl;
    */
  }
  
  virtual ~BmlLc1176Readout() {
  }
  
  bool record(RcdRecord &r) {
    if(doPrint(r.recordType())) {
      std::cout << "BmlLc1176Readout::record()" << std::endl;
      r.RcdHeader::print(std::cout," ") << std::endl;
    }
    
    // Check record type
    switch (r.recordType()) {
    
    // Run start and end are treated the same
    case RcdHeader::runStart:
    case RcdHeader::runEnd: {
      
      if(_device!=0) {
	SubInserter inserter(r);
	
	assert(_device->reset());
	
	BmlLc1176RunData *d(inserter.insert<BmlLc1176RunData>());
	assert(_device->readRunData(*d));
	if(doPrint(r.recordType(),1)) d->print(std::cout," ") << std::endl;
      }

      break;
    }
      
      // Configuration start is used to set up system
    case RcdHeader::configurationStart: {

      /*      
      SubInserter inserter(r);
      
      BmlLc1176ConfigurationData *c(inserter.insert<BmlLc1176ConfigurationData>(true));
      c->csr(0x8600);
      if(doPrint(r.recordType(),1)) c->print(std::cout," ") << std::endl;
      */

      _enable=false;
      _trigger=false;

      if(_device!=0) {
	/*
	_device->testMode();
	BmlLc1176ConfigurationData xxx;
	xxx.csr(0x8600);
	    assert(_device->writeConfigurationData(xxx));
	  assert(_device->clear());
	  for(unsigned i(0);i<10;i++) {
	    assert(_device->readConfigurationData(xxx));
	    xxx.print(std::cout,"CSR loop ");
	    sleep(1);
	  }
	*/

	SubAccessor extracter(r);
      
	std::vector<const BmlLc1176ConfigurationData*>
	  b(extracter.access<BmlLc1176ConfigurationData>());

	// Use presence of configuration data to enable/disable readout
	_enable=(b.size()==1 && b[0]->label()==1);

	if(_enable) {
	  if(doPrint(r.recordType(),1)) b[0]->print(std::cout," ") << std::endl;

	  BmlLc1176ConfigurationData d(*(b[0]));
	  _trigger=d.executeTestCycle();
	  d.executeTestCycle(false);

	  assert(_device->writeConfigurationData(d));
	  
	  assert(_device->clear());

	  // Now read back
	  SubInserter inserter(r);
	     
	  BmlLc1176ConfigurationData *c(inserter.insert<BmlLc1176ConfigurationData>());
	  assert(_device->readConfigurationData(*c));
	  c->label(0);

	  if(doPrint(r.recordType())) c->print(std::cout," ") << std::endl;

	  _nEvents=0;
	}	
      }

      break;
    }

      
      
      // Configuration end reads back setup
    case RcdHeader::configurationEnd: {
      
      if(_device!=0 && _enable) {
	SubInserter inserter(r);
	
	BmlLc1176ConfigurationData *b(inserter.insert<BmlLc1176ConfigurationData>());
	assert(_device->readConfigurationData(*b));
	b->label(0);

	if(doPrint(r.recordType(),1)) b->print(std::cout," ") << std::endl;
      }

      break;
    }
      
    case RcdHeader::acquisitionStart: {
      
      if(_device!=0 && _enable) {
	assert(_device->clear());
      }
      _nEvents=0;
      break;
    }
      
    case RcdHeader::trigger:
    case RcdHeader::triggerBurst: {
      
      if(_device!=0 && _enable) {
	if(_trigger) {
	  if(doPrint(r.recordType(),1)) 
	    std::cout << "Lc1176 soft trigger" << std::endl;
	  _device->softTrigger();
	}
      }
      break;
    }
      
    case RcdHeader::event: {
      
      if(_device!=0 && _enable) {
	SubInserter inserter(r);
	
	BmlLc1176EventData *b(inserter.insert<BmlLc1176EventData>());
	//assert(_device->readEventData(*b,_nEvents));
	_device->readEventData(*b,_nEvents); // Not assert for trigger debug tests 09/06/06
	inserter.extend(b->numberOfWords()*4);
	
	if(doPrint(r.recordType(),1)) b->print(std::cout," ") << std::endl;

	// Compare number of events stored to number read out and clear if needed
	if(b->csr().numberOfEvents()<=(_nEvents+1)) {
	  _nEvents=0;
	  assert(_device->clear());
	} else {
	  _nEvents++;
	}
      }
      
      break;
    }
      
    default: {
      break;
    }
    };
    
    return true;
  }
  
  
private:
  DaqBusAdapter *_busAdapter;
  HAL::VMEAddressTableASCIIReader _addressTableReader;
  HAL::VMEAddressTable _addressTable;
  
  BmlLc1176VmeDevice *_device;
  bool _enable;
  bool _trigger;

  unsigned _nEvents;
};

#endif
