#ifndef BmlLc1176VmeDevice_HH
#define BmlLc1176VmeDevice_HH

#include <iostream>
#include <vector>

// hal
#include "VMEDevice.hh"
#include "VMEAddressTable.hh"
#include "VMEBusAdapterInterface.hh"

// dual/inc/bml
#include "BmlLc1176RunData.hh"
#include "BmlLc1176ConfigurationData.hh"
#include "BmlLc1176EventData.hh"


class BmlLc1176VmeDevice : public HAL::VMEDevice {
public:
  BmlLc1176VmeDevice(HAL::VMEAddressTable &t,
		     HAL::VMEBusAdapterInterface &b,
		     unsigned char a=0x25)
    : HAL::VMEDevice(t,b,a<<16), _baseAddress(a) {
  }

  bool alive() {
    /*unsigned long*/ uint32_t value;
    try {
      UtlPack csr;
      read("CSRa",&value);
      //std::cout << "CSRa = " << printHex((unsigned)value) << std::endl;
      csr.byte(3,value);
      read("CSRb",&value);
      //std::cout << "CSRb = " << printHex((unsigned)value) << std::endl;
      csr.byte(2,value);
      read("CSRc",&value);
      //std::cout << "CSRc = " << printHex((unsigned)value) << std::endl;
      csr.byte(1,value);
      read("CSRd",&value);
      //std::cout << "CSRd = " << printHex((unsigned)value) << std::endl;
      csr.byte(0,value);

      read("CSR",&value);
      //std::cout << "CSR = " << printHex((unsigned)value) << std::endl;
      return UtlPack(value)==csr;

    } catch ( HAL::HardwareAccessException& e ) {
      //cout << "*** Exception occurred : " << e.what() << endl;
      return false;

    } catch ( std::exception e ) {
      //cout << "*** Unknown exception occurred" << endl;
      return false;
    }
  }

  bool reset() {
    return true;
  }

  bool softTrigger() {
    /*unsigned long*/ uint32_t value;
    read("CSRd",&value);
    write("CSRd",value|0x01);
    return true;
  }

  bool clear() {
    ///*unsigned long*/ uint32_t value;
    //read("CSR",&value);
    //write("CSR",value&0xe0ffffff);

    //write("CSRa",0);


    ///*unsigned long*/ uint32_t value;
    //read("CSR",&value);
    //std::cout << "Read1 CSR " << printHex((unsigned)value) << std::endl;
    write("CSRb",0x80);
    //read("CSR",&value);
    //std::cout << "Read2 CSR " << printHex((unsigned)value) << std::endl;


    return true;
  }

  bool readRunData(BmlLc1176RunData &d) {
    d.baseAddress(_baseAddress);
    return true;
  }

  bool writeConfigurationData(const BmlLc1176ConfigurationData &d) {
    write("CSRa",0);
    write("CSR",d.csr());
    write("CSR",d.csr()|0x00800000);
    return true;
  }

  bool readConfigurationData(BmlLc1176ConfigurationData &d) {
    /*unsigned long*/ uint32_t value;
    read("CSR",&value);
    d.csr(value);
    return true;
  }

  bool readEventData(BmlLc1176EventData &d, unsigned n=0) {

    // Zero number of extra words and store buffer number
    d.numberOfWords(0);
    d.bufferNumber(n);
    //d.bufferNumber(0); // KEEP FORWARDS COMPATIBLE FOR NOW!

    // Fill in CSR value and check event number is OK
    /*unsigned long*/ uint32_t value;
    read("CSR",&value);
    d.csr(value);
    if(n>=d.csr().numberOfEvents()) {
      std::cerr << "BmlLc1176VmeDevice::readEventData()  Requested event "
		<< n << " >= CSR event number " << d.csr().numberOfEvents() << std::endl;
      return false;
    }

    /*
    unsigned nEvents((value>>24)&0x1f);
    if((value&0x20000000)!=0) nEvents=32; // Catch FULL buffer
    //cout << "Number of events = " << nEvents << endl;
    if(n>=nEvents) return false;
    */

    // Now get the data with block transfers
    unsigned nWords(256);
    unsigned *p(d.data());

    for(unsigned i(0);i<4 && nWords==256;i++) {
      readBlock("DataBlt",256,(char*)(p+64*i),HAL::HAL_DO_INCREMENT,0x0400*n+256*i);
      for(unsigned j(0);j<64 && nWords==256;j++) {
	//std::cout << "Word " << 64*i+j << printHex(p[64*i+j]) << std::endl;

	// Check for last word flag
	if((p[64*i+j]&0x00800000)==0) {
	  nWords=64*i+j;
	  // Check for valid data
	  if((p[64*i+j]&0x00200000)==0) nWords++;
	}
      }
    }

    d.numberOfWords(nWords);

    return true;
  }




  bool testMode() {
    /*unsigned long*/ uint32_t value;

    write("CSR",0x000040f6);
    read("CSR",&value);
    std::cout << "CSR = " << printHex((unsigned)value) << std::endl;

    write("CSRb",0x80);
    write("CSRd",0xf7);

    for(unsigned i(0);i<10;i++) {
    sleep(1);
    read("CSR",&value);
    std::cout << "CSR = " << printHex((unsigned)value) << std::endl;
    }
    write("CSRa",0x00);

    for(unsigned j(0);j<32;j++) {
      std::cout << "Event " << j << std::endl;
    unsigned array[256];
    readBlock("DataBlt",256,(char*)(array    ),HAL::HAL_DO_INCREMENT    ,0x0400*j);
    readBlock("DataBlt",256,(char*)(array+ 64),HAL::HAL_DO_INCREMENT,256+0x0400*j);
    readBlock("DataBlt",256,(char*)(array+128),HAL::HAL_DO_INCREMENT,512+0x0400*j);
    readBlock("DataBlt",256,(char*)(array+192),HAL::HAL_DO_INCREMENT,768+0x0400*j);

    for(unsigned i(0);i<256;i++) {
      read("Data",&value,4*i+1024*j);
      std::cout << "Data = " << printHex((unsigned)value) << " = " << printHex((unsigned)array[i]) << std::endl;
    }
    }
    return true;
  }


  void print(std::ostream &o) {
    o << "BmlLc1176VmeDevice::print()" << std::endl;
    /*
    unsigned long value;
    read("FirmwareId",&value);
    o << hex << "  FirmwareId = 0x" << value << dec << std::endl;
    */
    o << std::endl;

  }

private:
  const unsigned char _baseAddress;
};

#endif
