#ifndef CrcReadout_HH
#define CrcReadout_HH

#include <iostream>
#include <fstream>

#include "VMEAddressTable.hh"
#include "VMEAddressTableASCIIReader.hh"
#include "DaqBusAdapter.hh"
#include "HardwareAccessException.hh"

#include "RcdHeader.hh"
#include "RcdUserRW.hh"

#include "SubAccessor.hh"
#include "SubInserter.hh"

#include "CrcVmeDevice.hh"
#include "CrcReadoutConfigurationData.hh"


//#define DOUBLE_READ_SOFTWARE_FIX


class CrcReadout : public RcdUserRW {

public:
  CrcReadout(unsigned b, unsigned char c) : 
    _pciCard(b), _location(c,0,0,0), _crateNumber(c), _busAdapter(0),
    _addressTableReader("online/hal/CrcVmeAddress.hal"),
    _addressTable("CRC VME Address Table",_addressTableReader) {

    for(unsigned i(0);i<=21;i++) _device[i]=0;

    // Catch exceptions locally here in case PCI card not installed
    try {
      _busAdapter=new DaqBusAdapter(b);
    } catch ( std::exception e ) {
      _busAdapter=0;
    }

    if(_busAdapter!=0) {

#ifdef SBC_BUS_ADAPTER
      // UCL SBC bus adapter can only handle one CRC
      for(unsigned i(TRG_SLOT);i<=TRG_SLOT;i++) {
#else
      for(unsigned i(5);i<=21;i++) {
#endif

#ifdef SPLIT_CRATE_CLUDGE
	if((c==0xec && (i>12 || i<8) ) || (c==0xac && (i<=12 && i>8))) {
       //	if((c==0xec && i<12) || (c==0xac && i>=12)) {
#endif

	  _device[i]=new CrcVmeDevice(_addressTable,*_busAdapter,i);
	  if(_device[i]->alive()) {
	    std::cout << "CrcReadout::ctor()  PCI card " << b
		      << " crate 0x" << std::hex << (unsigned)c << std::dec
		      << " slot " << std::setw(2)
		      << i << " found alive" << std::endl;
	  } else {
#ifndef SBC_BUS_ADAPTER
	    delete _device[i];
#endif
	    _device[i]=0;
	  }

#ifdef SPLIT_CRATE_CLUDGE
	} else {
	  std::cout << "CrcReadout::ctor()  PCI card " << b
		    << " crate 0x" << std::hex << (unsigned)c << std::dec
		    << " slot " << std::setw(2)
		    << i << " ignored!!!" << std::endl;
	}
#endif


      }
    }
    std::cout << std::endl;

    // Turn off informational messages
    assert(btInfoSet());
    /*
    std::ostringstream sout;
    sout << "sbs/src/bt_info -p BT_INFO_TRACE -u " << b;
    system(sout.str().c_str());
    system((sout.str()+" -v 24580").c_str());
    std::cout << std::endl;
    */

    // Set crate number of readout configuration
    _config.crateNumber(c);

    // Define label for multi-timer
    _tid.halfWord(1,SubHeader::crc);
    _tid.byte(2,_location.crateNumber());
  }

  virtual ~CrcReadout() {
    /* Seems to cause a core dump???
    for(unsigned i(2);i<=21;i++) {
      if(_device[i]!=0) delete _device[i];
    }
    if(_busAdapter!=0) delete _busAdapter;
    */
  }

  virtual bool record(RcdRecord &r) {
    if(doPrint(r.recordType())) {
      std::cout << "CrcReadout::record()" << std::endl;
      r.RcdHeader::print(std::cout," ") << std::endl;
    }
 
    SubInserter inserter(r);
    _daqMultiTimer=inserter.insert<DaqMultiTimer>(true);
    _daqMultiTimer->timerId(_tid.word());
    inserter.extend(32*sizeof(UtlTime));


    // Catch exceptions from HAL code
    try {

      // Check record type
      switch (r.recordType()) {

      case RcdHeader::startUp: {
	assert(reset());

#ifdef DOUBLE_READ_SOFTWARE_FIX
	for(unsigned i(0);i<=21;i++) {
	  _offset[i]=0;
	}
#endif
	break;
      }

      case RcdHeader::runStart: {
	assert(btInfoSet());

	// Access the DaqRunStart to get print level 
	SubAccessor accessor(r);
	std::vector<const DaqRunStart*> v(accessor.extract<DaqRunStart>());
	if(v.size()==1)  _printLevel=v[0]->runType().printLevel();

	assert(readRunData(r));
	assert(readSlowRunData(r));
	break;
      }

      case RcdHeader::runEnd:
      case RcdHeader::runStop: {
	assert(readRunData(r));
	assert(readSlowRunData(r));

	// Turn off all DACs
	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0) {
	    for(unsigned f(0);f<8;f++) {
	      assert(_device[i]->feZeroDacs((CrcLocation::CrcComponent)f));
	    }
	  }
	}
	break;
      }

	// Configuration start is used to set up system
      case RcdHeader::configurationStart: {
	assert(writeSlowConfigurationData(r));
	assert(readSlowConfigurationData(r));

	assert(writeConfigurationData(r));
	assert(readConfigurationData(r));

	break;
      }


	// Configuration end reads back setup
      case RcdHeader::configurationEnd:
      case RcdHeader::configurationStop: {
	assert(readSlowConfigurationData(r));

	assert(readConfigurationData(r));

	break;
      }

      case RcdHeader::acquisitionStart: {
	_nEvents=0;

	// Flush any data out
	unsigned char *array(new unsigned char[16*1024*1024]);

	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0) {
	  
	    // Handle BE
	    assert(_device[i]->clearBeFifos());
	    
	    // Handle FE (broadcast)
	    assert(_device[i]->feClearFifos());

	    // Really check for no data
	    assert(_device[i]->readVlinkEventData(*((CrcVlinkEventData*)array),true));
	  }
	}

	delete [] array;


	// Read trigger data after clearing
	assert(readTriggerData(r,true));

	break;
      }

      case RcdHeader::acquisitionEnd:
      case RcdHeader::acquisitionStop: {
	/*
	if(_config.vlinkSpill() && _nEvents>0) {
	  std::cout << "About to read " << _nEvents << " events from Vlink" << std::endl;

	  UtlTime start;
	  start.update();

	  for(unsigned i(0);i<_nEvents;i++) {
	    //readVlinkEventData(*(_events[i])); // MEMORY???
	  }

	  UtlTime finish;
	  finish.update();
	  std::cout << "Done " << _nEvents << " in " << (finish-start).deltaTime() << " sec, "
	       << _nEvents/(finish-start).deltaTime() << " events/sec, "
	       << (finish-start).deltaTime()/_nEvents << " secs/event" << std::endl;
	}

	// Check for any remaining Vlink data
	RcdArena arena;
	arena.deleteData();
	arena.recordType(RcdHeader::event);

	readVlinkEventData(arena);

	unsigned nw(0);
	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0 && _config.slotEnable(i)) nw+=3; // SubHeader, CrcLocation, No of Words = 0
	}

	if(arena.numberOfWords()>nw) {
	  std::cout << "ERROR ERROR ERROR" << std::endl;
	  arena.print(std::cout," ") << std::endl;
	}
	*/


	// Read trigger data before clearing
	assert(readTriggerData(r,true));

	/*
	// Flush any data out
	unsigned char *array(new unsigned char[16*1024*1024]);

	for(unsigned i(0);i<=21;i++) {
#ifdef DOUBLE_READ_SOFTWARE_FIX
	  _offset[i]+=_nEvents;
#endif

	  if(_device[i]!=0) {
	  
	    // Handle BE
	    assert(_device[i]->clearBeFifos());
	    
	    // Handle FE (broadcast)
	    assert(_device[i]->feClearFifos());

	    // Really check for no data
	    assert(_device[i]->readVlinkEventData(*((CrcVlinkEventData*)array),true));
	  }
	}

	delete [] array;

	*/



	break;
      }

      case RcdHeader::spillStart: {
	break;
      }

      case RcdHeader::spillEnd: {
	break;
      }
	/*
      case RcdHeader::spill: {
	_nEvents=0;
	_printLevel+=2;
	assert(readTriggerData(r));
	_printLevel-=2;
	break;
      }
	*/
      case RcdHeader::transferStart: {
	break;
      }

      case RcdHeader::transferEnd: {
	break;
      }

      case RcdHeader::trigger: {
	_nEvents++;
	assert(takeTrigger(r));
	assert(readTriggerData(r));
	break;
      }

      case RcdHeader::triggerBurst: {
	_nEvents++;
	assert(takeTrigger(r)); // NOT RIGHT FOR SOFT TRGS!
	assert(readTriggerData(r,true));
	break;
      }

      case RcdHeader::event: {
	//assert(takeTrigger(r));
	//assert(readTriggerData(r));
	assert(readEventData(r));

	/*
	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0 && _config.slotEnable(i)) {

	if(_config.vlinkSpill()) _events[_nEvents]=&r; // MEMORY???
	else {
	  readVlinkEventData(r);
	  //                    readVlinkEventData(r,99); // Test!
	}
	*/

	//_nEvents++;

	break;
      }

      case RcdHeader::shutDown: {
	break;
      }

	/*
      case RcdHeader::slowStart:
      case RcdHeader::slowEnd: {
        if(doPrint(r.recordType(),1)) {
	  std::cout << std::endl << "CrcReadout::record()" << std::endl;
          r.RcdHeader::print(std::cout," ") << std::endl;
        }

	assert(readStartupData(r));
	break;
      }
	*/

	/*
      case RcdHeader::slowControl: {
        if(doPrint(r.recordType(),1)) {
	  std::cout << std::endl << "CrcReadout::record()" << std::endl;
          r.RcdHeader::print(std::cout," ") << std::endl;
        }

	assert(writeConfigurationData(r));
	assert(readConfigurationData(r));
	break;
      }
	*/

      case RcdHeader::slowReadout: {
	assert(readSlowReadoutData(r));
	break;
      }

      default: {
	break;
      }
      };

      // Close off overall timer
      _daqMultiTimer->addTimer();
      if(doPrint(r.recordType(),1)) _daqMultiTimer->print(std::cout," ") << std::endl;

      return true;

    } catch ( HAL::HardwareAccessException& e ) {
      std::cout << "*** Exception occurred : " << e.what() << std::endl;
      std::cerr << "*** Exception occurred : " << e.what() << std::endl;
      
    } catch ( std::exception e ) {
      std::cout << "*** Unknown exception occurred" << std::endl;
      std::cerr << "*** Unknown exception occurred" << std::endl;
    }
    return false;
  }
  
///////////////////////////////////////////////////////////////////////////////

  bool btInfoSet() {

    // Turn off informational messages to /var/log/messages
    std::ostringstream sout;
    sout << "sbs/src/bt_info -p BT_INFO_TRACE -u " << _pciCard;
    system(sout.str().c_str());
    system((sout.str()+" -v 24580").c_str());
    std::cout << std::endl;

    return true;
  }


  bool reset() {
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0) {
	assert(_device[i]->reset());
	assert(_device[i]->beSoftReset());
	assert(_device[i]->beTrgSoftReset()); // Do for all, even if not actual trigger CRC
	assert(_device[i]->feSoftReset()); // This is an feBroadcast reset
      }
    }
    return true;
  }
  
  bool takeTrigger(const RcdRecord &r) {
    for(unsigned i(2);i<=21;i++) {
      if(_device[i]!=0 && _config.slotEnable(i)) {

	if(_config.beSoftTrigger()) {
	  if(doPrint(r.recordType(),1)) std::cout << " BE soft trigger for slot "
				      << std::setw(2) << i << std::endl;
	  assert(_device[i]->beSoftTrigger());
	}
	
	if(_config.feBroadcastSoftTrigger()) {
	  if(doPrint(r.recordType(),1)) std::cout << " FE broadcast soft trigger for slot "
				      << std::setw(2) << i << std::endl;
	  assert(_device[i]->feSoftTrigger());
	}
	
	for(unsigned f(0);f<8;f++) {
	  if(_config.slotFeEnable(i,f)) {
	    if(_config.feSoftTrigger(f)) {
	      if(doPrint(r.recordType(),1)) std::cout << " FE" << f << " soft trigger for slot "
					      << std::setw(2) << i << std::endl;
	      assert(_device[i]->feSoftTrigger((CrcLocation::CrcComponent)f));
	    }
	  }
	}
      }
    }

    _daqMultiTimer->addTimer();

    return true;
  }
  
  bool readTriggerData(RcdRecord &r, bool forceRead=false) {
    SubInserter inserter(r);
    
    // Handle VME
    if(forceRead || (_config.vmePeriod()>0 && (_nEvents%_config.vmePeriod())==0)) {
      for(unsigned i(0);i<=21;i++) {
	if(_device[i]!=0 && _config.slotEnable(i)) {
	  _location.slotNumber(_device[i]->slot());
	  _location.crcComponent(CrcLocation::vme);

	  CrcLocationData<CrcVmeEventData>
	    *p(inserter.insert< CrcLocationData<CrcVmeEventData> >());
	  p->location(_location);

	  assert(_device[i]->readVmeEventData(*p->data()));
	  if(doPrint(r.recordType(),1)) p->print(std::cout," ") << std::endl;
	}
      }
    }

    _daqMultiTimer->addTimer();

    // Handle BE
    if(forceRead || (_config.bePeriod()>0 && (_nEvents%_config.bePeriod())==0)) {
      for(unsigned i(0);i<=21;i++) {
	if(_device[i]!=0 && _config.slotEnable(i)) {
	  _location.slotNumber(_device[i]->slot());
	  _location.crcComponent(CrcLocation::be);
	  
	  CrcLocationData<CrcBeEventData>
	    *p(inserter.insert< CrcLocationData<CrcBeEventData> >());
	  p->location(_location);
	  
	  assert(_device[i]->readBeEventData(*p->data()));
	  if(doPrint(r.recordType(),1)) p->print(std::cout," ") << std::endl;
	}
      }

    } else if(_config.becPeriod()>0 && (_nEvents%_config.becPeriod())==0) {
      for(unsigned i(0);i<=21;i++) {
	if(_device[i]!=0 && _config.slotEnable(i)) {
	  _location.slotNumber(_device[i]->slot());
	  _location.crcComponent(CrcLocation::be);
	
	  CrcLocationData<CrcBeEventData>
	    *p(inserter.insert< CrcLocationData<CrcBeEventData> >());
	  p->location(_location);
	  
	  assert(_device[i]->readBeEventData(*p->data(),true));
	  if(doPrint(r.recordType(),1)) p->print(std::cout," ") << std::endl;
	}
      }
    }

    _daqMultiTimer->addTimer();
    
    // Handle FEs
    if(forceRead || (_config.fePeriod()>0 && (_nEvents%_config.fePeriod())==0)) {
      for(unsigned i(0);i<=21;i++) {
	if(_device[i]!=0 && _config.slotEnable(i)) {
	  _location.slotNumber(_device[i]->slot());

	  for(unsigned f(0);f<8;f++) {
	    if(_config.slotFeEnable(i,f)) {
	      _location.crcComponent((CrcLocation::CrcComponent)f);
	      
	      CrcLocationData<CrcFeEventData>
		*p(inserter.insert< CrcLocationData<CrcFeEventData> >());
	      p->location(_location);
	      
	      assert(_device[i]->readFeEventData(_location.crcComponent(),*p->data()));
	      if(doPrint(r.recordType(),1)) p->print(std::cout," ") << std::endl;
	    }
	  }
	}
      }
    }

    _daqMultiTimer->addTimer();
	
    return true;
  }
  
  bool readEventData(RcdRecord &r) {
    SubInserter inserter(r);

    //_config.print(std::cout," readEventData ");

    /*
    // Read into local arrays
    unsigned char *array(new unsigned char[22*16*1024]);
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0 && _config.slotEnable(i)) {
	//std::cout << "About to read slot " << i << std::endl;
	assert(_device[i]->readVlinkEventData(*((CrcVlinkEventData*)(array+i*16*1024)),_config.vlinkBlt(),_config.vlinkFlag()));
	//((CrcVlinkEventData*)(array+i*16*1024))->numberOfWords(1078);

#ifdef DOUBLE_READ_SOFTWARE_FIX
	const CrcVlinkEventData *p((const CrcVlinkEventData*)(array+i*16*1024));
	bool readAgain(false);

	// Check all FEs to see if any are out of synch
	for(unsigned fe(0);fe<8;fe++) {
	  const CrcVlinkFeData *fd(p->feData(fe));
	  if(fd!=0) {
	    if((fd->triggerCounter()+_offset[i])!=p->bufferNumber()) readAgain=true;
	  }
	}
	if(readAgain) {
	  std::ostringstream sout;
	  sout << "SLOT " << std::setw(2) << i << " OFFSET = " << _offset[i];

	  p->print(std::cerr,std::string("DOUBLE READ OLD ")+sout.str()) << std::endl;
	  p->print(std::cout,std::string("DOUBLE READ OLD ")+sout.str()) << std::endl;

	  assert(_device[i]->readVlinkEventData(*((CrcVlinkEventData*)(array+i*16*1024)),_config.vlinkBlt(),_config.vlinkFlag()));

	  ((const CrcVlinkEventData*)(array+i*16*1024))->print(std::cerr,std::string("DOUBLE READ NEW ")+sout.str()) << std::endl;
	  ((const CrcVlinkEventData*)(array+i*16*1024))->print(std::cout,std::string("DOUBLE READ NEW ")+sout.str()) << std::endl;

	  _offset[i]++;
	}
#endif

      }
    }
    */

    bool allFinished(false);
    bool finished[22];
    for(unsigned i(0);i<=21;i++) finished[i]=false;

    _location.crcComponent(CrcLocation::be);

    while(!allFinished) {
      for(unsigned i(0);i<=21;i++) {
	if(_device[i]!=0 && _config.slotEnable(i)) {

	  if(!finished[i]) {
	    _location.slotNumber(_device[i]->slot());
	    _location.label(0);
	    
	    // Make the object in the record
	    CrcLocationData<CrcVlinkEventData>
	      *p(inserter.insert< CrcLocationData<CrcVlinkEventData> >());
	    p->location(_location);
	    
	    
	    /*
	      memcpy(p->data()  ,array+i*16*1024,sizeof(CrcVlinkEventData));
	      memcpy(p->data()+1,array+i*16*1024+sizeof(CrcVlinkEventData),p->data()->numberOfWords()*4);
	    */
	    
	    // Do the read
	    assert(_device[i]->readVlinkEventData(*(p->data()),_config.vlinkBlt(),_config.vlinkFlag()));
	    inserter.extend(p->data()->numberOfWords()*4);
	    if(doPrint(r.recordType(),2)) p->print(std::cout," ") << std::endl;
	    if(doPrint(r.recordType(),3)) p->data()->print(std::cout," ",true);
	    finished[i]=true;
	
#ifdef DOUBLE_READ_SOFTWARE_FIX
	    // Check data structure
	    if(!p->data()->verify(true)) {
	      p->print(std::cout,"NOT VERIFIED  ") << std::endl;
	      
	      // Check all FEs to see if any are out of synch
	    } else {
	      bool noData(true);
	      for(unsigned fe(0);fe<8;fe++) {
		const CrcVlinkFeData *fd(p->data()->feData(fe));
		if(fd!=0) {
		  noData=false;
		  if(fd->triggerCounter()<_nEvents) {
		    std::cout << "DOUBLE Slot " << i << " FE" << fe << " trg counter = " << fd->triggerCounter()
			      << " nEvents = " << _nEvents << std::endl;
		    finished[i]=false;
		    p->label(4);
		  }
		}
	      }
	      
	      if(noData) {
		p->label(4);
	      }
	    }
	    
	    /*
	      if(readAgain) {
	      std::ostringstream sout;
	      sout << _nEvents;
	      p->print(std::cerr,std::string("DOUBLE READ OLD ")+sout.str()+"  ") << std::endl;
	      p->print(std::cout,std::string("DOUBLE READ OLD ")+sout.str()+"  ") << std::endl;
	      
	      p=inserter.insert< CrcLocationData<CrcVlinkEventData> >();
	      p->location(_location);
	      
	      // Do the read again
	      assert(_device[i]->readVlinkEventData(*(p->data()),_config.vlinkBlt(),_config.vlinkFlag()));
	      inserter.extend(p->data()->numberOfWords()*4);
	      p->print(std::cerr,"DOUBLE READ NEW  ") << std::endl;
	      p->print(std::cout,"DOUBLE READ NEW  ") << std::endl;
	      
	      if(doPrint(r.recordType(),1)) p->print(std::cout," ") << std::endl;
	      if(doPrint(r.recordType(),2)) p->data()->print(std::cout," ",true);
	      }
	    */
#endif
	  }

	} else {
	  finished[i]=true;
	}
      }
      
      allFinished=true;
      for(unsigned i(0);i<=21;i++) {
	if(!finished[i]) allFinished=false;
      }
    }
    
    _daqMultiTimer->addTimer();

    //delete [] array;
    
    return true;
  }

  bool readRunData(RcdRecord &r) {

    SubInserter inserter(r);

    // Loop over all the CRCs; ignor config enables as outside configuration
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0) {
	CrcLocation cl(_crateNumber,i,CrcLocation::vme);
	
	// Handle VME
	CrcLocationData<CrcVmeRunData> *d(inserter.insert< CrcLocationData<CrcVmeRunData> >());
	cl.crcComponent(CrcLocation::vme);
	d->location(cl);
	assert(_device[i]->readVmeRunData(*d->data()));
	//_device[i]->readVmeRunData(*d->data()); // Unreliable EPROM read!
	if(doPrint(r.recordType(),1)) d->print(std::cout," ") << std::endl;

	// Check for sensible firmware ID
	if((d->data()->firmwareId()&0xffff0000)!=0x21000000) {
	  std::cout << "CrcReadout::readRunData()  ERROR  VME firmware wrong!" << std::endl;
	  d->print(std::cout) << std::endl;
	  std::cerr << "CrcReadout::readRunData()  ERROR  VME firmware wrong!" << std::endl;
	  d->print(std::cerr) << std::endl;
	}
	
	// Handle BE
	CrcLocationData<CrcBeRunData> *b(inserter.insert< CrcLocationData<CrcBeRunData> >());
	cl.crcComponent(CrcLocation::be);
	b->location(cl);
	assert(_device[i]->readBeRunData(*b->data()));
	if(doPrint(r.recordType(),1)) b->print(std::cout," ") << std::endl;
	
	// Check for sensible firmware ID
	if((b->data()->firmwareId()&0xffff0000)!=0x12000000) {
	  std::cout << "CrcReadout::readRunData()  ERROR  BE firmware wrong!" << std::endl;
	  b->print(std::cout) << std::endl;
	  std::cerr << "CrcReadout::readRunData()  ERROR  BE firmware wrong!" << std::endl;
	  b->print(std::cerr) << std::endl;
	}
	
	// Handle BE-Trg
	CrcLocationData<CrcBeTrgRunData> *a(inserter.insert< CrcLocationData<CrcBeTrgRunData> >());
	cl.crcComponent(CrcLocation::beTrg);
	a->location(cl);
	assert(_device[i]->readBeTrgRunData(*a->data()));
	if(doPrint(r.recordType(),1)) a->print(std::cout," ") << std::endl;
	
	// Check for sensible firmware ID
	if((a->data()->firmwareId()&0xffff0000)!=0x12090000) {
	  std::cout << "CrcReadout::readRunData()  ERROR  BETrg firmware wrong!" << std::endl;
	  a->print(std::cout) << std::endl;
	  std::cerr << "CrcReadout::readRunData()  ERROR  BETrg firmware wrong!" << std::endl;
	  a->print(std::cerr) << std::endl;
	}
	
	// Handle FEs; ignor config enables as outside configuration
	for(unsigned f(0);f<8;f++) {
	  CrcLocationData<CrcFeRunData> *e(inserter.insert< CrcLocationData<CrcFeRunData> >());
	  cl.crcComponent((CrcLocation::CrcComponent)f);
	  e->location(cl);
	  assert(_device[i]->readFeRunData(cl.crcComponent(),*e->data()));
	  if(doPrint(r.recordType(),1)) e->print(std::cout," ") << std::endl;

	  // Check for sensible firmware ID
	  if((e->data()->firmwareId()&0xffff0000)!=0x0dff0000) {
	    std::cout << "CrcReadout::readRunData()  ERROR  FE firmware wrong!" << std::endl;
	    e->print(std::cout) << std::endl;
	    std::cerr << "CrcReadout::readRunData()  ERROR  FE firmware wrong!" << std::endl;
	    e->print(std::cerr) << std::endl;
	  }
	}
      }
    }
    
    return true;
  }

  bool writeConfigurationData(RcdRecord &r) {
    SubAccessor accessor(r);

    std::vector<const CrcReadoutConfigurationData*>
      bc(accessor.access< CrcReadoutConfigurationData>());
    
    if(bc.size()>0) {
      unsigned found(0);
      for(unsigned j(0);j<bc.size();j++) {
	if(bc[j]->crateNumber()==_crateNumber) {
	  found++;
    	  _config=*(bc[j]);
	  if(doPrint(r.recordType(),1)) _config.print(std::cout," ") << std::endl;
	}
      }
      //assert(found==1);
    }
    
    std::vector<const TrgReadoutConfigurationData*>
      tc(accessor.access< TrgReadoutConfigurationData>());
    
    if(tc.size()>0) {
      _trgConfig=*(tc[0]);
      if(doPrint(r.recordType(),1)) _trgConfig.print(std::cout," ") << std::endl;
    }
    
    //CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vme);
    
    // Handle VME
    // No CrcVmeConfigurationData
    
    // Handle BE
    std::vector<const CrcLocationData<CrcBeConfigurationData>*>
      b(accessor.access< CrcLocationData<CrcBeConfigurationData> >());
    
    // Check for consistency and store values for this crate
    std::vector<const CrcLocationData<CrcBeConfigurationData>*> bCrate;

    for(unsigned j(0);j<b.size();j++) {
      assert(b[j]->slotBroadcast() || b[j]->slotNumber()<=21);
      assert(b[j]->crcComponent()==CrcLocation::be);

      if(b[j]->crateNumber()==_location.crateNumber() && b[j]->label()==1) bCrate.push_back(b[j]);
    }
    
    if(doPrint(r.recordType(),1)) {
      std::cout << " Number of CrcBeConfigurationData write subrecords accessed for crate = "
		<< bCrate.size() << " (of " << b.size() << " total)" << std::endl << std::endl;
      for(unsigned i(0);i<bCrate.size();i++) {
	bCrate[i]->print(std::cout,"  ") << std::endl;
      }
    }

    // Look for configuration for all slots first
    for(unsigned j(0);j<bCrate.size();j++) {
      if(bCrate[j]->slotBroadcast()) {
	  
	// Loop over all the CRCs
	for(unsigned i(2);i<=21;i++) {
	  if(_device[i]!=0 && _config.slotEnable(i)) {
	      
	    // Override FE trigger and data enables
	    //CrcBeConfigurationData cbcd(*bCrate[j]->data());
	    //cbcd.feDataEnable(_config.slotFeEnables(i));
	    //cbcd.feTriggerEnable(_config.slotFeEnables(i));

	    // Now .OR. them with the existing values
	    CrcBeConfigurationData cbcd(*bCrate[j]->data());
	    cbcd.feDataEnable   (cbcd.feDataEnable()    | _config.slotFeEnables(i));
	    cbcd.feTriggerEnable(cbcd.feTriggerEnable() | _config.slotFeEnables(i));
	    



	    // FIRMWARE FIX!!! Write small amount of mplex ticks for each disabled FE
	    // so enable all here
	    /*
	  unsigned char xx;
	  xx=0xff;
	  if((cbcd.feDataEnable()&0x80)==0) {
	    xx=0x7f;
	    if((cbcd.feDataEnable()&0x40)==0) {
	      xx=0x3f;
	      if((cbcd.feDataEnable()&0x20)==0) {
		xx=0x1f;
		if((cbcd.feDataEnable()&0x10)==0) {
		  xx=0x0f;
		  if((cbcd.feDataEnable()&0x08)==0) {
		    xx=0x07;
		    if((cbcd.feDataEnable()&0x04)==0) {
		      xx=0x03;
		      if((cbcd.feDataEnable()&0x02)==0) {
			xx=0x01;
			if((cbcd.feDataEnable()&0x01)==0) {
			  xx=0x00;
			}
		      }
		    }
		  }
		}
	      }
	    }
	  }

	  cbcd.feDataEnable   (xx);
	  cbcd.feTriggerEnable(xx);
	    */



	    if(doPrint(r.recordType(),1)) {
	      std::cout << " Writing to slot " << std::setw(2) << i << " ";
	      cbcd.print(std::cout,"  ") << std::endl;
	    } 

	    assert(_device[i]->writeBeConfigurationData(cbcd));
	  }
	}
      }
    }
    
    // Now look for individual slot configurations to override the broadcast
    for(unsigned j(0);j<bCrate.size();j++) {
      if(!bCrate[j]->slotBroadcast()) {
	  
	unsigned i(bCrate[j]->slotNumber());
	if(_device[i]!=0 && _config.slotEnable(i)) {
	    
	  // Override FE trigger and data enables
	  //CrcBeConfigurationData cbcd(*bCrate[j]->data());
	  //cbcd.feDataEnable(_config.slotFeEnables(i));
	  //cbcd.feTriggerEnable(_config.slotFeEnables(i));

	  // Now .OR. them with the existing values
	  CrcBeConfigurationData cbcd(*bCrate[j]->data());
	  cbcd.feDataEnable   (cbcd.feDataEnable()    | _config.slotFeEnables(i));
	  cbcd.feTriggerEnable(cbcd.feTriggerEnable() | _config.slotFeEnables(i));
	  


	  // FIRMWARE FIX!!! Write small amount of mplex ticks for each disabled FE
	  // so enable all here up to last enabled
	  /*
	  unsigned char xx;
	  xx=0xff;
	  if((cbcd.feDataEnable()&0x80)==0) {
	    xx=0x7f;
	    if((cbcd.feDataEnable()&0x40)==0) {
	      xx=0x3f;
	      if((cbcd.feDataEnable()&0x20)==0) {
		xx=0x1f;
		if((cbcd.feDataEnable()&0x10)==0) {
		  xx=0x0f;
		  if((cbcd.feDataEnable()&0x08)==0) {
		    xx=0x07;
		    if((cbcd.feDataEnable()&0x04)==0) {
		      xx=0x03;
		      if((cbcd.feDataEnable()&0x02)==0) {
			xx=0x01;
			if((cbcd.feDataEnable()&0x01)==0) {
			  xx=0x00;
			}
		      }
		    }
		  }
		}
	      }
	    }
	  }

	  cbcd.feDataEnable   (xx);
	  cbcd.feTriggerEnable(xx);
	  */	    



	  if(doPrint(r.recordType(),1)) {
	    std::cout << " Writing to slot " << std::setw(2) << i << " ";
	    cbcd.print(std::cout,"  ") << std::endl;
	  } 

	  assert(_device[i]->writeBeConfigurationData(cbcd));
	}
      }
    }
    
    // Handle FEs; configuration data is different for ECAL and AHCAL
    writeFeConfigurationData(r);
    
    // Fake event data are the same for both; first set all fake event data off
    CrcFeFakeEventData fkData; // Defaults to no data
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0 && _config.slotEnable(i)) {
	for(unsigned f(0);f<8;f++) {
	  if(_config.slotFeEnable(i,f)) {
	    if(doPrint(r.recordType(),1)) fkData.print(std::cout," ") << std::endl;
	    assert(_device[i]->writeFeFakeEventData((CrcLocation::CrcComponent)f,fkData));
	  }
	}
      }
    }
    
    std::vector<const  CrcLocationData<CrcFeFakeEventData>*>
      fk(accessor.access< CrcLocationData<CrcFeFakeEventData> >());
    
    // Look for broadcast configuration for all slots first
    for(unsigned j(0);j<fk.size();j++) {
      if(fk[j]->crateNumber()==_crateNumber && fk[j]->slotBroadcast()) {
	if(doPrint(r.recordType(),1)) fk[j]->print(std::cout," ") << std::endl;
	
	// Loop over all the CRCs
	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0 && _config.slotEnable(i)) {
	    if(fk[j]->crcComponent()==CrcLocation::feBroadcast) {
	      for(unsigned f(0);f<8;f++) {
		if(_config.slotFeEnable(i,f)) {
		  assert(_device[i]->writeFeFakeEventData((CrcLocation::CrcComponent)f,*fk[j]->data()));
		}
	      }
	    } else {
	      assert(_device[i]->writeFeFakeEventData(fk[j]->crcComponent(),*fk[j]->data()));
	    }
	  }
	}
      }
    }
    
    // Next do broadcast configuration for all FEs
    for(unsigned j(0);j<fk.size();j++) {
      if(fk[j]->crateNumber()==_crateNumber && !fk[j]->slotBroadcast()) {
	assert(fk[j]->slotNumber()<=21);
	
	if(fk[j]->crcComponent()==CrcLocation::feBroadcast) {
	  if(_device[fk[j]->slotNumber()]!=0) {
	    if(doPrint(r.recordType(),1)) fk[j]->print(std::cout," ") << std::endl;
	    for(unsigned f(0);f<8;f++) {
	      if(_config.slotFeEnable(fk[j]->slotNumber(),f)) {
		assert(_device[fk[j]->slotNumber()]->writeFeFakeEventData((CrcLocation::CrcComponent)f,*fk[j]->data()));
	      }
	    }
	  }
	}
      }
    }
    
    // Finally do individual FE configurations
    for(unsigned j(0);j<fk.size();j++) {
      if(fk[j]->crateNumber()==_crateNumber) {
	if(!fk[j]->slotBroadcast() && fk[j]->crcComponent()!=CrcLocation::feBroadcast) {
	  assert(fk[j]->slotNumber()<=21);
	  
	  if(_device[fk[j]->slotNumber()]!=0 &&
	     _config.slotFeEnable(fk[j]->slotNumber(),fk[j]->componentNumber())) {
	    
	    if(doPrint(r.recordType(),1)) fk[j]->print(std::cout," ") << std::endl;
	    assert(_device[fk[j]->slotNumber()]->writeFeFakeEventData(fk[j]->crcComponent(),*fk[j]->data()));
	  }
	}
      }
    }
    
    
    
    // BE-TRG
    
    std::vector<const CrcLocationData<CrcBeTrgConfigurationData>*>
      bt(accessor.access< CrcLocationData<CrcBeTrgConfigurationData> >());
    
    if(doPrint(r.recordType(),1)) {
      std::cout << " Number of CrcBeTrgConfigurationData subrecords = "
		<< bt.size() << std::endl;
      for(unsigned i(0);i<bt.size();i++) {
	bt[i]->print(std::cout,"  ") << std::endl;
      }
    }

    // Consistency check
    for(unsigned j(0);j<bt.size();j++) {
      assert(bt[j]->crcComponent()==CrcLocation::beTrg);
      if(bt[j]->crateNumber()==_location.crateNumber()) {
	assert(bt[j]->label()==1);
      }
    }
	
    // Do slot broadcast first
    for(unsigned j(0);j<bt.size();j++) {
      if(bt[j]->crateNumber()==_location.crateNumber()) {
	if(bt[j]->slotBroadcast()) {
	  if(doPrint(r.recordType(),1)) bt[j]->print(std::cout," ") << std::endl;
	
	  // Loop over all the CRCs
	  for(unsigned i(0);i<=21;i++) {
	    if(_device[i]!=0 && _config.slotEnable(i)) {
	      assert(_device[i]->writeBeTrgConfigurationData(*bt[j]->data()));
	    }
	  }
	}
      }
    }
	  
    for(unsigned j(0);j<bt.size();j++) {
      if(bt[j]->crateNumber()==_location.crateNumber()) {
	if(!bt[j]->slotBroadcast()) {
	  if(doPrint(r.recordType(),1)) bt[j]->print(std::cout," ") << std::endl;

	  unsigned slot(bt[j]->slotNumber());

	  if(_device[slot]!=0 && _config.slotEnable(slot)) {
	    assert(_device[slot]->writeBeTrgConfigurationData(*bt[j]->data()));
	  }
	}
      }
    }

    return true;
  }

  bool readConfigurationData(RcdRecord &r) {

    _location.label(0);
    SubInserter inserter(r);

    // Handle VME
    // No CrcVmeConfigurationData
	
    if(doPrint(r.recordType(),1)) std::cout 
      << " Reading CrcBeConfigurationData subrecords" << std::endl << std::endl;

    // Loop over all the CRCs
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0 && _config.slotEnable(i)) {
	_location.slotNumber(i);

	// Handle BE
	_location.crcComponent(CrcLocation::be);
	CrcLocationData<CrcBeConfigurationData> *b(inserter.insert< CrcLocationData<CrcBeConfigurationData> >());
	b->location(_location);
	assert(_device[i]->readBeConfigurationData(*b->data()));
	if(doPrint(r.recordType(),1)) b->print(std::cout,"  ") << std::endl;
      }
    }
    
    if(doPrint(r.recordType(),1)) std::cout 
      << " Reading CrcBeTrgConfigurationData subrecords" << std::endl << std::endl;

    // Loop over all the CRCs
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0 && _config.slotEnable(i)) {
	_location.slotNumber(i);

	// Handle BE
	_location.crcComponent(CrcLocation::beTrg);
	CrcLocationData<CrcBeTrgConfigurationData> *b(inserter.insert< CrcLocationData<CrcBeTrgConfigurationData> >());
	b->location(_location);
	assert(_device[i]->readBeTrgConfigurationData(*b->data()));
	if(doPrint(r.recordType(),1)) b->print(std::cout,"  ") << std::endl;
	
      }
    }
    
	// Fake event ???
	
    // Handle FEs
    assert(readFeConfigurationData(r));
    
    return true;
  }
  
  virtual bool writeFeConfigurationData(RcdRecord &r) {
    SubAccessor extracter(r);

    std::vector<const CrcLocationData<CrcFeConfigurationData>*>
      femc(extracter.extract< CrcLocationData<CrcFeConfigurationData> >());




    // FIRMWARE FIX!!! Write for each disabled FE
    CrcFeConfigurationData xxx;
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0) {
	for(unsigned f(0);f<8;f++) {
	  if(!_config.slotFeEnable(i,f)) {
	    assert(_device[i]->writeFeConfigurationData((CrcLocation::CrcComponent)f,xxx));
	  }
	}
      }
    }
	    



    
    // First do slot broadcast
    for(unsigned j(0);j<femc.size();j++) {
      if(femc[j]->crateNumber()==_crateNumber && femc[j]->slotBroadcast()) {
	if(doPrint(r.recordType(),1)) femc[j]->print(std::cout," ") << std::endl;

	// Loop over all the CRCs
	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0) {
	    if(_config.slotEnable(i)) {
	      if(femc[j]->crcComponent()==CrcLocation::feBroadcast) {
		for(unsigned f(0);f<8;f++) {
		  if(_config.slotFeEnable(i,f)) {
		    assert(_device[i]->writeFeConfigurationData((CrcLocation::CrcComponent)f,*femc[j]->data()));
		  }
		}
	      } else {
		assert(_device[i]->writeFeConfigurationData(femc[j]->crcComponent(),*femc[j]->data()));
	      }
	    } else {
	      // If disabled via config, should disable triggers and event data
	    }
	  }
	}
      }
    }

    // Next do FE broadcast 
    for(unsigned j(0);j<femc.size();j++) {
      if(femc[j]->crateNumber()==_crateNumber && !femc[j]->slotBroadcast()) {
	assert(femc[j]->slotNumber()<=21);
	
	if(femc[j]->crcComponent()==CrcLocation::feBroadcast) {
	  
	  if(_device[femc[j]->slotNumber()]!=0) {
	    if(doPrint(r.recordType(),1)) femc[j]->print(std::cout," ") << std::endl;
	    for(unsigned f(0);f<8;f++) {
	      if(_config.slotFeEnable(femc[j]->slotNumber(),f)) {
		assert(_device[femc[j]->slotNumber()]->writeFeConfigurationData((CrcLocation::CrcComponent)f,*femc[j]->data()));
	      }
	    }
	  }
	}
      }
    }


    // Finally do individual FE configurations
    for(unsigned j(0);j<femc.size();j++) {
      if(femc[j]->crateNumber()==_crateNumber) {
	if(!femc[j]->slotBroadcast() && femc[j]->crcComponent()!=CrcLocation::feBroadcast) {

	  assert(femc[j]->slotNumber()<=21);
	  if(_device[femc[j]->slotNumber()]!=0) {

	    if(_config.slotFeEnable(femc[j]->slotNumber(),femc[j]->crcComponent())) {
	      
	      if(doPrint(r.recordType(),1)) femc[j]->print(std::cout," ") << std::endl;
	      assert(_device[femc[j]->slotNumber()]->writeFeConfigurationData(femc[j]->crcComponent(),*femc[j]->data()));
	    }
	  }
	}
      }
    }

    return true;
  }

  virtual bool readFeConfigurationData(RcdRecord &r) {
    SubInserter inserter(r);
    
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0 && _config.slotEnable(i)) {
	CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vme,0);
	
	for(unsigned f(0);f<8;f++) {
	  if(_config.slotFeEnable(i,f)) {
            CrcLocationData<CrcFeConfigurationData>
              *e(inserter.insert< CrcLocationData<CrcFeConfigurationData> >());

	    cl.crcComponent((CrcLocation::CrcComponent)f);
            e->location(cl);
            assert(_device[i]->readFeConfigurationData(cl.crcComponent(),*e->data()));

            if(doPrint(r.recordType(),1)) e->print(std::cout," ") << std::endl;
	  }
	}
      }
    }
    
    return true;
  }



  bool readSlowRunData(RcdRecord &r) {
    SubInserter inserter(r);
    
    // Loop over all the CRCs; ignor config enables as outside configuration
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0) {
	CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vmeAdm1025,0);

	// Handle VME
	cl.crcComponent(CrcLocation::vme);

	CrcLocationData<CrcAdm1025RunData> *a(inserter.insert< CrcLocationData<CrcAdm1025RunData> >());
	a->location(cl);
	assert(_device[i]->readAdm1025Run(*a->data()));
	if(doPrint(r.recordType(),1)) a->print(std::cout," ") << std::endl;

	CrcLocationData<CrcLm82RunData> *d(inserter.insert< CrcLocationData<CrcLm82RunData> >());
	d->location(cl);
	assert(_device[i]->readLm82Run(CrcLocation::vmeLm82,*d->data()));
	if(doPrint(r.recordType(),1)) d->print(std::cout," ") << std::endl;

	// Handle BE
	cl.crcComponent(CrcLocation::be);

	CrcLocationData<CrcLm82RunData> *b(inserter.insert< CrcLocationData<CrcLm82RunData> >());
	b->location(cl);
	assert(_device[i]->readLm82Run(cl.crcComponent(),*b->data()));
	if(doPrint(r.recordType(),1)) b->print(std::cout," ") << std::endl;

	// Handle FEs; ignor config enables as outside configuration
	for(unsigned f(0);f<8;f++) {
	  cl.crcComponent((CrcLocation::CrcComponent)f);

	  CrcLocationData<CrcLm82RunData> *e(inserter.insert< CrcLocationData<CrcLm82RunData> >());
	  e->location(cl);
	  //assert(_device[i]->readLm82Run(cl.crcComponent(),*e->data()));
	  _device[i]->readLm82Run(cl.crcComponent(),*e->data()); // No assert as prototypes don't respond
	  if(doPrint(r.recordType(),1)) e->print(std::cout," ") << std::endl;
	}
      }
    }

    return true;
  }

  virtual bool writeSlowConfigurationData(RcdRecord &r) {
    SubAccessor accessor(r);
    
    std::vector<const CrcLocationData<CrcAdm1025ConfigurationData>*>
      a(accessor.access< CrcLocationData<CrcAdm1025ConfigurationData> >());
    
    if(doPrint(r.recordType(),1)) std::cout << " CrcReadout::record()  size of CrcLocationData<CrcAdm1025ConfigurationData> vector = "
				  << a.size() << " for slowControl" << std::endl;
    
    
    // Check for slot broadcast
    for(unsigned j(0);j<a.size();j++) {
	if(doPrint(r.recordType(),1)) a[j]->print(std::cout," ") << std::endl;
	if(a[j]->label()==1) {
      assert(a[j]->crcComponent()==CrcLocation::vmeAdm1025);
      if(a[j]->crateNumber()==_crateNumber && a[j]->slotBroadcast()) {
	if(doPrint(r.recordType(),1)) a[j]->print(std::cout," ") << std::endl;
	
	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0) {
	    assert(_device[i]->writeAdm1025Configuration(*a[j]->data()));
	    }
	}
	}
      }
    }
    
    // Now do individual slots
    for(unsigned j(0);j<a.size();j++) {
	if(a[j]->label()==1) {
      if(a[j]->crateNumber()==_crateNumber && !a[j]->slotBroadcast()) {
	if(doPrint(r.recordType(),1)) a[j]->print(std::cout," ") << std::endl;

	assert(a[j]->slotNumber()<22);
	if(_device[a[j]->slotNumber()]!=0) {
	  assert(_device[a[j]->slotNumber()]->writeAdm1025Configuration(*a[j]->data()));
	}
	}
      }
    }
    
    std::vector<const CrcLocationData<CrcLm82ConfigurationData>*>
      b(accessor.access< CrcLocationData<CrcLm82ConfigurationData> >());
    
    if(doPrint(r.recordType(),1)) std::cout << "CrcReadout::record()  size of CrcLocationData<CrcLm82ConfigurationData> vector = "
				<< b.size() << " for slowControl" << std::endl;
    
    // Check for slot broadcast
    for(unsigned j(0);j<b.size();j++) {
      if(b[j]->crateNumber()==_crateNumber && b[j]->slotBroadcast()) {
	if(doPrint(r.recordType(),1)) b[j]->print(std::cout," ") << std::endl;
	
	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0) {
	    //assert(_device[i]->writeLm82Configuration(b[j]->crcComponent(),*b[j]->data()));
	    _device[i]->writeLm82Configuration(b[j]->crcComponent(),*b[j]->data()); // No assert as prototypes don't respond to FE
	  }
	}
      }
    }
    
    // Now do individual slots
    for(unsigned j(0);j<b.size();j++) {
      if(b[j]->crateNumber()==_location.crateNumber() && !b[j]->slotBroadcast()) {
	if(doPrint(r.recordType(),1)) b[j]->print(std::cout," ") << std::endl;
	
	assert(b[j]->slotNumber()<22);
	if(_device[b[j]->slotNumber()]!=0) {
	  //assert(_device[b[j]->slotNumber()]->writeLm82Configuration(b[j]->crcComponent(),*b[j]->data()));
	  _device[b[j]->slotNumber()]->writeLm82Configuration(b[j]->crcComponent(),*b[j]->data()); // No assert as prototypes don't respond to FE
	}
      }
    }

    return true;
  }

  bool readSlowConfigurationData(RcdRecord &r) {
    SubInserter inserter(r);
    
    // Loop over all the CRCs
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0) {
	CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vmeAdm1025,0);

	// Handle VME
	cl.crcComponent(CrcLocation::vme);

	CrcLocationData<CrcAdm1025ConfigurationData> *as(inserter.insert< CrcLocationData<CrcAdm1025ConfigurationData> >());
	as->location(cl);
	assert(_device[i]->readAdm1025Configuration(*as->data()));
	if(doPrint(r.recordType(),1)) as->print(std::cout," ") << std::endl;

	CrcLocationData<CrcLm82ConfigurationData> *ds(inserter.insert< CrcLocationData<CrcLm82ConfigurationData> >());
	ds->location(cl);
	assert(_device[i]->readLm82Configuration(CrcLocation::vmeLm82,*ds->data()));
	if(doPrint(r.recordType(),1)) ds->print(std::cout," ") << std::endl;

	// Handle BE
	cl.crcComponent(CrcLocation::be);

	CrcLocationData<CrcLm82ConfigurationData> *bs(inserter.insert< CrcLocationData<CrcLm82ConfigurationData> >());
	bs->location(cl);
	assert(_device[i]->readLm82Configuration(cl.crcComponent(),*bs->data()));
	if(doPrint(r.recordType(),1)) bs->print(std::cout," ") << std::endl;

	// Handle FEs
	for(unsigned f(0);f<8;f++) {
	  if(_config.slotFeEnable(i,f)) {
	    cl.crcComponent((CrcLocation::CrcComponent)f);

	    CrcLocationData<CrcLm82ConfigurationData> *es(inserter.insert< CrcLocationData<CrcLm82ConfigurationData> >());
	    es->location(cl);
	    //assert(_device[i]->readLm82Configuration(cl.crcComponent(),*es->data()));
	    _device[i]->readLm82Configuration(cl.crcComponent(),*es->data()); // No assert as prototypes don't respond
	    if(doPrint(r.recordType(),1)) es->print(std::cout," ") << std::endl;
	  }
	}
      }
    }

    return true;
  }

  bool readSlowReadoutData(RcdRecord &r) {
    SubInserter inserter(r);

    // Loop over all the CRCs
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0) {
	
	// Handle VME
	CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vmeAdm1025);
	  
	CrcLocationData<CrcAdm1025SlowReadoutData> *a(inserter.insert< CrcLocationData<CrcAdm1025SlowReadoutData> >());
	a->location(cl);
	assert(_device[i]->readAdm1025SlowReadout(*a->data()));
	if(doPrint(r.recordType(),1)) a->print(std::cout," ") << std::endl;
	
	cl.crcComponent(CrcLocation::vmeLm82);
	CrcLocationData<CrcLm82SlowReadoutData> *d(inserter.insert< CrcLocationData<CrcLm82SlowReadoutData> >());
	d->location(cl);
	assert(_device[i]->readLm82SlowReadout(cl.crcComponent(),*d->data()));
	if(doPrint(r.recordType(),1)) d->print(std::cout," ") << std::endl;
	
	// Handle BE
	cl.crcComponent(CrcLocation::be);
	CrcLocationData<CrcLm82SlowReadoutData> *b(inserter.insert< CrcLocationData<CrcLm82SlowReadoutData> >());
	b->location(cl);
	assert(_device[i]->readLm82SlowReadout(cl.crcComponent(),*b->data()));
	if(doPrint(r.recordType(),1)) b->print(std::cout," ") << std::endl;
	
	// Handle FEs
	for(unsigned f(0);f<8;f++) {
	  if(_config.slotFeEnable(i,f)) {
	    cl.crcComponent((CrcLocation::CrcComponent)f);
	    
	    CrcLocationData<CrcLm82SlowReadoutData> *e(inserter.insert< CrcLocationData<CrcLm82SlowReadoutData> >());
	    e->location(cl);
	    //assert(_device[i]->readLm82SlowReadout(cl.crcComponent(),*e->data()));
	    _device[i]->readLm82SlowReadout(cl.crcComponent(),*e->data()); // No assert as prototypes don't respond
	    if(doPrint(r.recordType(),1)) e->print(std::cout," ") << std::endl;
	  }
	}
      }
    }
    
    return true;
  }

protected:
  unsigned _pciCard;
  CrcLocation _location;
  unsigned char _crateNumber;
  DaqBusAdapter *_busAdapter;
  HAL::VMEAddressTableASCIIReader _addressTableReader;
  HAL::VMEAddressTable _addressTable;

  UtlPack _tid;
  DaqMultiTimer *_daqMultiTimer;

  CrcVmeDevice *_device[22];
  CrcReadoutConfigurationData _config;
  TrgReadoutConfigurationData _trgConfig;
  unsigned _nEvents;

#ifdef DOUBLE_READ_SOFTWARE_FIX
  unsigned _offset[22];
#endif

};

#endif
