#ifndef CrcReadout_HH
#define CrcReadout_HH

#include <iostream>
#include <fstream>

// hal
#include "VMEAddressTable.hh"
#include "VMEAddressTableASCIIReader.hh"
#include "SBS620x86LinuxBusAdapter.hh"
//nclude "VMEDummyBusAdapter.hh"
#include "HardwareAccessException.hh"

// dual/inc/rcd
#include "RcdHeader.hh"
#include "RcdUserRW.hh"

// dual/inc/sub
#include "SubInserter.hh"

// online/inc/crc
#include "CrcVmeDevice.hh"
#include "CrcReadoutStartupData.hh"
#include "CrcReadoutConfigurationData.hh"
#include "TrgReadoutStartupData.hh"


class CrcReadout : public RcdUserRW {

public:
  CrcReadout() : _crateNumber(0), _busAdapter(0),
    _addressTableReader("online/hal/CrcVmeAddress.hal"),
    _addressTable("CRC VME Address Table",_addressTableReader) {
    _trgSlot=99;
    for(unsigned i(0);i<=21;i++) _device[i]=0;
  }

  CrcReadout(unsigned char c) : _crateNumber(c), _busAdapter(0),
    _addressTableReader("online/hal/CrcVmeAddress.hal"),
    _addressTable("CRC VME Address Table",_addressTableReader) {
    _trgSlot=99;
    for(unsigned i(0);i<=21;i++) _device[i]=0;
  }

  virtual ~CrcReadout() {
  }

  void devices(unsigned trg=12) {
    
  }

  void device(CrcVmeDevice *c, bool trg=false) {
    if(c==0) return;

    unsigned slot(c->slot());
    if(slot>21) return;

    if(_printLevel>0) {
      if(_device[slot]==0) {
	std::cout << std::endl << std::endl 
		  << "CrcReadout::device()  CrcVmeDevice registered in slot ";
      } else {
	std::cout << std::endl << std::endl 
		  << "CrcReadout::device()  CrcVmeDevice replaced in slot ";
      }

      std::cout << slot;
      if(trg) std::cout << " as BE-Trg";
      std::cout << std::endl;
    }

    _device[slot]=c;



    if(trg) {
      _trgSlot=slot;

      // Define the poll object
      _poll.crateNumber(_crateNumber);
      _poll.slotNumber(slot);
      _poll.crcComponent(CrcLocation::beTrg);
      _poll.label(0);
    }

    for(unsigned f(0);f<8;f++) _feEnable[slot][f]=true;
  }

  virtual bool record(RcdRecord &r) {

    // Catch exceptions from HAL code
    try {

      // Print initial record
      unsigned pl(4);

      if(_printLevel>pl) {
	std::cout << std::endl << std::endl << "Initial record" << std::endl;
	r.RcdHeader::print(std::cout);
	if(_printLevel>pl+1) {
	  SubAccessor ex(r);
	  ex.print(std::cout);
	  if(_printLevel>pl+2) {
	    r.print(std::cout);
	  }
	}
	std::cout << std::endl << std::endl;
      }


      // Check record type
      switch (r.recordType()) {

      case RcdHeader::startUp: {

	SubAccessor accessor(r);

	std::vector<const CrcReadoutStartupData*>
	  v(accessor.access<CrcReadoutStartupData>());

	unsigned pciNumber(0);

	for(unsigned i(0);i<v.size();i++) {
	  if(v[i]->crateNumber()==_crateNumber) {
	    v[i]->print(std::cout,"==>> ");
	    pciNumber=v[i]->pciNumber();
	  }
	}

	// Catch exceptions locally here in case PCI card not installed
	try {
	  _busAdapter=new SBS620x86LinuxBusAdapter(pciNumber);
	} catch ( std::exception e ) {
	  _busAdapter=0;
	}

	if(_busAdapter!=0) {
	  for(unsigned i(2);i<=21;i++) {
	    _device[i]=new CrcVmeDevice(_addressTable,*_busAdapter,i);
	    if(_device[i]->alive()) {
	      std::cout << "CrcReadout::record()  Slot " << std::setw(2)
			<< i << " found alive" << std::endl;
	    } else {
	      delete _device[i];
	      _device[i]=0;
	    }
	  }
	}

	std::vector<const TrgReadoutStartupData*>
	  w(accessor.access<TrgReadoutStartupData>());

	for(unsigned i(0);i<w.size();i++) {
	  if(w[i]->crateNumber()==_crateNumber) {
	    w[i]->print(std::cout,"==>> ");
	    _trgSlot=w[i]->slotNumber();
	  }
	}

	// Define the poll object
	if(_trgSlot<22) {
	  _poll.crateNumber(_crateNumber);
	  _poll.slotNumber(_trgSlot);
	  _poll.crcComponent(CrcLocation::beTrg);
	  _poll.label(0);
	}

	assert(reset());
	assert(readRunData(r));
	assert(readStartupData(r));
	break;
      }

      case RcdHeader::runStart: {
	assert(reset());
	if(_trgSlot<22) assert(_device[_trgSlot]->beTrgSoftReset());
	// No break!
      }

      case RcdHeader::runEnd:
      case RcdHeader::runStop: {
	assert(readRunData(r));
	assert(readStartupData(r));
	assert(readSlowControlsData(r));

	SubInserter inserter(r);

	if(_trgSlot<22) {
	  CrcLocationData<CrcBeTrgRunData> *t(inserter.insert< CrcLocationData<CrcBeTrgRunData> >());
	  CrcLocation cl(_crateNumber,_trgSlot,CrcLocation::beTrg,0);
	  t->location(cl);
	  assert(_device[_trgSlot]->readBeTrgRunData(*t->data()));
	  if(_printLevel>2) t->print(std::cout);
	}

	break;
      }

	// Configuration start is used to set up system
      case RcdHeader::configurationStart: {

	SubAccessor accessor(r);

	std::vector<const CrcReadoutConfigurationData*>
	  bc(accessor.access< CrcReadoutConfigurationData>());
	assert(bc.size()==1);
	_config=*(bc[0]);
	if(_printLevel>2) _config.print(std::cout) << std::endl;
      
	//CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vme);
      
	// Handle VME
	// No CrcVmeConfigurationData

	// Handle BE
	std::vector<const CrcLocationData<CrcBeConfigurationData>*>
	  b(accessor.access< CrcLocationData<CrcBeConfigurationData> >());
      
	// Look for configuration for all slots first
	for(unsigned j(0);j<b.size();j++) {
	  assert(b[j]->crcComponent()==CrcLocation::be);
	  if(b[j]->crateNumber()==_crateNumber && b[j]->slotBroadcast()) {
	    if(_printLevel>2) b[j]->print(std::cout) << std::endl;

	    // Loop over all the CRCs
	    for(unsigned i(2);i<=21;i++) {
	      if(_device[i]!=0 && _config.slotEnable(i)) {

		// Override FE trigger and data enables
		CrcBeConfigurationData cbcd(*b[j]->data());
		cbcd.feDataEnable(_config.slotFeEnables(i));
		cbcd.feDataEnable(_config.slotFeEnables(i));

		assert(_device[i]->writeBeConfigurationData(cbcd));
	      }
	    }
	  }
	}

	// Now look for individual slot configurations to override the broadcast
	for(unsigned j(0);j<b.size();j++) {
	  if(b[j]->crateNumber()==_crateNumber && !b[j]->slotBroadcast()) {
	    assert(b[j]->slotNumber()<=21);
	    if(_device[b[j]->slotNumber()]!=0 && _config.slotEnable(b[j]->slotNumber())) {

	      if(_printLevel>2) b[j]->print(std::cout) << std::endl;

	      // Override FE trigger and data enables
	      CrcBeConfigurationData cbcd(*b[j]->data());
	      cbcd.feDataEnable(_config.slotFeEnables(b[j]->slotNumber()));
	      cbcd.feDataEnable(_config.slotFeEnables(b[j]->slotNumber()));
	    
	      assert(_device[b[j]->slotNumber()]->writeBeConfigurationData(cbcd));
	    }
	  }
	}
    
	/*  
	    CrcBeConfigurationData r;
	    assert(_device[i]->readBeConfigurationData(r));
	    //if(r!=*b[j]->data()) {
	    r.print(std::cout," ");
	    //}
	    */ 
   
	// Handle BE-Trg
	std::vector<const CrcLocationData<CrcBeTrgConfigurationData>*>
	  bt(accessor.access< CrcLocationData<CrcBeTrgConfigurationData> >());
	assert(bt.size()<=1);

	for(unsigned j(0);j<bt.size() && _trgSlot<22;j++) {
	  assert(bt[j]->crcComponent()==CrcLocation::beTrg);
	  if(bt[j]->crateNumber()==_crateNumber) {

	    // Check if broadcast; if so, set for trigger slot
	    if(bt[j]->slotBroadcast()) {
	      if(_printLevel>2) bt[j]->print(std::cout);
	      assert(_device[_trgSlot]->writeBeTrgConfigurationData(*bt[j]->data()));

	      // If not broadcast, check slot numbers for match
	    } else {
	      assert(bt[j]->slotNumber()<=21);
	      if(_device[bt[j]->slotNumber()]!=0 &&
		 _trgSlot==bt[j]->slotNumber()) {
		if(_printLevel>2) bt[j]->print(std::cout);
		assert(_device[bt[j]->slotNumber()]->writeBeTrgConfigurationData(*bt[j]->data()));
	      }
	    }
	  }
	}
      
	// Handle FEs; configuration data is different for ECAL and AHCAL
	writeFeConfigurationData(r);


	// Fake event data are the same for both; first set all fake event data off
	CrcFeFakeEventData fkData; // Defaults to no data
	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0 && _config.slotEnable(i)) {
	    for(unsigned f(0);f<8;f++) {
	      if(_config.slotFeEnable(i,f)) {
		if(_printLevel>2) fkData.print(std::cout) << std::endl;
		assert(_device[i]->writeFeFakeEventData((CrcLocation::CrcComponent)f,fkData));
	      }
	    }
	  }
	}

	std::vector<const  CrcLocationData<CrcFeFakeEventData>*>
	  fk(accessor.access< CrcLocationData<CrcFeFakeEventData> >());

	// Look for broadcast configuration for all slots first
	for(unsigned j(0);j<fk.size();j++) {
	  if(fk[j]->crateNumber()==_crateNumber && fk[j]->slotBroadcast()) {
	    if(_printLevel>2) fk[j]->print(std::cout) << std::endl;

	    // Loop over all the CRCs
	    for(unsigned i(0);i<=21;i++) {
	      if(_device[i]!=0 && _config.slotEnable(i)) {
		if(fk[j]->crcComponent()==CrcLocation::feBroadcast) {
		  for(unsigned f(0);f<8;f++) {
		    if(_config.slotFeEnable(i,f)) {
		      assert(_device[i]->writeFeFakeEventData((CrcLocation::CrcComponent)f,*fk[j]->data()));
		    }
		  }
		} else {
		  assert(_device[i]->writeFeFakeEventData(fk[j]->crcComponent(),*fk[j]->data()));
		}
	      }
	    }
	  }
	}
      
	// Next do broadcast configuration for all FEs
	for(unsigned j(0);j<fk.size();j++) {
	  if(fk[j]->crateNumber()==_crateNumber && !fk[j]->slotBroadcast()) {
	    assert(fk[j]->slotNumber()<=21);

	    if(fk[j]->crcComponent()==CrcLocation::feBroadcast) {
	      if(_device[fk[j]->slotNumber()]!=0) {
		if(_printLevel>2) fk[j]->print(std::cout) << std::endl;
		for(unsigned f(0);f<8;f++) {
		  if(_config.slotFeEnable(fk[j]->slotNumber(),f)) {
		    assert(_device[fk[j]->slotNumber()]->writeFeFakeEventData((CrcLocation::CrcComponent)f,*fk[j]->data()));
		  }
		}
	      }
	    }
	  }
	}

	// Finally do individual FE configurations
	for(unsigned j(0);j<fk.size();j++) {
	  if(fk[j]->crateNumber()==_crateNumber) {
	    if(!fk[j]->slotBroadcast() && fk[j]->crcComponent()!=CrcLocation::feBroadcast) {
	      assert(fk[j]->slotNumber()<=21);

	      if(_device[fk[j]->slotNumber()]!=0 &&
		 _config.slotFeEnable(fk[j]->slotNumber(),fk[j]->componentNumber())) {
	
		if(_printLevel>2) fk[j]->print(std::cout);
		assert(_device[fk[j]->slotNumber()]->writeFeFakeEventData(fk[j]->crcComponent(),*fk[j]->data()));
	      }
	    }
	  }
	}

	/*
	  CrcFeConfigurationData r;
	  assert(_device[i]->readFeConfigurationData(cl.crcComponent(),r));
	  //if(r!=*b[j]->data()) {
	  r.print(std::cout," ");
	  //}
	  */
      

	// Read back all configuration!

	assert(readConfigurationData(r));

	/*

	// Flush any data out
	for(unsigned i(0);i<=21;i++) {
	if(_device[i]!=0) {
	  
	// Handle BE
	assert(_device[i]->clearBeFifos());

	// Handle FE (broadcast)
	assert(_device[i]->clearFeFifos());
	}
	}

	assert(_device[_trgSlot]->clearBeTrgFifos());

	// Enables triggers
	assert(_device[_trgSlot]->beTrgForceBusy(false));
	*/

	_nEvents=0;
	break;
      }


	// Configuration end reads back setup
      case RcdHeader::configurationEnd:
      case RcdHeader::configurationStop: {

	/*
	  SubInserter inserter(r);

	  // Loop over all the CRCs
	  for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0) {
	  CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vme);

	  // Handle VME
	  // No CrcVmeConfigurationData

	  // Handle BE
	  CrcLocationData<CrcBeConfigurationData> *b(inserter.insert< CrcLocationData<CrcBeConfigurationData> >());
	  cl.crcComponent(CrcLocation::be);
	  b->location(cl);
	  assert(_device[i]->readBeConfigurationData(*b->data()));
	  if(_printLevel>2) b->print(std::cout);
	  
	  // Handle BE-Trg
	  if(i==_trgSlot) {
	  CrcLocationData<CrcBeTrgConfigurationData> *b(inserter.insert< CrcLocationData<CrcBeTrgConfigurationData> >());
	  cl.crcComponent(CrcLocation::beTrg);
	  b->location(cl);
	  assert(_device[i]->readBeTrgConfigurationData(*b->data()));
	  if(_printLevel>2) b->print(std::cout);
	  }

	  // Fake event ???

	  }
	  }

	  // Handle FEs
	  assert(readFeConfigurationData(r));
	*/

	assert(readConfigurationData(r));
	break;
      }

      case RcdHeader::acquisitionStart: {

	// Flush any data out
	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0 && _config.slotEnable(i)) {

	    // Handle BE
	    assert(_device[i]->clearBeFifos());

	    // Handle FE (broadcast)
	    assert(_device[i]->clearFeFifos());
	  }
	}

	if(_trgSlot<22) assert(_device[_trgSlot]->clearBeTrgFifos());

	// Enables triggers
	if(_trgSlot<22) assert(_device[_trgSlot]->beTrgForceBusy(false));

	_nEvents=0;
	break;
      }

      case RcdHeader::event: {

	SubInserter inserter(r);

	// First do any triggers
	if(_trgSlot<22) {
	  if(_config.clearBeTrgTrigger()) {
	    if(_printLevel>6) std::cout << "CrcReadout::record()  BE-Trg clear trigger for slot "
					<< std::setw(2) << _trgSlot << " in event" << std::endl;
	    assert(_device[_trgSlot]->clearBeTrgTrigger());
	  }

	  if(_config.beTrgSoftTrigger()) {
	    if(_printLevel>6) std::cout << "CrcReadout::record()  BE-Trg soft trigger for slot "
					<< std::setw(2) << _trgSlot << " in event" << std::endl;
	    assert(_device[_trgSlot]->beTrgSoftTrigger());
	  }
	}

	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0 && _config.slotEnable(i)) {
	    if(_config.beSoftTrigger()) {
	      if(_printLevel>6) std::cout << "CrcReadout::record()  BE soft trigger for slot "
					  << std::setw(2) << i << " in event" << std::endl;
	      assert(_device[i]->beSoftTrigger());
	    }

	    if(_config.feBroadcastSoftTrigger()) {
	      if(_printLevel>6) std::cout << "CrcReadout::record()  FE broadcast soft trigger for slot "
					  << std::setw(2) << i << " in event" << std::endl;
	      assert(_device[i]->feSoftTrigger());
	    }

	    for(unsigned f(0);f<8;f++) {
	      if(_config.slotFeEnable(i,f)) {
		if(_config.feSoftTrigger(f)) {
		  if(_printLevel>0) std::cout << "CrcReadout::record()  FE" << f << " soft trigger for slot "
					      << std::setw(2) << i << " in event" << std::endl;
		  assert(_device[i]->feSoftTrigger((CrcLocation::CrcComponent)f));
		}
	      }
	    }
	  }
	}

	// Poll for trigger
	if(_trgSlot<22) {
	  if(_config.clearBeTrgTrigger() && !_config.beTrgSoftTrigger()) {
	    if(_printLevel>6) std::cout << "CrcReadout::record()  BE-Trg poll trigger for slot "
					<< std::setw(2) << _trgSlot << " in event" << std::endl;
	    assert(_device[_trgSlot]->pollBeTrgTrigger(*(_poll.data()),_config.beTrgPollNumber()));
	    if(_printLevel>6) _poll.print(std::cout);
	    inserter.insert< CrcLocationData<CrcBeTrgPollData> >(_poll);
	  }
	}

	r.updateRecordTime();


	// Now do event readout

	// Loop over all the CRCs
	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0 && _config.slotEnable(i)) {

	    CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vme);
	  
	    // Handle VME
	    if(_config.vmePeriod()>0 && (_nEvents%_config.vmePeriod())==0) {
	      cl.crcComponent(CrcLocation::vme);
	      CrcLocationData<CrcVmeEventData> *bv(inserter.insert< CrcLocationData<CrcVmeEventData> >());
	      bv->location(cl);
	      assert(_device[i]->readVmeEventData(*bv->data()));
	      if(_printLevel>4) bv->print(std::cout);
	    }

	    // Handle BE
	    if(_config.bePeriod()>0 && (_nEvents%_config.bePeriod())==0) {
	      cl.crcComponent(CrcLocation::be);
	      CrcLocationData<CrcBeEventData> *b(inserter.insert< CrcLocationData<CrcBeEventData> >());
	      b->location(cl);
	      assert(_device[i]->readBeEventData(*b->data()));
	      if(_printLevel>4) b->print(std::cout);
	    }

	    // Handle BE-Trg
	    if(i==_trgSlot && _config.beTrgPeriod()>0 && (_nEvents%_config.beTrgPeriod())==0) {
	      cl.crcComponent(CrcLocation::beTrg);
	      CrcLocationData<CrcBeTrgEventData> *t(inserter.insert< CrcLocationData<CrcBeTrgEventData> >());
	      t->location(cl);
	      //assert(_device[i]->readBeTrgEventData(*t->data()));
	      _device[i]->readBeTrgEventData(*t->data(),_config.beTrgSquirt()); // Not assert because of FIFO
	      inserter.extend(t->data()->numberOfFifoWords()*4); // For Be-Trg FIFO
	      if(_printLevel>4) t->print(std::cout);
	    }

	    // Handle FEs
	    if(_config.fePeriod()>0 && (_nEvents%_config.fePeriod())==0) {
	      for(unsigned f(0);f<8;f++) {
		if(_config.slotFeEnable(i,f)) {
		  cl.crcComponent((CrcLocation::CrcComponent)f);
		
		  CrcLocationData<CrcFeEventData> *e(inserter.insert< CrcLocationData<CrcFeEventData> >());
		  e->location(cl);
		  assert(_device[i]->readFeEventData(cl.crcComponent(),*e->data()));
		  if(_printLevel>4) e->print(std::cout);
		}
	      }
	    }

	    // Handle Vlink/
	    /*
	      cl.crcComponent(CrcLocation::be);
	      CrcLocationData<CrcVlinkEventData> *l(inserter.insert< CrcLocationData<CrcVlinkEventData> >());
	      l->location(cl);
	      assert(_device[i]->readVlinkEventData(*l->data(),true));
	      inserter.extend(l->data()->numberOfWords()*4);
	      if(_printLevel>4) l->print(std::cout);
	    */
	  }
	}

	if(_config.vlinkSpill()) _events[_nEvents]=&r;
	else {
	  readVlinkEventData(r);
	  //                    readVlinkEventData(r,99); // Test!
	}
	_nEvents++;

	break;
      }

      case RcdHeader::acquisitionEnd:
      case RcdHeader::acquisitionStop: {

	if(_config.vlinkSpill() && _nEvents>0) {
	  std::cout << "About to read " << _nEvents << " events from Vlink" << std::endl;

	  UtlTime start;
	  start.update();

	  for(unsigned i(0);i<_nEvents;i++) {
	    readVlinkEventData(*(_events[i]));
	  }

	  UtlTime finish;
	  finish.update();
	  std::cout << "Done " << _nEvents << " in " << (finish-start).deltaTime() << " sec, "
	       << _nEvents/(finish-start).deltaTime() << " events/sec, "
	       << (finish-start).deltaTime()/_nEvents << " secs/event" << std::endl;
	}

	// Check for any remaining Vlink data
	RcdArena arena;
	arena.deleteData();
	arena.recordType(RcdHeader::event);

	readVlinkEventData(arena);

	unsigned nw(0);
	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0 && _config.slotEnable(i)) nw+=3; // SubHeader, CrcLocation, No of Words = 0
	}

	if(arena.numberOfWords()>nw) {
	  std::cout << "ERROR ERROR ERROR" << std::endl;
	  arena.print(std::cout);
	}

	break;
      }

      case RcdHeader::shutdown: {
	assert(readRunData(r));
	assert(readStartupData(r));
	break;
      }

      case RcdHeader::slowStart:
      case RcdHeader::slowEnd: {
	assert(readStartupData(r));
	break;
      }

      case RcdHeader::slowControl: {
	assert(writeSlowControlsData(r));
	assert(readSlowControlsData(r));
	break;
      }

      case RcdHeader::slowReadout: {
	assert(readSlowReadoutData(r));
	break;
      }

      default: {
	break;
      }
      };


      // Print final record
      if(_printLevel>pl) {
	std::cout << std::endl << std::endl << "Final record" << std::endl;
	r.RcdHeader::print(std::cout);
	if(_printLevel>pl+1) {
	  SubAccessor ex(r);
	  ex.print(std::cout);
	  if(_printLevel>pl+2) {
	    r.print(std::cout);
	  }
	}
	std::cout << std::endl << std::endl;
      }

      return true;

    } catch ( HardwareAccessException& e ) {
      std::cout << "*** Exception occurred : " << e.what() << std::endl;
      
    } catch ( std::exception e ) {
      std::cout << "*** Unknown exception occurred" << std::endl;
    }
    return false;
  }

///////////////////////////////////////////////////////////////////////////////////////

  bool reset() {
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0) {
	assert(_device[i]->reset());
	assert(_device[i]->beSoftReset());
	assert(_device[i]->feSoftReset()); // This is an feBroadcast reset
      }
    }
    return true;
  }
  
  bool readVlinkEventData(RcdRecord &r, unsigned char label=0) {
    SubInserter inserter(r);
    
    // Loop over all the CRCs
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0 && _config.slotEnable(i)) {
	CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::be,label);
	CrcLocationData<CrcVlinkEventData> *l(inserter.insert< CrcLocationData<CrcVlinkEventData> >());
	l->location(cl);
	assert(_device[i]->readVlinkEventData(*l->data(),_config.vlinkBlt()));
	inserter.extend(l->data()->numberOfWords()*4);
	if(_printLevel>4) l->print(std::cout);
	if(_printLevel>5) l->data()->print(std::cout,"",true);
      }
    }

    return true;
  }

  bool readRunData(RcdRecord &r) {

    SubInserter inserter(r);

    // Loop over all the CRCs
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0) {
	CrcLocation cl(_crateNumber,i,CrcLocation::vme);
	
	// Handle VME
	CrcLocationData<CrcVmeRunData> *d(inserter.insert< CrcLocationData<CrcVmeRunData> >());
	cl.crcComponent(CrcLocation::vme);
	d->location(cl);
	assert(_device[i]->readVmeRunData(*d->data())); // Unreliable EPROM read!
	//_device[i]->readVmeRunData(*d->data());
	if(_printLevel>1) d->print(std::cout);
	
	// Handle BE
	CrcLocationData<CrcBeRunData> *b(inserter.insert< CrcLocationData<CrcBeRunData> >());
	cl.crcComponent(CrcLocation::be);
	b->location(cl);
	assert(_device[i]->readBeRunData(*b->data()));
	if(_printLevel>1) b->print(std::cout);
	
	// Handle FEs
	for(unsigned f(0);f<8;f++) {
	  if(_config.slotFeEnable(i,f)) {
	    CrcLocationData<CrcFeRunData> *e(inserter.insert< CrcLocationData<CrcFeRunData> >());
	    cl.crcComponent((CrcLocation::CrcComponent)f);
	    e->location(cl);
	    assert(_device[i]->readFeRunData(cl.crcComponent(),*e->data()));
	    if(_printLevel>1) e->print(std::cout);
	  }
	}
      }
    }
    
    return true;
  }

  bool readConfigurationData(RcdRecord &r) {
    SubInserter inserter(r);

    // Loop over all the CRCs
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0 && _config.slotEnable(i)) {
	CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vme);
	
	// Handle VME
	// No CrcVmeConfigurationData
	
	// Handle BE
	CrcLocationData<CrcBeConfigurationData> *b(inserter.insert< CrcLocationData<CrcBeConfigurationData> >());
	cl.crcComponent(CrcLocation::be);
	b->location(cl);
	assert(_device[i]->readBeConfigurationData(*b->data()));
	if(_printLevel>2) b->print(std::cout);
	
	// Handle BE-Trg
	if(i==_trgSlot) {
	  CrcLocationData<CrcBeTrgConfigurationData> *b(inserter.insert< CrcLocationData<CrcBeTrgConfigurationData> >());
	  cl.crcComponent(CrcLocation::beTrg);
	  b->location(cl);
	  assert(_device[i]->readBeTrgConfigurationData(*b->data()));
	  if(_printLevel>2) b->print(std::cout);
	}
	
	// Fake event ???
	
      }
    }
    
    // Handle FEs
    assert(readFeConfigurationData(r));
    
    return true;
  }
  
  virtual bool writeFeConfigurationData(RcdRecord &r) {
    SubAccessor extracter(r);

    std::vector<const CrcLocationData<CrcFeConfigurationData>*>
      femc(extracter.extract< CrcLocationData<CrcFeConfigurationData> >());
    
    // First do slot broadcast
    for(unsigned j(0);j<femc.size();j++) {
      if(femc[j]->crateNumber()==_crateNumber && femc[j]->slotBroadcast()) {
	if(_printLevel>2) femc[j]->print(std::cout);

	// Loop over all the CRCs
	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0 && _config.slotEnable(i)) {
	    if(femc[j]->crcComponent()==CrcLocation::feBroadcast) {
	      for(unsigned f(0);f<8;f++) {
		if(_config.slotFeEnable(i,f)) {
		  assert(_device[i]->writeFeConfigurationData((CrcLocation::CrcComponent)f,*femc[j]->data()));
		}
	      }
	    } else {
	      assert(_device[i]->writeFeConfigurationData(femc[j]->crcComponent(),*femc[j]->data()));
	    }
	  }
	}
      }
    }

    // Next do FE broadcast 
    for(unsigned j(0);j<femc.size();j++) {
      if(femc[j]->crateNumber()==_crateNumber && !femc[j]->slotBroadcast()) {
	assert(femc[j]->slotNumber()<=21);
	
	if(femc[j]->crcComponent()==CrcLocation::feBroadcast) {
	  
	  if(_device[femc[j]->slotNumber()]!=0) {
	    if(_printLevel>2) femc[j]->print(std::cout) << std::endl;
	    for(unsigned f(0);f<8;f++) {
	      if(_config.slotFeEnable(femc[j]->slotNumber(),f)) {
		assert(_device[femc[j]->slotNumber()]->writeFeConfigurationData((CrcLocation::CrcComponent)f,*femc[j]->data()));
	      }
	    }
	  }
	}
      }
    }


    // Finally do individual FE configurations
    for(unsigned j(0);j<femc.size();j++) {
      if(femc[j]->crateNumber()==_crateNumber) {
	if(!femc[j]->slotBroadcast() && femc[j]->crcComponent()!=CrcLocation::feBroadcast) {
	  if(_config.slotFeEnable(femc[j]->slotNumber(),femc[j]->crcComponent())) {

	    if(_printLevel>2) femc[j]->print(std::cout);
	    assert(_device[femc[j]->slotNumber()]->writeFeConfigurationData(femc[j]->crcComponent(),*femc[j]->data()));
	  }
	}
      }
    }

    return true;
  }

  virtual bool readFeConfigurationData(RcdRecord &r) {
    SubInserter inserter(r);
    
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0 && _config.slotEnable(i)) {
	CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vme,0);
	
	for(unsigned f(0);f<8;f++) {
	  if(_config.slotFeEnable(i,f)) {
            CrcLocationData<CrcFeConfigurationData>
              *e(inserter.insert< CrcLocationData<CrcFeConfigurationData> >());

	    cl.crcComponent((CrcLocation::CrcComponent)f);
            e->location(cl);
            assert(_device[i]->readFeConfigurationData(cl.crcComponent(),*e->data()));

            if(_printLevel>4) e->print(std::cout);
	  }
	}
      }
    }
    
    return true;
  }



  bool readStartupData(RcdRecord &r) {
    SubInserter inserter(r);
    
    // Loop over all the CRCs
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0) {
	CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vmeAdm1025,0);

	// Handle VME
	cl.crcComponent(CrcLocation::vme);

	CrcLocationData<CrcAdm1025StartupData> *a(inserter.insert< CrcLocationData<CrcAdm1025StartupData> >());
	a->location(cl);
	assert(_device[i]->readAdm1025Startup(*a->data()));
	if(_printLevel>4) a->print(std::cout);

	//CrcLocationData<CrcAdm1025SlowControlsData> *as(inserter.insert< CrcLocationData<CrcAdm1025SlowControlsData> >());
	//as->location(cl);
	//assert(_device[i]->readAdm1025SlowControls(*as->data()));
	//if(_printLevel>4) as->print(std::cout);

	CrcLocationData<CrcLm82StartupData> *d(inserter.insert< CrcLocationData<CrcLm82StartupData> >());
	d->location(cl);
	assert(_device[i]->readLm82Startup(CrcLocation::vmeLm82,*d->data()));
	if(_printLevel>4) d->print(std::cout);

	//CrcLocationData<CrcLm82SlowControlsData> *ds(inserter.insert< CrcLocationData<CrcLm82SlowControlsData> >());
	//ds->location(cl);
	//assert(_device[i]->readLm82SlowControls(CrcLocation::vmeLm82,*ds->data()));
	//if(_printLevel>4) ds->print(std::cout);

	// Handle BE
	cl.crcComponent(CrcLocation::be);

	CrcLocationData<CrcLm82StartupData> *b(inserter.insert< CrcLocationData<CrcLm82StartupData> >());
	b->location(cl);
	assert(_device[i]->readLm82Startup(cl.crcComponent(),*b->data()));
	if(_printLevel>4) b->print(std::cout);

	//CrcLocationData<CrcLm82SlowControlsData> *bs(inserter.insert< CrcLocationData<CrcLm82SlowControlsData> >());
	//bs->location(cl);
	//assert(_device[i]->readLm82SlowControls(cl.crcComponent(),*bs->data()));
	//if(_printLevel>4) bs->print(std::cout);

	// Handle FEs
	for(unsigned f(0);f<8;f++) {
	  if(_config.slotFeEnable(i,f)) {
	    cl.crcComponent((CrcLocation::CrcComponent)f);

	    CrcLocationData<CrcLm82StartupData> *e(inserter.insert< CrcLocationData<CrcLm82StartupData> >());
	    e->location(cl);
	    //assert(_device[i]->readLm82Startup(cl.crcComponent(),*e->data()));
	    _device[i]->readLm82Startup(cl.crcComponent(),*e->data()); // No assert as prototypes don't respond
	    if(_printLevel>4) e->print(std::cout);

	    //CrcLocationData<CrcLm82SlowControlsData> *es(inserter.insert< CrcLocationData<CrcLm82SlowControlsData> >());
	    //es->location(cl);
	    ////assert(_device[i]->readLm82SlowControls(cl.crcComponent(),*es->data()));
	    //_device[i]->readLm82SlowControls(cl.crcComponent(),*es->data()); // No assert as prototypes don't respond
	    //if(_printLevel>4) es->print(std::cout);
	  }
	}
      }
    }

    return true;
  }

  virtual bool writeSlowControlsData(RcdRecord &r) {
    SubAccessor accessor(r);
    
    std::vector<const CrcLocationData<CrcAdm1025SlowControlsData>*>
      a(accessor.access< CrcLocationData<CrcAdm1025SlowControlsData> >());
    
    if(_printLevel>2) std::cout << "CrcReadout::record()  size of CrcLocationData<CrcAdm1025SlowControlsData> vector = "
				  << a.size() << " for slowControl" << std::endl;
    
    
    // Check for slot broadcast
    for(unsigned j(0);j<a.size();j++) {
      assert(a[j]->crcComponent()==CrcLocation::vme);
      if(a[j]->crateNumber()==_crateNumber && a[j]->slotBroadcast()) {
	if(_printLevel>2) a[j]->print(std::cout);
	
	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0) {
	    assert(_device[i]->writeAdm1025SlowControls(*a[j]->data()));
	    }
	}
      }
    }
    
    // Now do individual slots
    for(unsigned j(0);j<a.size();j++) {
      if(a[j]->crateNumber()==_crateNumber && !a[j]->slotBroadcast()) {
	if(_printLevel>2) a[j]->print(std::cout);

	assert(a[j]->slotNumber()<22);
	if(_device[a[j]->slotNumber()]!=0) {
	  assert(_device[a[j]->slotNumber()]->writeAdm1025SlowControls(*a[j]->data()));
	}
      }
    }
    
    std::vector<const CrcLocationData<CrcLm82SlowControlsData>*>
      b(accessor.access< CrcLocationData<CrcLm82SlowControlsData> >());
    
    if(_printLevel>2) std::cout << "CrcReadout::record()  size of CrcLocationData<CrcLm82SlowControlsData> vector = "
				<< b.size() << " for slowControl" << std::endl;
    
    // Check for slot broadcast
    for(unsigned j(0);j<b.size();j++) {
      if(b[j]->crateNumber()==_crateNumber && b[j]->slotBroadcast()) {
	if(_printLevel>2) b[j]->print(std::cout);
	
	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0) {
	    //assert(_device[i]->writeLm82SlowControls(b[j]->crcComponent(),*b[j]->data()));
	    _device[i]->writeLm82SlowControls(b[j]->crcComponent(),*b[j]->data()); // No assert as prototypes don't respond to FE
	  }
	}
      }
    }
    
    // Now do individual slots
    for(unsigned j(0);j<b.size();j++) {
      if(b[j]->crateNumber()==_crateNumber && !b[j]->slotBroadcast()) {
	if(_printLevel>2) b[j]->print(std::cout);
	
	assert(b[j]->slotNumber()<22);
	if(_device[b[j]->slotNumber()]!=0) {
	  //assert(_device[b[j]->slotNumber()]->writeLm82SlowControls(b[j]->crcComponent(),*b[j]->data()));
	  _device[b[j]->slotNumber()]->writeLm82SlowControls(b[j]->crcComponent(),*b[j]->data()); // No assert as prototypes don't respond to FE
	}
      }
    }

    return true;
  }

  bool readSlowControlsData(RcdRecord &r) {
    SubInserter inserter(r);
    
    // Loop over all the CRCs
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0) {
	CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vmeAdm1025,0);

	// Handle VME
	cl.crcComponent(CrcLocation::vme);

	CrcLocationData<CrcAdm1025SlowControlsData> *as(inserter.insert< CrcLocationData<CrcAdm1025SlowControlsData> >());
	as->location(cl);
	assert(_device[i]->readAdm1025SlowControls(*as->data()));
	if(_printLevel>4) as->print(std::cout);

	CrcLocationData<CrcLm82SlowControlsData> *ds(inserter.insert< CrcLocationData<CrcLm82SlowControlsData> >());
	ds->location(cl);
	assert(_device[i]->readLm82SlowControls(CrcLocation::vmeLm82,*ds->data()));
	if(_printLevel>4) ds->print(std::cout);

	// Handle BE
	cl.crcComponent(CrcLocation::be);

	CrcLocationData<CrcLm82SlowControlsData> *bs(inserter.insert< CrcLocationData<CrcLm82SlowControlsData> >());
	bs->location(cl);
	assert(_device[i]->readLm82SlowControls(cl.crcComponent(),*bs->data()));
	if(_printLevel>4) bs->print(std::cout);

	// Handle FEs
	for(unsigned f(0);f<8;f++) {
	  if(_config.slotFeEnable(i,f)) {
	    cl.crcComponent((CrcLocation::CrcComponent)f);

	    CrcLocationData<CrcLm82SlowControlsData> *es(inserter.insert< CrcLocationData<CrcLm82SlowControlsData> >());
	    es->location(cl);
	    //assert(_device[i]->readLm82SlowControls(cl.crcComponent(),*es->data()));
	    _device[i]->readLm82SlowControls(cl.crcComponent(),*es->data()); // No assert as prototypes don't respond
	    if(_printLevel>4) es->print(std::cout);
	  }
	}
      }
    }

    return true;
  }

  bool readSlowReadoutData(RcdRecord &r) {
    SubInserter inserter(r);

    // Loop over all the CRCs
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0) {
	
	// Handle VME
	CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vmeAdm1025);
	  
	CrcLocationData<CrcAdm1025SlowReadoutData> *a(inserter.insert< CrcLocationData<CrcAdm1025SlowReadoutData> >());
	a->location(cl);
	assert(_device[i]->readAdm1025SlowReadout(*a->data()));
	if(_printLevel>4) a->print(std::cout);
	
	cl.crcComponent(CrcLocation::vmeLm82);
	CrcLocationData<CrcLm82SlowReadoutData> *d(inserter.insert< CrcLocationData<CrcLm82SlowReadoutData> >());
	d->location(cl);
	assert(_device[i]->readLm82SlowReadout(cl.crcComponent(),*d->data()));
	if(_printLevel>4) d->print(std::cout);
	
	// Handle BE
	cl.crcComponent(CrcLocation::be);
	CrcLocationData<CrcLm82SlowReadoutData> *b(inserter.insert< CrcLocationData<CrcLm82SlowReadoutData> >());
	b->location(cl);
	assert(_device[i]->readLm82SlowReadout(cl.crcComponent(),*b->data()));
	if(_printLevel>4) b->print(std::cout);
	
	// Handle FEs
	for(unsigned f(0);f<8;f++) {
	  if(_config.slotFeEnable(i,f)) {
	    cl.crcComponent((CrcLocation::CrcComponent)f);
	    
	    CrcLocationData<CrcLm82SlowReadoutData> *e(inserter.insert< CrcLocationData<CrcLm82SlowReadoutData> >());
	    e->location(cl);
	    //assert(_device[i]->readLm82SlowReadout(cl.crcComponent(),*e->data()));
	    _device[i]->readLm82SlowReadout(cl.crcComponent(),*e->data()); // No assert as prototypes don't respond
	    if(_printLevel>5) e->print(std::cout);
	  }
	}
      }
    }
    
    return true;
  }

protected:
  unsigned char _crateNumber;
  SBS620x86LinuxBusAdapter *_busAdapter;
  VMEAddressTableASCIIReader _addressTableReader;
  VMEAddressTable _addressTable;


  




  CrcVmeDevice *_device[22];
  bool _feEnable[22][8];
  unsigned _trgSlot;

  CrcLocationData<CrcBeTrgPollData> _poll;
  CrcReadoutConfigurationData _config;

  unsigned _nEvents;
  RcdRecord* _events[4*1024];
};

#endif
