#ifndef CrcVmeDevice_HH
#define CrcVmeDevice_HH

#include <iostream>
#include <vector>

// hal
#include "VMEDevice.hh"
#include "VMEAddressTable.hh"
#include "VMEBusAdapterInterface.hh"

// records/inc/crc
#include "CrcLocation.hh"
#include "CrcVmeRunData.hh"
#include "CrcVmeEventData.hh"
#include "CrcBeRunData.hh"
#include "CrcBeConfigurationData.hh"
#include "CrcBeEventData.hh"
#include "CrcBeTrgRunData.hh"
#include "CrcBeTrgConfigurationDataV0.hh"
#include "CrcBeTrgConfigurationDataV1.hh"
#include "CrcBeTrgConfigurationData.hh"
#include "CrcBeTrgEventData.hh"
#include "CrcBeTrgPollData.hh"
#include "CrcFeRunData.hh"
#include "CrcFeConfigurationData.hh"
#include "CrcFeFakeEventData.hh"
#include "CrcFeEventData.hh"
#include "CrcVlinkEventData.hh"
#include "CrcLm82RunData.hh"
#include "CrcLm82SlowReadoutData.hh"
#include "CrcLm82ConfigurationData.hh"
#include "CrcAdm1025RunData.hh"
#include "CrcAdm1025SlowReadoutData.hh"
#include "CrcAdm1025ConfigurationData.hh"

#include "TrgSpillPollData.hh"

#include "EmcFeConfigurationData.hh"

#include "AhcFeConfigurationDataV0.hh"
#include "AhcFeConfigurationData.hh"
#include "AhcVfeConfigurationDataFine.hh"
#include "AhcVfeConfigurationDataCoarse.hh"
#include "AhcVfeStartUpDataFine.hh"
#include "AhcVfeStartUpDataCoarse.hh"


// online/inc/crc
#include "CrcSerialHeader.hh"
#include "CrcSerialCommandI2CControl.hh"
#include "CrcSerialCommandWord.hh"
#include "CrcSerialCommandShort.hh"
#include "CrcSerialCommandByte.hh"
#include "CrcSerialCommandBit.hh"
#include "CrcSerialCommandFeAddress.hh"
#include "CrcSerialCommandBeCounter.hh"
#include "CrcSerialCommandBeBxCounter.hh"
#include "CrcSerialCommandBeDaqId.hh"
#include "CrcSerialCommandBeQdrDataCounter.hh"
#include "CrcSerialCommandBeControl.hh"
#include "CrcSerialCommandBeMode.hh"
#include "CrcSerialCommandBeTriggerSelect.hh"


bool serialPrint(false);
bool vlinkPrint(false);


class CrcVmeDevice : public HAL::VMEDevice {

public:
  CrcVmeDevice(HAL::VMEAddressTable &t, HAL::VMEBusAdapterInterface &b, unsigned a)
    : HAL::VMEDevice(t,b,a*0x10000), _slot(a), vlinkTotalLengthSum(0) {
  }

  unsigned slot() const {
    return _slot;
  }

  bool alive() {
    /*unsigned long*/ uint32_t value;
    try {
      value=0;
      read("FirmwareId",&value);
      read("FirmwareId",&value);
      read("FirmwareId",&value);
      /*
      std::cout << "CrcVmeDevice::alive()  Slot " << _slot
      	<< " gives VME firmware id "
      	<< printHex((unsigned)value,false) << std::endl;
      */
      return (value&0xfffff000)==0x21000000;

    } catch ( HAL::HardwareAccessException& e ) {
      //cout << "*** Exception occurred : " << e.what() << endl;
      return false;

    } catch ( std::exception e ) {
      //cout << "*** Unknown exception occurred" << endl;
      return false;
    }
  }

  void clockSelect(unsigned clock=0) {
    write("ClockSelect",clock);
  }

  bool reset(unsigned clock=0) {
    //for(unsigned i(0);i<10000;i++) 
    writePulse("SoftwareReset");
    write("ClockSelect",clock);
    vlinkTotalLengthSum=0;
    //sleep(1);
    usleep(100000);

    // Set clock phase between BE and FEs
    CrcSerialCommandWord w(CrcSerialHeader::be,CrcSerialHeader::beTest);
    //const CrcBeConfigurationData d;

    //w.data(d.test());
    w.data( 7<<4);
    if(!serialWrite(&w)) return false;    
    //w.data(d.test()+(1<<30));
    w.data((7<<4)+(1<<30));
    if(!serialWrite(&w)) return false;    
    //w.data(d.test());
    w.data( 7<<4);
    if(!serialWrite(&w)) return false;    

    feSoftReset();

    // System ace half-word access
    write("SystemAceBusModeReg",1);
    return true;
  }

  bool systemAceReset(void) {
    /*unsigned long*/ uint32_t n(0);
    UtlPack &u((UtlPack&)n);

    std::cout << std::endl << "systemAceReset: Starting ... " << std::endl;

    /* This may work now that u is setup correctly, will test oneday ...
    std::cout << std::endl << "Setting Command:FORCECFGMODE" << std::endl;
    write("SystemAceControlRegLSBs",0x08);

    systemAcePrint(std::cout);
    std::cout << "Trying to release lock" << std::endl;
    // Unset bit 1 of the control register to release lock
    read("SystemAceControlRegLSBs",&n);
    u.bit(0,false);
    u.bit(1,false);
    write("SystemAceControlRegLSBs",u.halfWord(0));

    // Wait for the lock to be released
    pollItem("SystemAceLockStatus",0x0,5000,&n);

    systemAcePrint(std::cout);
    
    std::cout << "Trying to get lock (back)" << std::endl;
    // Now get lock; set bit 1 of the control register to request lock
    read("SystemAceControlRegLSBs",&n);
    u.bit(1,true);
    write("SystemAceControlRegLSBs",u.word());

    // Wait for the lock request to be granted; this throw an exception if it times out
    pollItem("SystemAceLockStatus",0x2,5000,&n);
    */

    std::cout << std::endl << "Trying CFCMD:Abort(0x600)" << std::endl;
    write("SystemAceSecCntCmdReg",0x600);
    sleep(2);
    read("SystemAceSecCntCmdReg",&n); 
    std::cout << "SystemAceSecCntCmdReg" << printHex((unsigned)n) << std::endl;
   
    std::cout << "Resetting" << std::endl;

    write("SystemAceControlRegLSBs",0x00);
    write("SystemAceControlRegLSBs",0x80);
    //sleep(1);
    //write("SystemAceControlRegLSBs",0x08); // Don't boot from CF
    write("SystemAceControlRegLSBs",0x00);
    //sleep(5);

    std::cout << "Trying to get CF Lock" << std::endl;
    // Now get lock; set bit 1 of the control register to request lock
    read("SystemAceControlRegLSBs",&n);
    u.bit(1,true);
    write("SystemAceControlRegLSBs",u.word());
    // Wait for the lock request to be granted; this throw an exception if it times out
    pollItem("SystemAceLockStatus",0x2,5000,&n);


    return true;
  }

  bool systemAcePrint(std::ostream &o) {
    /*unsigned long*/ uint32_t n(0);
    UtlPack u(0);
    
    read("SystemAceBusModeReg",&n);
    std::cout << "SystemAceBusModeReg   " << printHex((unsigned)n) << std::endl;

    read("SystemAceStatusRegLSBs",&n);
    u.halfWord(0,n);
    read("SystemAceStatusRegMSBs",&n);
    u.halfWord(1,n);
    std::cout << "SystemAceStatusReg    " << printHex(u) << std::endl;

    read("SystemAceErrorRegLSBs",&n);
    u.halfWord(0,n);
    read("SystemAceErrorRegMSBs",&n);
    u.halfWord(1,n);
    std::cout << "SystemAceErrorReg     " << printHex(u) << std::endl;

    read("SystemAceCfgLbaRegLSBs",&n);
    u.halfWord(0,n);
    read("SystemAceCfgLbaRegMSBs",&n);
    u.halfWord(1,n);
    std::cout << "SystemAceCfgLbaReg    " << printHex(u) << std::endl;

    read("SystemAceMpuLbaRegLSBs",&n);
    u.halfWord(0,n);
    read("SystemAceMpuLbaRegMSBs",&n);
    u.halfWord(1,n);
    std::cout << "SystemAceMpuLbaReg    " << printHex(u) << std::endl;

    read("SystemAceSecCntCmdReg",&n);
    std::cout << "SystemAceSecCntCmdReg " << printHex((unsigned)n) << std::endl;

    read("SystemAceVersionReg",&n);
    std::cout << "SystemAceVersionReg   " << printHex((unsigned)n) << std::endl;

    read("SystemAceControlRegLSBs",&n);
    u.halfWord(0,n);
    read("SystemAceControlRegMSBs",&n);
    u.halfWord(1,n);
    std::cout << "SystemAceControlReg   " << printHex(u) << std::endl;

    read("SystemAceFatStatReg",&n);
    std::cout << "SystemAceFatStatReg   " << printHex((unsigned)n) << std::endl;

    return true;
  }

  bool systemAceReadCfImage(const std::string &f) {
    std::vector<unsigned short> v;
    if(!systemAceReadCfImage(v)) return false;
    unsigned short *p(&(v[0]));
    
    //int fd=::creat(f.c_str(),O_WRONLY|O_CREAT|O_TRUNC);
    int fd=::creat(f.c_str(),S_IRUSR|S_IRGRP|S_IROTH);
    //int fd=::creat(f.c_str(),O_WRONLY);

    if(fd<0) { 
	/*** */
      	std::cout << "systemAceWriteCfImage: Failed to open " << f << ". ABORTING!" << std::endl; 

	return false;
    }

    //for(unsigned i(0);i<v.size();i++) ::write(fd,p+i,2);
    int err=::write(fd,p,2*v.size());
    if(err<0) return false;

    err=::close(fd);
    if(err<0) return false;

    return true;
}

  bool systemAceReadCfImage(std::vector<unsigned short> &v) {
    /*unsigned long*/ uint32_t n(0);
    UtlPack &u((UtlPack&)n);

    // Get number of blocks (=0.5 kByte) to read
    UtlPack maxLba(0);
    read("SystemAceCfgLbaRegLSBs",&n);
    maxLba.halfWord(0,n&0xffff);
    read("SystemAceCfgLbaRegMSBs",&n);
    maxLba.halfWord(1,n&0xffff);

    if(maxLba.word()==0) maxLba.word(1);

    std::cout << std::endl << "CrcVmeDevice::systemAceReadCfImage()  Reading "
	      << std::setw(6) << maxLba.word() << " blocks total" << std::endl;

    // Need to lock/unlock and set up for each 256 blocks = 128kBytes
    for(unsigned lba(0);lba<maxLba.word();lba+=256) {

      unsigned nBlk(maxLba.word()-lba);
      if(nBlk>256) nBlk=256;
      std::cout << "CrcVmeDevice::systemAceReadCfImage()  Reading blocks "
		<< std::setw(6) << lba << " to "
		<< std::setw(6) << lba+nBlk-1 << std::endl;

      // Now get lock; set bit 1 of the control register to request lock
      read("SystemAceControlRegLSBs",&n);
      u.bit(1,true);
      write("SystemAceControlRegLSBs",u.word());

      // Wait for the lock request to be granted; this throw an exception if it times out
      pollItem("SystemAceLockStatus",0x2,5000,&n);

      

      // Wait until ready for CF controller command
      pollItem("SystemAceRdyForCfCmd",0x1,5000,&n);

      // Set the LBA value
      UtlPack uLba(lba);
      write("SystemAceMpuLbaRegLSBs",uLba.halfWord(0));
      write("SystemAceMpuLbaRegMSBs",uLba.halfWord(1));

      // Set the read command and number of blocks to transfer
      if(nBlk<256) write("SystemAceSecCntCmdReg",0x300+nBlk);
      else         write("SystemAceSecCntCmdReg",0x300);

      // Wait for data buffer ready
      pollItem("SystemAceDataBufRdy",0x1,5000,&n);

      /*
      bool empty(false);
      for(unsigned j(0);j<5000 && !empty;j++) {
	read("SystemAceStatusRegLSBs",&n);
	if(u.bit(5)) {
	  for(unsigned i(0);i<16;i++) {
	    read("SystemAceDataBufReg",&n);
	    v.push_back(u.halfWord(0));
	  }
	} else {
	  empty=true;
	  std::cout << "Empty set for j = " << j << std::endl;
	}
      }
      */
      // Read the blocks
      for(unsigned i(0);i<nBlk;i++) {

	// Each read is 2 bytes so need 256 reads for a block
	for(unsigned j(0);j<256;j++) {
	  read("SystemAceDataBufReg",&n);
	  v.push_back(u.halfWord(0));
	}
      }

      sleep(1);

      // Release lock; unset bit 1 of the control register to release lock
      read("SystemAceControlRegLSBs",&n);
      u.bit(0,false);
      u.bit(1,false);
      write("SystemAceControlRegLSBs",u.halfWord(0));

      // Wait for the lock to be released
      pollItem("SystemAceLockStatus",0x0,5000,&n);
    }

    return true;
  }

  bool systemAceWriteCfImage(const std::string &f) {
    int fd=::open(f.c_str(),O_RDONLY);
    if(fd<0) {
	/*** */
      std::cout << "systemAceWriteCfImage: Failed to open " << f << ". ABORTING!" << std::endl; 
    
      return false;
    }

    std::vector<unsigned short> v;
    unsigned short d;
    
    int err(2);
    while(err==2) {
      err=::read(fd,&d,2);
      if(err==2) v.push_back(d);
    }

    for(unsigned i(0);i<32;i++) {
      std::cout << "Word " << i << " = " << printHex(v[i]) << std::endl;
    }

    unsigned short *p(&(v[0]));
    for(unsigned i(0);i<32;i++) {
      std::cout << "Worp " << i << " = " << printHex(p[i]) << std::endl;
    }



    if(err<0) return false;

    err=::close(fd);
    if(err<0) return false;

    if(!systemAceWriteCfImage(v)) return false;
    return true;
  }

  bool systemAceWriteCfImage(const std::vector<unsigned short> &v) {
    /*unsigned long*/ uint32_t n(0);
    UtlPack &u((UtlPack&)n);

    // Get number of blocks (=0.5 kByte) to read
    UtlPack maxLba((2*v.size()+511)/512);

    std::cout << std::endl << "CrcVmeDevice::systemAceWriteCfImage()  Writing "
	      << std::setw(6) << maxLba.word() << " blocks total" << std::endl;

    // Need to lock/unlock and set up for each 256 blocks = 128kBytes
    for(unsigned lba(0);lba<maxLba.word();lba+=256) {

      unsigned nBlk(maxLba.word()-lba);
      if(nBlk>256) nBlk=256;
      std::cout << "CrcVmeDevice::systemAceWriteCfImage()  Writing blocks "
		<< std::setw(6) << lba << " to "
		<< std::setw(6) << lba+nBlk-1 << std::endl;

      // Now get lock; set bit 1 of the control register to request lock
      read("SystemAceControlRegLSBs",&n);
      u.bit(1,true);
      write("SystemAceControlRegLSBs",u.word());


      //***MW
      std::cout << "Trying for Lock" << std::endl;
      systemAcePrint(std::cout);


      // Wait for the lock request to be granted; this throw an exception if it times out
      pollItem("SystemAceLockStatus",0x2,5000,&n);




	
      /***MW */
      /*
      std::cout << std::endl << "Trying to CF-Abort" << std::endl;
      
      read("SystemAceSecCntCmdReg",&n);
      std::cout << "SystemAceSecCntCmdReg" << printHex((unsigned)n) << std::endl;

      write("SystemAceSecCntCmdReg",0x600);
      sleep(1);
      read("SystemAceSecCntCmdReg",&n);
      std::cout << "SystemAceSecCntCmdReg" << printHex((unsigned)n) << std::endl << std::endl;
      write("SystemAceSecCntCmdReg",0x000);
      */
      /* MW***/ 





      //***MW
      std::cout << "Trying for RdyForCfCmd" << std::endl;
      systemAcePrint(std::cout);

      // Wait until ready for CF controller command
      pollItem("SystemAceRdyForCfCmd",0x1,5000,&n);




      // Set the LBA value
      UtlPack uLba(lba);
      write("SystemAceMpuLbaRegLSBs",uLba.halfWord(0));
      write("SystemAceMpuLbaRegMSBs",uLba.halfWord(1));

      // Set the write command and number of blocks to transfer
      if(nBlk<256) write("SystemAceSecCntCmdReg",0x400+nBlk);
      else         write("SystemAceSecCntCmdReg",0x400);

      // Put into reset
      //read("SystemAceControlRegLSBs",&n);
      //u.bit(7,true);
      //write("SystemAceControlRegLSBs",u.word());

      // Wait for data buffer ready
      pollItem("SystemAceDataBufRdy",0x1,5000,&n);

      // Set to single byte writes
      //write("SystemAceBusModeReg",0);

      // Write the blocks
      for(unsigned i(0);i<nBlk;i++) {

	// Each write is 2 bytes so need 256 writes for a block
	/*
	for(unsigned j(0);j<256;j++) {
	  if((256*(lba+i)+j)<32) {
	    std::cout << "Word " << 256*(lba+i)+j << " = " << printHex(v[256*(lba+i)+j]) << std::endl;
	  }

	  //write("SystemAceDataBufReg",v[256*(lba+i)+j]);
	  //write("SystemAceDataBufReg16b",v[256*(lba+i)+j]);

	  //UtlPack uv(v[256*(lba+i)+j]);
	  //write("SystemAceDataBufReg16b",uv.byte(0));
	  //write("SystemAceDataBufReg16b",uv.byte(1),HAL::HAL_NO_VERIFY,1);
	}
	*/

	/* This lost the second 2 bytes
	for(unsigned j(0);j<128;j++) {
	  UtlPack uv;
	  uv.halfWord(0,v[256*(lba+i)+2*j  ]);
	  uv.halfWord(1,v[256*(lba+i)+2*j+1]);

	  if((256*(lba+i)+2*j)<16) {
	    std::cout << "Word " << 256*(lba+i)+2*j << " = " << printHex(uv) << std::endl;
	  }

	  write("SystemAceDataBufReg32b",uv.word());
	}
        */

	for(unsigned j(0);j<256;j++) {
	  UtlPack uv;
	  uv.halfWord(0,v[256*(lba+i)+j  ]);

	  if((256*(lba+i)+j)<257) {
	   read("SystemAceStatusRegLSBs",&n);
	   std::cout << "Word "   << 256*(lba+i)+j << " = " << printHex(uv) 
		     << "Status " << n             << " = " << printHex((unsigned)n) 
                     << std::endl;
	  }

	  write("SystemAceDataBufReg32b",uv.word());
	}

	systemAcePrint(std::cout);

	/***MW2 Test 
	    bool empty(false);
	    for(unsigned j(0);j<5000 && !empty;j++) {
	      read("SystemAceStatusRegLSBs",&n);
	      if(u.bit(5)) {
	       for(unsigned i(0);i<16;i++) {
	       read("SystemAceDataBufReg",&n);
	      //v.push_back(u.halfWord(0));
	      }
	     } else {
	      empty=true;
	      std::cout << "Empty set for j = " << j << std::endl;
	     }
	    }
	***MW2 */


      }

      // Set to two byte access
      //write("SystemAceBusModeReg",1);

      sleep(1);

      // Clear reset
      //read("SystemAceControlRegLSBs",&n);
      //u.bit(1,false);
      //write("SystemAceControlRegLSBs",u.word());

      //sleep(5);

      // Release lock; unset bit 1 of the control register to release lock
      read("SystemAceControlRegLSBs",&n);
      u.bit(0,false);
      u.bit(1,false);
      write("SystemAceControlRegLSBs",u.halfWord(0));

      // Wait for the lock to be released
      pollItem("SystemAceLockStatus",0x0,5000,&n);
    }

    return true;
  }

  bool serialWrite(const CrcSerialHeader *h) {
    if(h==0) return false;

    if(serialPrint) {
      std::cout << std::endl << "serialWrite ";
      h->print(std::cout);
      std::cout << std::endl;
    }

    assert(!h->readback());

    /*unsigned long*/ uint32_t *b((/*unsigned long*/ uint32_t*)h);


#ifdef FAKE_SPILL
#ifdef SPILL_INPUT
    bool trgInvert(h->target()==CrcSerialHeader::beTrg &&
		   h->designator()==CrcSerialHeader::beTrgInputInvert);

    if(trgInvert) {
      //std::cout << "serialWrite trigger input invert";
      UtlPack *p((UtlPack*)b);
      _spillInvert=p[1].bit(SPILL_INPUT);

      //if(_spillInvert) std::cout << " true" << std::endl;
      //else             std::cout << " false" << std::endl;
    }
#endif
#endif



    if(serialPrint) std::cout << "serialWrite  SerialBuffer[0] = 0x" 
			      << std::hex << b[0] << std::dec << std::endl;
    write("SerialBuffer",b[0]);

    unsigned nWords((h->length()+31)/32);
    for(unsigned i(0);i<nWords;i++) {
      if(serialPrint) std::cout << "serialWrite  SerialBuffer[" << i+1 << "] = 0x" 
				<< std::hex << b[i+1] << std::dec << std::endl;
      write("SerialBuffer",b[i+1],HAL::HAL_NO_VERIFY,4*(i+1));
    }
    write("SerialBuffer",0,HAL::HAL_NO_VERIFY,4*(nWords+1));
    write("SerialWrite",nWords+1);

    /*unsigned long*/ uint32_t ss(0);
    pollItem("SerialStatus",0xffffffff,1000,&ss);
    if(serialPrint) std::cout << "serialWrite  SerialStatus = 0x" 
			      << std::hex << ss << std::dec << std::endl << std::endl;
    return ss==0xffffffff;
  }

  bool serialWriteVector(const std::vector<const CrcSerialHeader*> &v) {
    for(unsigned i(0);i<v.size();i++) {
      if(!serialWrite(v[i])) return false;
    }
    return true;
  }

  bool serialRead(CrcSerialHeader *h) {
    bool replyA(serialReadA(h));
    bool replyB(serialReadB(h));
    return replyA && replyB;
  }

  bool serialReadA(CrcSerialHeader *h) {
    if(h==0) return false;

    h->readback(true);
    if(serialPrint) {
      std::cout << std::endl << "serialRead ";
      h->print(std::cout);
      std::cout << std::endl;
    }

    unsigned header(*((unsigned*)h));

    header++; // Increase length
    header|=0x400000; // Set readback bit

    if(h->target()==CrcSerialHeader::be && (h->designator()&0x10)!=0) {
      header++; // Increase length again for RO
      header<<=1; // Shift sideways for RO

    } else if(h->feTarget()) {
      if(h->designator()==30) {
	header++; // Increase length again for RO
	header++; // Increase length again for RO
	header<<=1; // Shift sideways for trigger counter
	header<<=1; // Shift sideways for trigger counter
      }
      if(h->designator()==27) {
	header++; // Increase length again for RO
	header<<=1; // Shift sideways for trigger counter
      }

      if(h->designator()== 3 || 
	 h->designator()== 9 ||
	 h->designator()==10 ||
	 h->designator()==28) {
	header++; // Increase length again for RO
	header++; // Increase length again for RO
	header<<=1; // Shift sideways for RO
      }

      if(h->designator()==20 
	 || h->designator()==29) {
	header++; // Increase length again for RO
	header++; // Increase length again for RO
	//header++; // Increase length again for RO
	//header+=2; // Increase length again for RO
	header<<=1; // Shift sideways for RO
	header<<=1; // Shift sideways for RO
	//header<<=1; // Shift sideways for RO
	//header<<=2; // Shift sideways for RO
      }
    }

    if(serialPrint) std::cout << "serialRead  Header = 0x" << std::hex << header << std::dec << std::endl;
    write("SerialRead",header);

    return true;
  }

  bool serialReadB(CrcSerialHeader *h) {



#ifdef FAKE_SPILL
#ifdef SPILL_INPUT
    //if(trgStatus) std::cout << "serialRead reading trigger input status" << std::endl;
#endif
#endif

    /*unsigned long*/ uint32_t ss(0);
    pollItem("SerialStatus",0xffffffff,1000,&ss);
    if(serialPrint) std::cout << "serialRead  SerialStatus = 0x" << std::hex << ss << std::dec << std::endl;
    if(ss!=0xffffffff) return false;

    /*unsigned long*/ uint32_t *b((/*unsigned long*/ uint32_t*)h);
    unsigned nWords((h->length()+31)/32);
    for(unsigned i(0);i<nWords;i++) {
      read("SerialBuffer",b+i+1,4*i);
      if(serialPrint) std::cout << "serialRead  SerialBuffer[" << i+1 << "] = 0x" 
			   << std::hex << b[i+1] << std::dec << std::endl << std::endl;
    }


#ifdef FAKE_SPILL
#ifdef SPILL_INPUT
    bool trgStatus(h->target()==CrcSerialHeader::beTrg &&
		   h->designator()==CrcSerialHeader::beTrgInputStatus);
    if(trgStatus) {
      UtlPack *p((UtlPack*)b);
#ifdef FNAL_SETTINGS
      //if(_spillInvert) p[1].bit(SPILL_INPUT,(time(0)%60)>0);
      //else             p[1].bit(SPILL_INPUT,(time(0)%60)<1);
      //if(_spillInvert) p[1].bit(SPILL_INPUT,(time(0)%20)>0); // TEMP!
      //else             p[1].bit(SPILL_INPUT,(time(0)%20)<1)

      // Invert is only used in logic, not in status word
      //p[1].bit(SPILL_INPUT,(time(0)%60)<1); // 1/60 sec
      //p[1].bit(SPILL_INPUT,(time(0)%30)<1); // 1/30 sec
      //p[1].bit(SPILL_INPUT,(time(0)%120)<4); // 4/120sec
      p[1].bit(SPILL_INPUT,(time(0)%60)<4); // 4/60sec

#else
      //if(_spillInvert) p[1].bit(SPILL_INPUT,(time(0)%17)>4);
      //else             p[1].bit(SPILL_INPUT,(time(0)%17)<5);

      // Invert is only used in logic, not in status word
      p[1].bit(SPILL_INPUT,(time(0)%17)<5);
#endif
    }
#endif
#endif


    return true;
  }

  /*
 bool serialRead(CrcSerialHeader *h) {
    if(h==0) return false;

    h->readback(true);
    if(serialPrint) {
      std::cout << std::endl << "serialRead ";
      h->print(std::cout);
      std::cout << std::endl;
    }

    unsigned long *b((unsigned long*)h);
    unsigned header(b[0]);
    header++; // Increase length
    header|=0x400000; // Set readback bit

    if(h->target()==CrcSerialHeader::be && (h->designator()&0x10)!=0) {
      header++; // Increase length again for RO
      header<<=1; // Shift sideways for RO

    } else if(h->feTarget()) {
      if(h->designator()==30) {
        header++; // Increase length again for RO
        header++; // Increase length again for RO
        header<<=1; // Shift sideways for trigger counter
        header<<=1; // Shift sideways for trigger counter
      }
      if(h->designator()==27) {
        header++; // Increase length again for RO
        header<<=1; // Shift sideways for trigger counter
      }

      if(h->designator()== 3 ||
         h->designator()== 9 ||
         h->designator()==10 ||
         h->designator()==28) {
	header++; // Increase length again for RO
        header++; // Increase length again for RO
        header<<=1; // Shift sideways for RO
      }

      if(h->designator()==20
         || h->designator()==29) {
        header++; // Increase length again for RO
        header++; // Increase length again for RO
        //header++; // Increase length again for RO
        //header+=2; // Increase length again for RO
        header<<=1; // Shift sideways for RO
        header<<=1; // Shift sideways for RO
        //header<<=1; // Shift sideways for RO
        //header<<=2; // Shift sideways for RO
      }
    }

    if(serialPrint) std::cout << "serialRead  Header = 0x" << std::hex << header << std::dec << std::endl;
    write("SerialRead",header);

    unsigned long ss(0);
    pollItem("SerialStatus",0xffffffff,1000,&ss);
    if(serialPrint) std::cout << "serialRead  SerialStatus = 0x" << std::hex << ss << std::dec << std::endl;
    if(ss!=0xffffffff) return false;

    unsigned nWords((h->length()+31)/32);
    for(unsigned i(0);i<nWords;i++) {
      read("SerialBuffer",b+i+1,4*i);
      if(serialPrint) std::cout << "serialRead  SerialBuffer[" << i+1 << "] = 0x"
                           << std::hex << b[i+1] << std::dec << std::endl << std::endl;
    }


#ifdef FAKE_SPILL
#ifdef SPILL_INPUT
    bool trgStatus(h->target()==CrcSerialHeader::beTrg &&
                   h->designator()==CrcSerialHeader::beTrgInputStatus);
    if(trgStatus) {
      UtlPack *p((UtlPack*)b);
#ifdef FNAL_SETTINGS
      //if(_spillInvert) p[1].bit(SPILL_INPUT,(time(0)%60)>0);
      //else             p[1].bit(SPILL_INPUT,(time(0)%60)<1);
      if(_spillInvert) p[1].bit(SPILL_INPUT,(time(0)%20)>0); // TEMP!
      else             p[1].bit(SPILL_INPUT,(time(0)%20)<1);
#else
      if(_spillInvert) p[1].bit(SPILL_INPUT,(time(0)%17)>4);
      else             p[1].bit(SPILL_INPUT,(time(0)%17)<5);
#endif
    }
#endif
#endif


    return true;
  }

  */


  bool serialReadVector(const std::vector<CrcSerialHeader*> &v) {
    for(unsigned i(0);i<v.size();i++) {
      if(!serialRead(v[i])) return false;
    }
    return true;
  }

  bool readVlinkEventData(CrcVlinkEventData &d, bool blt=false, bool flag=false) {
    return vlinkRead(d,blt,flag);
  }

  bool vlinkRead(CrcVlinkEventData &d, bool blt, bool flag) {
    d.numberOfWords(0);
    d.numberOfWords(vlinkRead(d.data(),blt,flag));
    return true;
  }

  unsigned vlinkRead(unsigned *d, bool blt, bool flag) {
    if(d==0) return 0;

    if(vlinkPrint) std::cout << "Entered vlinkRead" << std::endl;

    // Check for buffer ready; now allow multiple tries 11/10/06
    /*
    if(!isSet("VlinkBufferReady")) {
      if(vlinkPrint) std::cout << "vlinkRead  VlinkBufferReady not set"<< std::endl;
      return 0;
    }
    */

    bool vbReady(false);
    for(unsigned ii(0);ii<10 && !vbReady;ii++) { // Each loop ~10us, so 10 should be enough
      if(!isSet("VlinkBufferReady")) {
        if(vlinkPrint) {
	  std::cout << "CrcVmeDevice::vlinkRead  Read " << ii
		    << " vlinkRead  VlinkBufferReady not set" << std::endl;
	}
      } else {
        vbReady=true;
        if(ii>0) {
	  //  std::cout << "CrcVmeDevice::vlinkRead()  VlinkBufferReady set after read " << ii << std::endl;
	  //  std::cerr << "CrcVmeDevice::vlinkRead()  VlinkBufferReady set after read " << ii << std::endl;
	}
      }
    }
    if(!vbReady) return 0;

    unsigned nWords(0);

    bool original(flag);

    if(original) {

    if(isSet("VlinkLastBuffer")) {
      if(vlinkPrint) std::cout << "Last buffer flag set" << std::endl;
    }

    /*unsigned long*/ uint32_t n(0);
    read("VlinkEventNumber",&n);
    if(vlinkPrint) std::cout << "VlinkEventNumber " << n << std::endl;
    d[nWords++]=n;

    bool lastBuffer(false);
    while(isSet("VlinkBufferReady") && !lastBuffer) {
      read("VlinkBufferLength",&n);
      if(vlinkPrint) std::cout << "vlinkRead  VlinkBufferLength = " << n << std::endl;

      if(!blt) {
	for(unsigned i(0);i<n;i++) {
	  read("VlinkBuffer",(/*unsigned long*/ uint32_t*)(d+nWords+i),4*i);
	  //cout << "vlinkRead  d[" << nWords+i << "] = " << std::hex << d[nWords+i] << std::dec << std::endl;
	}
      } else {
	//readBlock("VlinkBufferBlt",4*n,(char*)(d+nWords));
	for(unsigned i(0);i<n;i+=64) {
	  unsigned m(4*(n-i));
	  if(m>256) m=256;
	  readBlock("VlinkBufferBlt",m,(char*)(d+nWords+i),HAL::HAL_DO_INCREMENT,4*i);
	}
      }
      nWords+=n;

      if(isSet("VlinkLastBuffer")) {
	if(vlinkPrint) std::cout << "Last buffer flag set" << std::endl;

	lastBuffer=true;
	/*unsigned long*/ uint32_t lWords(0);
	read("VlinkTotalLength",&lWords);
	if(vlinkPrint) std::cout << "VlinkTotalLength = " << lWords << std::endl;

	vlinkTotalLengthSum+=nWords-1;
	if(vlinkTotalLengthSum!=lWords) {
	  std::cout << "vlinkRead ERROR in number of words: read "
		    << nWords-1 << ", VlinkTotalLengthSum " 
		    << vlinkTotalLengthSum << " != VlinkTotalLength " 
		    << lWords << std::endl;
	  std::cerr << "vlinkRead ERROR in number of words: read "
		    << nWords-1 << ", VlinkTotalLengthSum " 
		    << vlinkTotalLengthSum << " != VlinkTotalLength " 
		    << lWords << std::endl;
	}

	d[nWords++]=lWords;
	writePulse("VlinkNextEvent");
	if(vlinkPrint) std::cout << "Pulsed for next event" << std::endl;

      } else {
	resetBit("VlinkBufferReady");
	//sleep(1);
	usleep(100000);
	if(isSet("VlinkBufferReady")) {
	  //if(vlinkPrint)
 std::cout << "Buffer ready flag set" << std::endl;
	} else {
	  //if(vlinkPrint)
 std::cout << "Buffer ready flag not set" << std::endl;
	}
      }
    }

    } else {

      /*unsigned long*/ uint32_t n(0);
      read("VlinkEventNumber",&n);
      if(vlinkPrint) std::cout << "VlinkEventNumber " << n << std::endl;
      d[nWords++]=n;

      // Assume only one buffer always
      read("VlinkBufferLength",&n);
      if(vlinkPrint) std::cout << "vlinkRead  VlinkBufferLength = " << n << std::endl;

      if(!blt) {

	// Read word by word
	for(unsigned i(0);i<n;i++) {

	  // Catch exceptions from HAL code during BLT
	  // These seem to only occur after a configurationStart
	  // The root cause of them must be understood
	  // This loop is a work around here for now 03/03/06
	  
	  bool notRead(true);
	  for(unsigned z(0);z<100 && notRead;z++) {
	    try {
	      read("VlinkBuffer",(/*unsigned long*/ uint32_t*)(d+nWords+i),4*i);
	      //cout << "vlinkRead  d[" << nWords+i << "] = " << std::hex << d[nWords+i] << std::dec << std::endl;
	      notRead=false;
	      
	    } catch ( HAL::HardwareAccessException& e ) {
	      std::cout << "*** Word " << i << " read " << z << " Exception occurred : "
			<< e.what() << std::endl;
	      std::cerr << "*** Word " << i << " read " << z << " Exception occurred : "
			<< e.what() << std::endl;
	    }
	  }
	  if(notRead) d[nWords+i]=0;
	}

      } else {
	// Read using block transfer

	//readBlock("VlinkBufferBlt",4*n,(char*)(d+nWords));
	/*
	for(unsigned i(0);i<n;i+=64) {
	  unsigned m(4*(n-i));
	  if(m>256) m=256;
	  readBlock("VlinkBufferBlt",m,(char*)(d+nWords+i),HAL::HAL_DO_INCREMENT,4*i);
	}
	*/

	unsigned m(64*(1+(n/64)));
	//std::cout << "About to do " << 4*m << std::endl;

        // Catch exceptions from HAL code during BLT
	// These seem to only occur after a configurationStart
	// The root cause of them must be understood
	// This loop is a work around here for now 03/03/06

        bool notRead(true);
        for(unsigned z(0);z<100 && notRead;z++) {
          try {
            readBlock("VlinkBufferBlt",4*m,(char*)(d+nWords),HAL::HAL_DO_INCREMENT,0);
            notRead=false;

          } catch ( HAL::HardwareAccessException& e ) {
            std::cout << "*** BLT read " << z << " Exception occurred : "
		      << e.what() << std::endl;
            std::cerr << "*** BLT read " << z << " Exception occurred : "
		      << e.what() << std::endl;
          }
        }

	// I hope this really never happens; it will mess up the
	// byte counting below if it does!!!
        if(notRead) n=0;
      }

      nWords+=n;

      /*unsigned long*/ uint32_t lWords(0);
      read("VlinkTotalLength",&lWords);
      if(vlinkPrint) std::cout << "VlinkTotalLength = " << lWords << std::endl;

      vlinkTotalLengthSum+=nWords-1;
      if(vlinkTotalLengthSum!=lWords) {
	std::cout << "vlinkRead ERROR in number of words: read "
		  << nWords-1 << ", VlinkTotalLengthSum " 
		  << vlinkTotalLengthSum << " != VlinkTotalLength " 
		  << lWords << std::endl;
	std::cerr << "vlinkRead ERROR in number of words: read "
		  << nWords-1 << ", VlinkTotalLengthSum " 
		  << vlinkTotalLengthSum << " != VlinkTotalLength " 
		  << lWords << std::endl;
      }
      
      d[nWords++]=lWords;
      writePulse("VlinkNextEvent");
      if(vlinkPrint) std::cout << "Pulsed for next event" << std::endl;
    }

    return nWords;
  }


  // ***** EPROM *****

  bool initialiseEprom(const unsigned char s) {
    /*unsigned long*/ uint32_t value;

    // Remove protection
    write("EpromControl",(0x7ff<<8)+2);
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    write("EpromControl",(0x7ff<<8)+6);
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    write("EpromControl",(0x7ff<<8)+2);
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    //sleep(1);
    usleep(100000);


    // Write the data
    std::cout << "Writing SN = " << (unsigned)s << " to board" << std::endl;

    write("EpromControl",(0x000<<8)+s);
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    //sleep(1);
    usleep(100000);

    write("EpromControl",(0x001<<8)+0);
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    //sleep(1);
    usleep(100000);

    write("EpromControl",(0x002<<8)+0);
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    //sleep(1);
    usleep(100000);

    write("EpromControl",(0x003<<8)+0);
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    //sleep(1);
    usleep(100000);


    // Reset protection
    write("EpromControl",(0x7ff<<8)+6);
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    write("EpromControl",(0x7ff<<8)+0x9a);
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    //sleep(1);
    usleep(100000);

    write("EpromControl",(0x7ff<<8)+0);
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    return true;
  }

  bool writeEpromMessage(std::string s) {
    /*unsigned long*/ uint32_t value;
    
    unsigned n(0);

    write("EpromControl",0x80000+(2<<8));
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    if(value!=0) return false;
    read("EpromStatus",&value);
    n+=(value&0xff);

    write("EpromControl",0x80000+(3<<8));
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    if(value!=0) return false;
    read("EpromStatus",&value);
    n+=(value&0xff)<<8;

    std::cout << "n = " << n << std::endl;


    // Remove protection
    write("EpromControl",(0x7ff<<8)+2);
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    write("EpromControl",(0x7ff<<8)+6);
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    write("EpromControl",(0x7ff<<8)+2);
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    //sleep(1);
    usleep(100000);

    // Write the data
    UtlPack t(time(0));
    for(unsigned i(0);i<4;i++) {
      write("EpromControl",((n+i+4)<<8)+t.byte(i));
      pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
      std::cout << "Value = " << value << std::endl;
      if(value!=0) return false;
      
      //sleep(1);
      usleep(100000);
    }

    UtlPack b(s.size());
    for(unsigned i(0);i<4;i++) {
      write("EpromControl",((n+i+8)<<8)+b.byte(i));
      pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
      std::cout << "Value = " << value << std::endl;
      if(value!=0) return false;
      
      //sleep(1);
      usleep(100000);
    }

    const unsigned char *c((const unsigned char*)s.c_str());
    for(unsigned i(0);i<s.size();i++) {
      write("EpromControl",((n+i+12)<<8)+c[i]);
      pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
      std::cout << "Value = " << value << std::endl;
      if(value!=0) return false;
      
      //sleep(1);
      usleep(100000);
    }

    n+=8+s.size();
    std::cout << "n = " << n << std::endl;


    write("EpromControl",(2<<8)+(n&0xff));
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    //sleep(1);
    usleep(100000);

    write("EpromControl",(3<<8)+(n>>8));
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    //sleep(1);
    usleep(100000);

    // Reset protection
    write("EpromControl",(0x7ff<<8)+6);
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    write("EpromControl",(0x7ff<<8)+0x9a);
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    //sleep(1);
    usleep(100000);

    write("EpromControl",(0x7ff<<8)+0);
    pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    return true;
  }

  /*
  bool writeVmeEprom(unsigned n, const unsigned *p) {
    unsigned long value;

    // Remove protection
    write("EpromControl",(0x7ff<<8)+2);
    pollItem("EpromBusyError",0x1,10000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;


    //write("EpromControl",0x80000+(0x7ff<<8));
	write("EpromControl",0x80000);
	pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
	if(value!=0) return false;
	read("EpromStatus",&value);
    std::cout << "Register = " << value << std::endl;



    write("EpromControl",(0x7ff<<8)+6);
    pollItem("EpromBusyError",0x1,10000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    //write("EpromControl",0x80000+(0x7ff<<8));
	write("EpromControl",0x80100);
	pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
	if(value!=0) return false;
	read("EpromStatus",&value);
    std::cout << "Register = " << value << std::endl;

    write("EpromControl",(0x7ff<<8)+2);
    pollItem("EpromBusyError",0x1,10000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
 
	write("EpromControl",0x80000+(0x7ff<<8));
	pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
	if(value!=0) return false;
	read("EpromStatus",&value);
    std::cout << "Register = " << value << std::endl;

    for(unsigned i(0);i<=n;i++) {
      UtlPack pack;
      if(i==0) pack.word(n);
      else     pack.word(p[i-1]);

      std::cout << "Word " << setw(4) << i << " = "
		<< printHex(pack.word()) << std::endl;

      for(unsigned j(0);j<4;j++) {
	write("EpromControl",((4*i+j)<<8)+pack.byte(j));
	pollItem("EpromBusyError",0x1,1000000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
	std::cout << "Value = " << value << std::endl;
	//if(value!=0) return false;
      }
    }

    // Reset protection
    write("EpromControl",(0x7ff<<8)+0x1a);
    pollItem("EpromBusyError",0x1,10000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    //if(value!=0) return false;

    read("FirmwareId",&value);
    read("ClockSelect",&value);

    for(unsigned i(0);i<=n;i++) {
      UtlPack pack;
      for(unsigned j(0);j<4;j++) {
	//write("EpromControl",0x80000+((4*i+j)<<8));
	//pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
	//if(value!=0) return false;
	read("EpromStatus",&value);
	pack.byte(j,value&0xff);
      }
      std::cout << "Word " << setw(4) << i << " = "
		<< printHex(pack.word()) << std::endl;
    }
    
    return true;
 }
*/


  // ***** VME *****

  bool readVmeRunData(CrcVmeRunData &d) {
    /*unsigned long*/ uint32_t value;

    read("FirmwareId",&value);
    d.firmwareId(value);
    read("ClockSelect",&value);
    d.clockSelect(value);

    /*
    write("EpromControl",(1<<8)+0xec);
    pollItem("EpromBusyError",0x1,10000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    if(value!=0) return false;
    */

    /*
    // Remove protection
    write("EpromControl",(0x7ff<<8)+2);
    pollItem("EpromBusyError",0x1,10000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    write("EpromControl",(0x7ff<<8)+6);
    pollItem("EpromBusyError",0x1,10000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    write("EpromControl",(0x7ff<<8)+2);
    pollItem("EpromBusyError",0x1,10000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    sleep(1);

    write("EpromControl",(0x7fe<<8)+2);
    pollItem("EpromBusyError",0x1,10000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    sleep(1);

    write("EpromControl",(0x000<<8)+0x78);
    pollItem("EpromBusyError",0x1,10000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    sleep(1);

    write("EpromControl",(0x001<<8)+0x56);
    pollItem("EpromBusyError",0x1,10000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    sleep(1);

    write("EpromControl",(0x002<<8)+0x34);
    pollItem("EpromBusyError",0x1,10000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    sleep(1);

    write("EpromControl",(0x003<<8)+0x12);
    pollItem("EpromBusyError",0x1,10000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    sleep(1);
    */

    /*

    for(unsigned i(0);i<512;i++) {
      for(unsigned j(0);j<4;j++) {
	write("EpromControl",((4*i+j)<<8)+0xaa);
    pollItem("EpromBusyError",0x1,10000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    if(value==0) {
      std::cout << "Add " << setw(4) << 4*i+j << " Value = " << value << std::endl;
    }
      }
    }
    */

    write("EpromControl",0x80000+((0x7ff)<<8));
    for(unsigned j(0);j<16;j++) {
      read("EpromStatus",&value);
      //cout << "WPR Byte = " << printHex((unsigned)value) << std::endl;
    }
    
    //sleep(1); // Gives 0xff for first char otherwise?!?
    usleep(100000);

    for(unsigned i(0);i<4;i++) {
      write("EpromControl",0x80000+(i<<8));
      pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
      if(value!=0) return false;

      read("EpromStatus",&value);
      //cout << "Header[" << i << "] = " << printHex((unsigned)value) << std::endl;

      d.epromHeader(i,value&0xff);
    }


    unsigned bytes(d.epromHeader()>>16);
    unsigned rBytes(0);
    //cout << "Number of bytes = " << bytes << std::endl;

    //unsigned char *c(d.data());

    //for(unsigned i(0);i<(d.epromHeader()>>16) || i<4;i++) {
    //for(unsigned i(0);i<(d.epromHeader()>>16) && i<8;i++) { // LIMIT FOR NOW!!!

      while(rBytes<bytes) {
	UtlPack time;
	for(unsigned j(0);j<4;j++) {
	  write("EpromControl",0x80000+((4+j)<<8));
	  pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
	  if(value!=0) return false;
	  read("EpromStatus",&value);
	  time.byte(j,value&0xff);
	}
	rBytes+=4;

	UtlPack leng;
	for(unsigned j(0);j<4;j++) {
	  write("EpromControl",0x80000+((8+j)<<8));
	  pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
	  if(value!=0) return false;
	  read("EpromStatus",&value);
	  leng.byte(j,value&0xff);
	}
	rBytes+=4+leng.word();
      }

      /*

      //cout << "Word " << setw(3) << i << " = " << printHex(pack.word()) << std::endl;      UtlPack time;
      for(unsigned j(0);j<4;j++) {
	write("EpromControl",0x80000+((4*i+4+j)<<8));
	pollItem("EpromBusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
	if(value!=0) return false;
	read("EpromStatus",&value);
	if(i<(d.epromHeader()>>16)) c[4*i+4+j]=value&0xff;

	pack.byte(j,value&0xff);
      }
      //cout << "Word " << setw(3) << i << " = " << printHex(pack.word()) << std::endl;
    }

      */
    
    return true;
  }

  bool readVmeEventData(CrcVmeEventData &d) {
    /*unsigned long*/ uint32_t value;

    read("VmeStatus",&value);
    d.status(value);

    read("TTCClockCounter",&value);
    d.ttcCounter(value);

    read("BackplaneClockCounter",&value);
    d.backplaneCounter(value);

    return true;
  }


  // ***** BE *****

  bool beSoftReset() {
    return true;
  }

  bool clearBeFifos() {
    CrcSerialCommandBit s(CrcSerialHeader::be,
			  CrcSerialHeader::beSoftReset);
    s.bit(false);
    return serialWrite(&s);
  }

  bool beSoftTrigger() {
    CrcSerialCommandBit s(CrcSerialHeader::be,
			  CrcSerialHeader::beSoftTrigger);
    s.bit(false);
    return serialWrite(&s);
  }

  bool readBeRunData(CrcBeRunData &d) {
    CrcSerialCommandWord w(CrcSerialHeader::be,CrcSerialHeader::beFirmwareId);
    if(!serialRead(&w)) return false;
    d.firmwareId(w.data());

    return true;
  }

  bool writeBeConfigurationData(const CrcBeConfigurationData &d) {
    CrcSerialCommandBeTriggerSelect b3(CrcSerialHeader::be,CrcSerialHeader::beTriggerSelect);
    b3.data(d.triggerSelect());
    if(!serialWrite(&b3)) return false;

    CrcSerialCommandBeMode b4;
    b4.data(d.mode());
    if(!serialWrite(&b4)) return false;
    
    CrcSerialCommandBeControl b2(CrcSerialHeader::be,CrcSerialHeader::beReadoutControl);
    b2.data(d.readoutControl());
    if(!serialWrite(&b2)) return false;    

    b2.designator(CrcSerialHeader::beRunControl);
    b2.data(d.runControl());
    if(!serialWrite(&b2)) return false;
    
    CrcSerialCommandWord w(CrcSerialHeader::be,CrcSerialHeader::beTest);
    w.data(d.test());
    if(!serialWrite(&w)) return false;    


    /*
    // RESET FES IN CASE CHANGED CLOCK POLARITY
    w.data(d.test()&(1<<30));
    if(!serialWrite(&w)) return false;    
    w.data(d.test());
    if(!serialWrite(&w)) return false;    
    */


    CrcSerialCommandByte de(CrcSerialHeader::be,CrcSerialHeader::beFeDataEnable);
    de.data(d.feDataEnable());
    if(!serialWrite(&de)) return false;

    CrcSerialCommandBeDaqId id;
    id.data(d.daqId());
    if(!serialWrite(&id)) return false;    

#ifdef OLD_TRIGGER_FW
    std::cout  << std::endl << "OLD TRIGGER FIRMWARE USED FOR BE CONFIGURATION" << std::endl << std::endl;

    w.designator(CrcSerialHeader::beFeTrgEnable);
    w.data(d.trgEnables()&0xfff);
    if(!serialWrite(&w)) return false;

    // Later included in beFeTrgEnable
    CrcSerialCommandWord te(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgTestLength); // N.B. from beTrg!
    te.data(d.testLength());
    if(!serialWrite(&te)) return false;

    te.designator(CrcSerialHeader::beTrgQdrAddress); // N.B. from beTrg!
    te.data(d.qdrAddress());
    if(!serialWrite(&te)) return false;
#else
    w.designator(CrcSerialHeader::beFeTrgEnable);
    w.data(d.trgEnables());
    if(!serialWrite(&w)) return false;
#endif

    return true;
  }

  bool readBeConfigurationData(CrcBeConfigurationData &d) {
    CrcSerialCommandBeTriggerSelect b3(CrcSerialHeader::be,CrcSerialHeader::beTriggerSelect);
    if(!serialRead(&b3)) return false;
    d.triggerSelect(b3.data());

    CrcSerialCommandBeMode b4;
    if(!serialRead(&b4)) return false;
    d.mode(b4.data());

    CrcSerialCommandBeControl b2(CrcSerialHeader::be,CrcSerialHeader::beReadoutControl);
    if(!serialRead(&b2)) return false;
    d.readoutControl(b2.data());

    b2.designator(CrcSerialHeader::beRunControl);
    if(!serialRead(&b2)) return false;
    d.runControl(b2.data());

    CrcSerialCommandWord w(CrcSerialHeader::be,CrcSerialHeader::beTest);
    if(!serialRead(&w)) return false;
    d.test(w.data());

    CrcSerialCommandByte de(CrcSerialHeader::be,CrcSerialHeader::beFeDataEnable);
    if(!serialRead(&de)) return false;
    d.feDataEnable(de.data());

    CrcSerialCommandBeDaqId id;
    if(!serialRead(&id)) return false;
    d.daqId(id.data());


#ifdef OLD_TRIGGER_FW
    std::cout  << std::endl << "OLD TRIGGER FIRMWARE USED FOR BE CONFIGURATION" << std::endl << std::endl;

    w.designator(CrcSerialHeader::beFeTrgEnable);
    if(!serialRead(&w)) return false;
    d.trgEnables(w.data()&0xfff);

    // Later included in beFeTrgEnable
    CrcSerialCommandWord te(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgTestLength); // N.B. from beTrg!
    if(!serialRead(&te)) return false;
    d.testLength(te.data());

    te.designator(CrcSerialHeader::beTrgQdrAddress); // N.B. from beTrg!
    if(!serialRead(&te)) return false;
    d.qdrAddress(te.data());

#else
    w.designator(CrcSerialHeader::beFeTrgEnable);
    if(!serialRead(&w)) return false;
    d.trgEnables(w.data());
#endif

    return true;
  }

  bool readBeEventData(CrcBeEventData &d, bool counterOnly=false) {
    CrcSerialCommandBeCounter c(CrcSerialHeader::beL1aCounter);
    if(!serialRead(&c)) return false;
    d.l1aCounter(c.data());

    if(!counterOnly) {
      CrcSerialCommandWord w(CrcSerialHeader::be,CrcSerialHeader::beStatus);
      if(!serialRead(&w)) return false;
      d.status(w.data());
      
      CrcSerialCommandBeBxCounter b12;
      if(!serialRead(&b12)) return false;
      d.bxCounter(b12.data());
      
      c.designator(CrcSerialHeader::beQdrFrameCounter);
      if(!serialRead(&c)) return false;
      d.qdrFrameCounter(c.data());
      
      CrcSerialCommandBeQdrDataCounter b18;
      if(!serialRead(&b18)) return false;
      d.qdrDataCounter(b18.data());
      
      c.designator(CrcSerialHeader::beTotalFrameCounter);
      if(!serialRead(&c)) return false;
      d.totalFrameCounter(c.data());

    } else {
      d.status(0xffffffff);
      d.bxCounter(0xffffffff);
      d.qdrFrameCounter(0xffffffff);
      d.qdrDataCounter(0xffffffff);
      d.totalFrameCounter(0xffffffff);
    }


    /* ??? WHY HERE?
      CrcSerialCommandWord tc(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgSignalCatch); // N.B. from beTrg!
    if(!serialRead(&tc)) return false;
    d.signalCatch(tc.data());
    */

    return true;
  }


  // ***** BE-Trigger *****

  bool beTrgSoftReset() {
    CrcSerialCommandWord r(CrcSerialHeader::beTrg,
			   CrcSerialHeader::beTrgCommands);
    if(!serialRead(&r)) return false;

    CrcSerialCommandWord w(CrcSerialHeader::beTrg,
			   CrcSerialHeader::beTrgCommands);
    w.data(r.data() | 0x2);
    if(!serialWrite(&w)) return true;

    //if(!serialRead(&r)) return false;
    //std::cout << "***** beTrgSoftReset **** data = " << r.data() << std::endl;

    return true;
  }

  bool beTrgSoftTrigger() {
    CrcSerialCommandWord r(CrcSerialHeader::beTrg,
			   CrcSerialHeader::beTrgCommands);
    if(!serialRead(&r)) return false;

    CrcSerialCommandWord w(CrcSerialHeader::beTrg,
			   CrcSerialHeader::beTrgCommands);

    // Is this first command needed/desirable?
    w.data(r.data() & 0xfffffffe);
    if(!serialWrite(&w)) return false;

    w.data(r.data() | 0x00000001);
    if(!serialWrite(&w)) return false;

    w.data(r.data() & 0xfffffffe);
    return serialWrite(&w);
  }

  bool beTrgSoftTriggers(unsigned n) {
    CrcSerialCommandWord r(CrcSerialHeader::beTrg,
			   CrcSerialHeader::beTrgGeneralEnable);
    if(!serialRead(&r)) return false;

    CrcSerialCommandWord w(CrcSerialHeader::beTrg,
			   CrcSerialHeader::beTrgGeneralEnable);

    // Turn on the trigger timeout
    w.data(r.data() | 0x00000002);
    if(!serialWrite(&w)) return false;

    CrcSerialCommandWord c(CrcSerialHeader::beTrg,
			   CrcSerialHeader::beTrgTriggerCounter);
    unsigned i(0);
    while(i<n) {
      if(!beTrgSoftTrigger()) return false;
      if(!serialRead(&c)) return false;

      std::cout << "CrcVmeDevice::beTrgSoftTriggers(" << n
		<< ")  In loop, returns beTrgTriggerCounter = " << i << std::endl;



      i=c.data();
    }

    // Turn off the trigger timeout
    w.data(r.data() & 0xfffffffd);
    if(!serialWrite(&w)) return false;

    return true;
  }

  bool beTrgSoftSpill(bool s) {
    return true;
  }

  bool clearBeTrgFifos() {
    CrcSerialCommandWord w(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgPrebusyTriggerCounter);
    w.data(0);
    if(!serialWrite(&w)) return false;

    w.designator(CrcSerialHeader::beTrgTriggerCounter);
    w.data(0);
    if(!serialWrite(&w)) return false;

    // Flush out FIFOs by reading them
    unsigned array[1024];
    CrcBeTrgEventData *d((CrcBeTrgEventData*)array);
    readBeTrgEventData(*d,false); // NOT squirt as no trigger!

    return true;
  }

  bool pollBeTrgTrigger(CrcBeTrgPollData &d, unsigned n) {
  /*
    d.update();

    CrcSerialCommandWord r(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgControl);
    d.tries(0);
    d.maximumTries(true);

    for(unsigned i(0);i<n;i++) {
      if(!serialRead(&r)) return false;
      if((r.data()&0x1)==0x1) {
	d.tries(i);
	d.maximumTries(false);
	return true;
      }
    }

    std::cout << "Poll failed: TrgControl = " << r.data() << std::endl;

    d.tries(n);
    d.maximumTries(true);
  */
    return true;
  }

  bool pollBeTrgTrigger(CrcBeTrgPollData &d) {

    // Set start and end time
    d.update(false);
    d.update(true);

    CrcSerialCommandWord r(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgControl);

    for(unsigned i(0);!d.timeout();i++) {
      d.update(true);
      d.numberOfPolls(i+1);

      if(!serialRead(&r)) return false;
      if((r.data()&0x1)==0x1) return true;
    }

    //cout << "Poll failed: TrgControl = " << r.data() << std::endl;

    // If timed out, then force a trigger
    if(!beTrgSoftTrigger()) return false;

    return true;
  }

  bool checkSpillSignal(unsigned n) {
    CrcSerialCommandWord r(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgInputStatus);
    for(unsigned i(0);i<n;i++) {
      if(!serialRead(&r)) return false;
      std::cout << " Loop " << std::setw(3) << i 
		<< " BeTrg input status = " << printHex(r.data()) << std::endl;
      sleep(1);
    }
    return true;
  }

  bool pollBeTrgSpillStart(CrcBeTrgPollData &d, bool invert=false) {

    // Set start and end time
    d.update(false);
    d.update(true);

    CrcSerialCommandWord r(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgInputStatus);

    for(unsigned i(0);!d.timeout();i++) {
      d.update(true);
      d.numberOfPolls(i+1);

      if(!serialRead(&r)) return false;
      //std::cout << "r = " << printHex(r.data()) << std::endl;
#ifdef SPILL_INPUT
      if(!invert) {
	if((r.data()&(1<<SPILL_INPUT))!=0) return true;
      } else {
	if((r.data()&(1<<SPILL_INPUT))==0) return true;
      }
#else 
     return true;
#endif
    }

    //cout << "Poll failed: TrgControl = " << r.data() << std::endl;

    return true;
  }

  bool trgSpillPoll(TrgSpillPollData &d) {

    // Set start and end poll times
    d.updateStartPollTime();
    d.updateEndPollTime();

    // Turn on the trigger timeout
    CrcSerialCommandWord r(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgGeneralEnable);
    if(!serialRead(&r)) return false;

    CrcSerialCommandWord w(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgGeneralEnable);
    w.data(r.data() | 0x00000002);
    if(!serialWrite(&w)) return false;

    // Ensure trigger not forced busy?
    CrcSerialCommandWord w1(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgControl);
    w1.data(0);
    if(!serialWrite(&w1)) return false;

    // Detect initial level of spill?
    bool inSpll(inSpill());

    CrcSerialCommandWord r0(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgTriggerCounter);
    CrcSerialCommandWord r1(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgGeneralEnable);
    CrcSerialCommandWord r2(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgControl); // spill?

    for(unsigned i(0);!d.spillTimeout() && !d.pollTimeout() && !d.maximumEvents();i++) {
      d.updateEndPollTime();
      d.numberOfPolls(i+1);
      //d.print(std::cout,"==>>>") << std::endl;

      // Check if spill level changed?
      if(!inSpll) {
	if(inSpill()) {
	  inSpll=true;
	  d.updateStartSpillTime();
	}

      } else {
	if(!inSpill()) {
	  inSpll=false;
	  d.updateEndSpillTime();
	}
      }

      if(!serialRead(&r0)) return false;
      if(!serialRead(&r1)) return false;
      if(!serialRead(&r2)) return false;

      d.actualNumberOfEvents(r0.data());
    }

    d.print(std::cout,"==>>>") << std::endl;
      
    // Turn off the trigger timeout
    w.data(r.data() & 0xfffffffd);
    if(!serialWrite(&w)) return false;

    return true;
  }

  bool inSpill() {
#ifdef DESY_SETTINGS
    return true;
#else
#ifdef FNAL_SETTINGS
    // Fake up FNAL 1sec on, 59sec off spill
    return (time(0)%60)<1;
#else
    // Fake up CERN 5sec on, 12sec off spill
    return (time(0)%17)<5;
#endif
#endif
  }

  bool beTrgForceBusy(bool b) {
    CrcSerialCommandWord r(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgControl);
    if(!serialRead(&r)) return false;

    CrcSerialCommandWord w(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgControl);
    if(b) w.data(r.data()|0x00000002);
    else  w.data(r.data()&0xfffffffd);
    if(!serialWrite(&w)) return false;

    return true;
  }

  bool readAndClearCatch() {
    CrcSerialCommandWord r(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgPrebusyCatch);
    if(!serialRead(&r)) return 0;
    unsigned c(r.data());

    CrcSerialCommandWord w(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgPrebusyCatch);
    w.data(0);
    if(!serialWrite(&w)) return 0;

    return c;
  }

  bool clearBeTrgTrigger() {

    // Get initial setting
    CrcSerialCommandWord r(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgControl);
    r.data(1);

    // TEST; IS THIS NEEDED? SHOULD ALWAYS BE ONE!
    if(!serialRead(&r)) return false;

    //assert(r.data()==1);
    if((r.data()&0x1)==0) return true;
    const unsigned control(r.data());

    // Get initial trigger counter value
    CrcSerialCommandWord tn(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgTriggerCounter);
    if(!serialRead(&tn)) return false;
    unsigned triggerCounter(tn.data());

    CrcSerialCommandWord w(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgControl);

    bool cleared(false);
    for(unsigned i(0);i<100 && !cleared;i++) {
      
      w.data(control&0xfffffffe);
      if(!serialWrite(&w)) return false;
      
      // See if trigger now not busy
      if(!serialRead(&r)) return false;
      //std::cout << "**** trig read = " << r.data() << " *****" << std::endl;

      if((r.data()&0x1)==0) {
	cleared=true;

      // Check for increment of counter if still set
      } else {
	if(!serialRead(&tn)) return false;
	cleared=tn.data()>triggerCounter;
      }

      if(!cleared) {
	std::cerr << "clearBeTrgTrigger  Clear loop " << i << " failed" << std::endl; 
	std::cout << "clearBeTrgTrigger  Clear loop " << i << " failed" << std::endl;
      }
    }

    return true;
  }

  bool readBeTrgRunData(CrcBeTrgRunData &d) {
    CrcSerialCommandWord w(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgFirmwareId);
    if(!serialRead(&w)) return false;
    d.firmwareId(w.data());

    w.designator(CrcSerialHeader::beTrgFirmwareDate);
    if(!serialRead(&w)) return false;
    d.firmwareDate(w.data());

    return true;
  }

  bool writeBeTrgConfigurationData(const CrcBeTrgConfigurationData &d) {
    // May need versioning?
    //CrcSerialCommandWord tr(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgFirmwareDate);
    //if(!serialRead(&tr)) return false;
    //if(tr.data()>1100536550) etc

    CrcSerialCommandWord w(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgInputEnable);
    w.data(d.inputEnable());
    if(!serialWrite(&w)) return false;

    w.designator(CrcSerialHeader::beTrgGeneralEnable);
    w.data(d.generalEnable());
    if(!serialWrite(&w)) return false;

    w.designator(CrcSerialHeader::beTrgOscillationPeriod);
    w.data(d.oscillationPeriod());
    if(!serialWrite(&w)) return false;

    w.designator(CrcSerialHeader::beTrgBurstCounter);
    w.data(d.burstCounter());
    if(!serialWrite(&w)) return false;

    w.designator(CrcSerialHeader::beTrgConfiguration);
    w.data(d.configuration());
    if(!serialWrite(&w)) return false;

    w.designator(CrcSerialHeader::beTrgBusyTimeout);
    w.data(d.busyTimeout());
    if(!serialWrite(&w)) return false;

    w.designator(CrcSerialHeader::beTrgBurstTimeout);
    w.data(d.burstTimeout());
    if(!serialWrite(&w)) return false;

    w.designator(CrcSerialHeader::beTrgAnd0Enable);
    w.data(d.andEnable(0));
    if(!serialWrite(&w)) return false;

    w.designator(CrcSerialHeader::beTrgAnd1Enable);
    w.data(d.andEnable(1));
    if(!serialWrite(&w)) return false;

    w.designator(CrcSerialHeader::beTrgAnd2Enable);
    w.data(d.andEnable(2));
    if(!serialWrite(&w)) return false;

    w.designator(CrcSerialHeader::beTrgAnd3Enable);
    w.data(d.andEnable(3));
    if(!serialWrite(&w)) return false;

    w.designator(CrcSerialHeader::beTrgExtBeamMode);
    w.data(d.extBeamMode());
    if(!serialWrite(&w)) return false;

    w.designator(CrcSerialHeader::beTrgInputInvert);
    w.data(d.inputInvert());
    if(!serialWrite(&w)) return false;

    w.designator(CrcSerialHeader::beTrgQdrConfiguration);
    w.data(d.qdrConfiguration());
    if(!serialWrite(&w)) return false;

    w.designator(CrcSerialHeader::beTrgSequencerControl);
    w.data(d.sequencerControl());
    if(!serialWrite(&w)) return false;

    return true;
  }

  bool readBeTrgConfigurationData(CrcBeTrgConfigurationData &d) {
    CrcSerialCommandWord w(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgInputEnable);
    if(!serialRead(&w)) return false;
    d.inputEnable(w.data());

    w.designator(CrcSerialHeader::beTrgGeneralEnable);
    if(!serialRead(&w)) return false;
    d.generalEnable(w.data());

    w.designator(CrcSerialHeader::beTrgOscillationPeriod);
    if(!serialRead(&w)) return false;
    d.oscillationPeriod(w.data());

    w.designator(CrcSerialHeader::beTrgBurstCounter);
    if(!serialRead(&w)) return false;
    d.burstCounter(w.data());

    w.designator(CrcSerialHeader::beTrgConfiguration);
    if(!serialRead(&w)) return false;
    d.configuration(w.data());

    w.designator(CrcSerialHeader::beTrgBusyTimeout);
    if(!serialRead(&w)) return false;
    d.busyTimeout(w.data());

    w.designator(CrcSerialHeader::beTrgBurstTimeout);
    if(!serialRead(&w)) return false;
    d.burstTimeout(w.data());

    w.designator(CrcSerialHeader::beTrgAnd0Enable);
    if(!serialRead(&w)) return false;
    d.andEnable(0,w.data());

    w.designator(CrcSerialHeader::beTrgAnd1Enable);
    if(!serialRead(&w)) return false;
    d.andEnable(1,w.data());

    w.designator(CrcSerialHeader::beTrgAnd2Enable);
    if(!serialRead(&w)) return false;
    d.andEnable(2,w.data());

    w.designator(CrcSerialHeader::beTrgAnd3Enable);
    if(!serialRead(&w)) return false;
    d.andEnable(3,w.data());

    w.designator(CrcSerialHeader::beTrgExtBeamMode);
    if(!serialRead(&w)) return false;
    d.extBeamMode(w.data());
    // Not yet implemented 16/03/06, so always returns with bits on; turn off
    if(w.data()==0xffffffff) d.extBeamMode(0);

    w.designator(CrcSerialHeader::beTrgInputInvert);
    if(!serialRead(&w)) return false;
    d.inputInvert(w.data());

    w.designator(CrcSerialHeader::beTrgQdrConfiguration);
    if(!serialRead(&w)) return false;
    d.qdrConfiguration(w.data());

    w.designator(CrcSerialHeader::beTrgSequencerControl);
    if(!serialRead(&w)) return false;
    d.sequencerControl(w.data());

    return true;
  }

  bool readBeTrgConfigurationData(CrcBeTrgConfigurationDataV0 &d) {
    CrcSerialCommandWord w(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgInputEnable);
    if(!serialRead(&w)) return false;
    d.inputEnable(w.data());

    w.designator(CrcSerialHeader::beTrgGeneralEnable);
    if(!serialRead(&w)) return false;
    d.outputEnable(w.data());

    w.designator(CrcSerialHeader::beTrgOscillationPeriod);
    if(!serialRead(&w)) return false;
    d.oscillationPeriod(w.data());

    /*
    w.designator((CrcSerialHeader::Designator)24);
    if(!serialRead(&w)) return false;
    std::cout << "readBeTrgEventData  Desig24 = " << printHex(w.data()) << std::endl;

    w.designator((CrcSerialHeader::Designator)25);
    if(!serialRead(&w)) return false;
    std::cout << "readBeTrgEventData  Desig25 = " << printHex(w.data()) << std::endl;
    */

    return true;
  }

  UtlPack readBeTrgInputStatus() {
    CrcSerialCommandWord r(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgInputStatus);
    if(!serialRead(&r)) return 0xffffffff;
    return r.data();
  }

  bool readBeTrgEventData(CrcBeTrgEventData &d, bool squirt=false, bool counterOnly=false) {
    d.numberOfFifoWords(0);

    CrcSerialCommandWord r(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgTriggerCounter);
    if(!serialRead(&r)) return false;
    d.triggerCounter(r.data());

    r.designator(CrcSerialHeader::beTrgInputStatus);
    if(!serialRead(&r)) return false;
    d.inputStatus(r.data());
    
    if(!counterOnly) {

      // Zero the catches after reading
      r.designator(CrcSerialHeader::beTrgInputCatch);
      if(!serialRead(&r)) return false;
      d.inputCatch(r.data());

      CrcSerialCommandWord w(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgInputCatch);
      w.data(0);
      if(!serialWrite(&w)) return false;

      r.designator(CrcSerialHeader::beTrgPrebusyCatch);
      if(!serialRead(&r)) return false;
      d.prebusyCatch(r.data());
      
      w.designator(CrcSerialHeader::beTrgPrebusyCatch);
      w.data(0);
      if(!serialWrite(&w)) return false;

      r.designator(CrcSerialHeader::beTrgSignalCatch);
      if(!serialRead(&r)) return false;
      d.signalCatch(r.data());
      
      w.designator(CrcSerialHeader::beTrgSignalCatch);
      w.data(0);
      if(!serialWrite(&w)) return false;
      
      r.designator(CrcSerialHeader::beTrgControl);
      if(!serialRead(&r)) return false;
      d.control(r.data());
      
      r.designator(CrcSerialHeader::beTrgPrebusyTriggerCounter);
      if(!serialRead(&r)) return false;
      d.prebusyTriggerCounter(r.data());
      
      r.designator(CrcSerialHeader::beTrgFifoStatus);
      if(!serialRead(&r)) return false;
      d.initialFifoStatus(r.data());
      
      unsigned fWords(r.data()&0x3ff);
      if((r.data()&0x20000)!=0) fWords++; // Full has one more word!
      
      unsigned *pf(d.fifoData());
      
      if(squirt) {
	CrcSerialCommand<32*CrcBeTrgEventData::maxFifoLength>
	  f(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgFifoSquirt);
	
	//serialPrint=true;
	if(!serialRead(&f)) return false;
	//serialPrint=false;
	
	// Truncate to 512!!! (limit in serial i/o?)
	if(fWords>512) fWords=512;
	if(fWords>CrcBeTrgEventData::maxFifoLength) fWords=CrcBeTrgEventData::maxFifoLength;
	
	const unsigned *qf(f.data());
	// use memcpy instead?
	for(unsigned i(0);i<fWords && i<CrcBeTrgEventData::maxFifoLength;i++) {
	  pf[i]=qf[i];
	}
	d.numberOfFifoWords(fWords);
	
      } else {
	CrcSerialCommandWord f(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgFifo);
	
	for(unsigned i(0);i<fWords;i++) {
	  if(!serialRead(&f)) return false;
	  //if(f.data()!=0)
	  //std::cout << "readBeTrgEventData  FIFO word " << setw(4) << i
	  //	  << " = " << printHex(f.data()) << std::endl;
	  
	  if(i<CrcBeTrgEventData::maxFifoLength) {
	    pf[i]=f.data();
	    d.numberOfFifoWords(i+1);
	  }
	  
	  if(i<(fWords-1)) {
	    if(!serialRead(&r)) return false;
	    if(r.data()!=(fWords-1-i)) {
	      std::cout << "readBeTrgEventData  FIFO status   "
			<< " = " << printHex(r.data()) << std::endl;
	      d.finalFifoStatus(r.data());
	      return false;
	    }
	  }
	}
      }

      r.designator(CrcSerialHeader::beTrgFifoStatus);
      if(!serialRead(&r)) return false;
      d.finalFifoStatus(r.data());

    } else {
      //d.inputStatus(0xffffffff); // Always read this out!
      d.inputCatch(0xffffffff);
      d.prebusyCatch(0xffffffff);
      d.signalCatch(0xffffffff);
      d.control(0xffffffff);
      d.prebusyTriggerCounter(0xffffffff);
      d.initialFifoStatus(0xffffffff);
      d.finalFifoStatus(0xffffffff);
    }

    return true;
  }


  // ***** FE *****

  bool feSoftReset() {
    CrcSerialCommandBit s(CrcSerialHeader::feBroadcast,
			  CrcSerialHeader::feSoftReset);
    s.bit(false);
    if(!serialWrite(&s)) return false;

    //sleep(1);
    usleep(100000);

    // Ensure HOLD is left open
    CrcSerialCommandFeAddress a(CrcSerialHeader::feBroadcast);
    CrcSerialCommandWord w(CrcSerialHeader::feBroadcast,
			   CrcSerialHeader::feDataOut);

    // Need to set HOLD not to be inverted
    a.address(6);
    if(!serialWrite(&a)) return false;
    w.data(0);
    if(!serialWrite(&w)) return false;

    // Need to send one mplex clock
    a.address(23);
    if(!serialWrite(&a)) return false;
    w.data(1);
    if(!serialWrite(&w)) return false;

    // Trigger is necessary to get HOLD un-invert to work
    //cout << "ABOUT TO BROADCAST TRIGGER " << std::endl;
    feSoftTrigger();

    return true;
  }

  bool feSoftReset(CrcLocation::CrcComponent c) {
    CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;
 
    CrcSerialCommandBit s(t,CrcSerialHeader::feSoftReset);
    s.bit(false);
    if(!serialWrite(&s)) return false;

    // Ensure HOLD is left open
    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord w(t,CrcSerialHeader::feDataOut);

    //CrcFeConfigurationData cd;
    //cd.holdInvert(false);
    //writeFeConfigurationData(c,cd);

    // Need to set HOLD not to be inverted
    a.address(6);
    if(!serialWrite(&a)) return false;
    w.data(0);
    if(!serialWrite(&w)) return false;

    // Need to send one mplex clock
    a.address(23);
    if(!serialWrite(&a)) return false;
    w.data(1);
    if(!serialWrite(&w)) return false;

    // Trigger is necessary to get HOLD un-invert to work
    //cout << "ABOUT TO TRIGGER " << std::endl;
    feSoftTrigger(c);

    return true;
  }

  bool feClearFifos() {
    CrcSerialCommandWord w(CrcSerialHeader::feBroadcast,
			   CrcSerialHeader::feTriggerCounter);
    w.data(0);
    if(!serialWrite(&w)) return false;

    CrcSerialCommandBit s(CrcSerialHeader::feBroadcast,
			  CrcSerialHeader::feFifoReset);
    s.bit(false);
    if(!serialWrite(&s)) return false;

    return true;
  }

  bool feSoftTrigger() {
    CrcSerialCommandBit s(CrcSerialHeader::feBroadcast,
			  CrcSerialHeader::feSoftTrigger);
    s.bit(false);
    return serialWrite(&s);
  }

  bool feSoftTrigger(CrcLocation::CrcComponent c) {
    CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;
    
    CrcSerialCommandBit s(t,CrcSerialHeader::feSoftTrigger);
    s.bit(false);
    return serialWrite(&s);
  }

  bool feZeroTriggerCounter() {
    CrcSerialCommandWord s(CrcSerialHeader::feBroadcast,
			   CrcSerialHeader::feTriggerCounter);
    s.data(0);
    return serialWrite(&s);
  }

  bool feZeroDacs(CrcLocation::CrcComponent c) {
    CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;
    
    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataOut);
    s.data(0);

    a.address(32);
    if(!serialWrite(&a)) return false;
    if(!serialWrite(&s)) return false;

    a.address(33);
    if(!serialWrite(&a)) return false;
    if(!serialWrite(&s)) return false;

    return true;
  }

  bool feZeroTriggerCounter(CrcLocation::CrcComponent c) {
    CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;
    
    CrcSerialCommandWord s(t,CrcSerialHeader::feTriggerCounter);
    s.data(0);
    return serialWrite(&s);
  }


  bool readFeRunData(CrcLocation::CrcComponent c, CrcFeRunData &d) {
    CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;
    
    CrcSerialCommandWord w(t,CrcSerialHeader::feFirmwareId);
    if(!serialRead(&w)) return false;
    d.firmwareId(w.data());

    w.designator(CrcSerialHeader::feLinkArray);
    if(!serialRead(&w)) return false;
    d.linkArray(w.data());

    w.designator(CrcSerialHeader::feVfeType);
    if(!serialRead(&w)) return false;
    d.vfeType(w.data());

    return true;
  }

  /*
  bool writeFeConfigurationData(CrcLocation::CrcComponent c, const CrcFeConfigurationData &d) {

    CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;
    
    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataOut);

    const unsigned *p(d.data());
    for(unsigned i(0);i<(sizeof(CrcFeConfigurationData)+3)/4;i++) {
      a.address(i);
      if(!serialWrite(&a)) return false;
      s.data(p[i]);
      if(!serialWrite(&s)) return false;
    }

    return true;
  }

  bool readFeConfigurationData(CrcLocation::CrcComponent c, CrcFeConfigurationData &d) {

    CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;
    
    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataIn);

    unsigned *p(d.data());
    for(unsigned i(0);i<(sizeof(CrcFeConfigurationData)+3)/4;i++) {
      a.address(i);
      if(!serialWrite(&a)) return false;
      if(!serialRead(&s)) return false;
      p[i]=s.data();
    }

    return true;
  }
*/

  bool writeFeConfigurationData(CrcLocation::CrcComponent c, const CrcFeConfigurationData &d) {

    CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;
    
    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataOut);

    a.address(0);
    if(!serialWrite(&a)) return false;
    s.data(d.calibWidth());
    if(!serialWrite(&s)) return false;

    a.address(1);
    if(!serialWrite(&a)) return false;
    s.data(d.calibControl());
    if(!serialWrite(&s)) return false;

    a.address(4);
    if(!serialWrite(&a)) return false;
    s.data(d.holdStart());
    if(!serialWrite(&s)) return false;

    a.address(5);
    if(!serialWrite(&a)) return false;
    s.data(d.holdWidth());
    if(!serialWrite(&s)) return false;

    a.address(6);
    if(!serialWrite(&a)) return false;
    s.data(d.holdControl());
    if(!serialWrite(&s)) return false;

    a.address(12);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeResetStart());
    if(!serialWrite(&s)) return false;

    a.address(13);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeResetEnd());
    if(!serialWrite(&s)) return false;

    a.address(16);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeSrinStart());
    if(!serialWrite(&s)) return false;

    a.address(17);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeSrinEnd());
    if(!serialWrite(&s)) return false;

    a.address(20);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeMplexClockStart());
    if(!serialWrite(&s)) return false;

    a.address(21);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeMplexClockMark());
    if(!serialWrite(&s)) return false;

    a.address(22);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeMplexClockSpace());
    if(!serialWrite(&s)) return false;

    a.address(23);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeMplexClockPulses());
    if(!serialWrite(&s)) return false;

    a.address(28);
    if(!serialWrite(&a)) return false;
    s.data(d.adcStart());
    if(!serialWrite(&s)) return false;

    a.address(29);
    if(!serialWrite(&a)) return false;
    s.data(d.adcEnd());
    if(!serialWrite(&s)) return false;

    a.address(30);
    if(!serialWrite(&a)) return false;
    s.data(d.adcControl());
    if(!serialWrite(&s)) return false;

    a.address(31);
    if(!serialWrite(&a)) return false;
    s.data(d.adcDelay());
    if(!serialWrite(&s)) return false;

    a.address(32);
    if(!serialWrite(&a)) return false;
    s.data(d.dacData(CrcFeConfigurationData::bot));
    if(!serialWrite(&s)) return false;

    a.address(33);
    if(!serialWrite(&a)) return false;
    s.data(d.dacData(CrcFeConfigurationData::top));
    if(!serialWrite(&s)) return false;

    a.address(34);
    if(!serialWrite(&a)) return false;
    s.data(d.frameSyncDelay());
    if(!serialWrite(&s)) return false;

    a.address(35);
    if(!serialWrite(&a)) return false;
    s.data(d.qdrDataDelay());
    if(!serialWrite(&s)) return false;

    a.address(36);
    if(!serialWrite(&a)) return false;
    s.data(d.readoutSyncDelay());
    if(!serialWrite(&s)) return false;
    
    a.address(24);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeControl().word());
    if(!serialWrite(&s)) return false;

    return true;
  }

  bool readFeConfigurationData(CrcLocation::CrcComponent c, CrcFeConfigurationData &d) {
    CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;
    
    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataIn);

    a.address(0);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.calibWidth(s.data());

    a.address(1);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.calibControl(s.data());

    a.address(4);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.holdStart(s.data());

    a.address(5);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.holdWidth(s.data());

    a.address(6);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.holdControl(s.data());

    a.address(12);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeResetStart(s.data());

    a.address(13);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeResetEnd(s.data());

    a.address(16);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeSrinStart(s.data());

    a.address(17);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeSrinEnd(s.data());

    a.address(20);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeMplexClockStart(s.data());

    a.address(21);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeMplexClockMark(s.data());

    a.address(22);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeMplexClockSpace(s.data());

    a.address(23);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeMplexClockPulses(s.data());

    a.address(28);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.adcStart(s.data());

    a.address(29);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.adcEnd(s.data());

    a.address(30);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.adcControl(s.data()&0xff);

    a.address(31);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.adcDelay(s.data());

    a.address(32);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.dacData(CrcFeConfigurationData::bot,s.data()&0xffff);

    a.address(33);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.dacData(CrcFeConfigurationData::top,s.data()&0xffff);

    a.address(34);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.frameSyncDelay(s.data());

    a.address(35);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.qdrDataDelay(s.data());

    a.address(36);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.readoutSyncDelay(s.data());
    
    a.address(24);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeControl(s.data());

    return true;
  }

private:
  bool writeVfeControlData(CrcLocation::CrcComponent c, const UtlPack &v) {
    CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;
    
    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataOut);

    a.address(24);
    if(!serialWrite(&a)) return false;
    s.data(v.word());
    if(!serialWrite(&s)) return false;

    return true;
  }

  bool readVfeControlData(CrcLocation::CrcComponent c, UtlPack &v) {
     CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;
    
    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataIn);

    a.address(24);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    v.word(s.data());

    return true;
 }

public:
  bool writeEmcFeConfigurationData(CrcLocation::CrcComponent c, const EmcFeConfigurationData &d) {
    if(!writeFeConfigurationData(c,d)) return false;
    if(!writeVfeControlData(c,d.vfeControl().data())) return false;
    return true;
  }

  bool readEmcFeConfigurationData(CrcLocation::CrcComponent c, EmcFeConfigurationData &d) {
    if(!readFeConfigurationData(c,d)) return false;

    UtlPack v;
    if(!readVfeControlData(c,v)) return false;
    d.vfeControl(EmcVfeControl(v));

    return true;
  }

  bool writeAhcFeConfigurationData(CrcLocation::CrcComponent c, const AhcFeConfigurationData &d) {
    if(!writeFeConfigurationData(c,d)) return false;
    if(!writeVfeControlData(c,d.vfeControl().data())) return false;
    return true;
  }

  bool readAhcFeConfigurationData(CrcLocation::CrcComponent c, AhcFeConfigurationData &d) {
    if(!readFeConfigurationData(c,d)) return false;

    UtlPack v;
    if(!readVfeControlData(c,v)) return false;
    d.vfeControl(AhcVfeControl(v));

    return true;
  }


  //private:

  // Write returning previous data from shift register
  
  bool loadAhcVfeData(CrcLocation::CrcComponent c,
		      unsigned nWords,
		      const UtlPack *wData,
		      UtlPack *rData,
		      bool reset, 
		      bool dac) {
    
    bool print(false);

    UtlPack vfeControlSave;
    if(!readVfeControlData(c,vfeControlSave)) return false;

    CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;
    
    CrcSerialCommandFeAddress a(t);
    a.address(24);
    if(!serialWrite(&a)) return false;
    
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataOut);
    CrcSerialCommandWord x(t,CrcSerialHeader::feSpyRegister);
    
    for(unsigned i(0);i<nWords;i++) {
      if(print) std::cout << std::setw(3) << i << " wData " << printHex(wData[i]) << std::endl;
    }
    
    // Set up constant signals, including DAC/SR switch
    
    AhcVfeControl send;

    send.swHoldIn(true);
    send.selOut(true);  // Use SEL_OUT to put SR_Q onto SR_OUT line
    if(dac) send.swDacIn(true);

    // Reset shift register if required
    if(reset) {
      s.data(send.data().word());
      if(!serialWrite(&s)) return false;
      if(print) std::cout << " -2" << printHex(send.data()) << std::endl;

      send.srResIn(true);
      s.data(send.data().word());
      if(!serialWrite(&s)) return false;
      if(print) std::cout << " -1" << printHex(send.data()) << std::endl;
      
      send.srResIn(false);
    }
    
    const unsigned nBits(16*nWords);

    // Loop over all bits to be sent
    for(unsigned i(0);i<nBits;i++) {
      
      // Set next data bits
      send.srDIn(CrcFeConfigurationData::boardA,wData[i/16].bit((i%16)   ));
      send.srDIn(CrcFeConfigurationData::boardB,wData[i/16].bit((i%16)+16));

      // Set clock off and send
      send.srClkIn(false);
      s.data(send.data().word());
      if(!serialWrite(&s)) return false;
      if(print) std::cout << std::setw(3) << i << printHex(send.data());
      
      // Get SR_Q values from SR_OUT
      if(!serialRead(&x)) return false;
      UtlPack xp(x.data());
      if(print) std::cout << "    " << printHex(xp) << std::endl;

      // Put into readback data or verify bits
      rData[i/16].bit((i%16)   ,xp.bit(21));
      rData[i/16].bit((i%16)+16,xp.bit(20));

      // Set clock on and send
      send.srClkIn(true);
      s.data(send.data().word());
      if(!serialWrite(&s)) return false;      
      if(print) std::cout << std::setw(3) << i << printHex(send.data()) << std::endl;
    }

    for(unsigned i(0);i<nWords;i++) {
      if(print) std::cout << std::setw(3) << i << " rData " << printHex(rData[i]) << std::endl;
    }

    // Ensure control is zeroed afterwards
    //if(!writeVfeControlData(c,vfeControlSave)) return false;
    if(!writeVfeControlData(c,0)) return false;

    return true;
  }

public:

  // Write returning previous data from shift register

  bool writeAhcFeConfigurationData(CrcLocation::CrcComponent c, const AhcFeConfigurationDataV0 &w, AhcFeConfigurationDataV0 &r) {
    if(!writeFeConfigurationData(c,w)) return false;

    CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;
    
    CrcSerialCommandFeAddress a(t);
    a.address(24);
    if(!serialWrite(&a)) return false;

    CrcSerialCommandWord s(t,CrcSerialHeader::feDataOut);
    CrcSerialCommandWord x(t,CrcSerialHeader::feSpyRegister);


    // Define bits
    unsigned srResInBot(6);
    unsigned srDInBot(7);
    unsigned selOutBot(8);
    unsigned srClkInBot(9);
    unsigned swHoldInBot(10);
    unsigned swDacInBot(11);
    //unsigned ledSelBot(15);
    
    unsigned srResInTop(0);
    unsigned srDInTop(1);
    unsigned selOutTop(2);
    unsigned srClkInTop(3);
    unsigned swHoldInTop(4);
    unsigned swDacInTop(5);
    //unsigned ledSelTop(13);
    
    // Set up initial signals
    UtlPack send;

    send.bit(srDInBot,true);
    send.bit(srDInTop,true);
    send.bit(swHoldInBot,true);
    send.bit(swHoldInTop,true);
    send.bit(swDacInBot,true);
    send.bit(swDacInTop,true);
    s.data(send.word());
    if(!serialWrite(&s)) return false;

    // Reset shift register and select SROUT
    send.bit(srResInBot,true);/////
    send.bit(srResInTop,true);/////
    s.data(send.word());
    if(!serialWrite(&s)) return false;
    
    send.bit(srResInBot,false);/////
    send.bit(srResInTop,false);/////
    s.data(send.word());
    if(!serialWrite(&s)) return false;
    
    send.bit(selOutBot,true);
    send.bit(selOutTop,true);
    s.data(send.word());
    if(!serialWrite(&s)) return false;
    
    // Send initial verification bit
    send.bit(srClkInBot,true);
    send.bit(srClkInTop,true);
    s.data(send.word());
    if(!serialWrite(&s)) return false;

    for(unsigned i(0);i<6;i++) {

      // Changed from i to 5-i to reverse board order 28/4/05
      const AhcVfeDataV0* srwBot(w.srData(CrcFeConfigurationData::bot,5-i));
      const AhcVfeDataV0* srwTop(w.srData(CrcFeConfigurationData::top,  i));
      AhcVfeDataV0* srrBot(r.srData(CrcFeConfigurationData::bot,5-i));
      AhcVfeDataV0* srrTop(r.srData(CrcFeConfigurationData::top,  i));
      
      // Do 18 DACs first
      for(unsigned j(0);j<18;j++) {
	
	// Changed from j to 17-j to reverse DAC order 28/4/05
	unsigned char wDataBot(srwBot->dac(17-j));
	unsigned char wDataTop(srwTop->dac(17-j));
	
	unsigned char rDataBot(0);
	unsigned char rDataTop(0);
      
	// 8 bits for DAC
	for(unsigned k(0);k<8;k++) {
	  
	  // Set clock off and next data bit
	  send.bit(srClkInBot,false);
	  send.bit(srClkInTop,false);	
	  // Changed from k to 7-k to reverse bit order 28/4/05
	  send.bit(srDInBot,((wDataBot>>(7-k))&0x1)==1);
	  send.bit(srDInTop,((wDataTop>>(7-k))&0x1)==1);
	  s.data(send.word());
	  if(!serialWrite(&s)) return false;
	  
	  // Set clock on
	  send.bit(srClkInBot,true);
	  send.bit(srClkInTop,true);
	  s.data(send.word());
	  if(!serialWrite(&s)) return false;
	  
	  // Get SROUT value and put in read data
	  if(!serialRead(&x)) return false;
	  // Changed from k to 7-k to reverse bit order 28/4/05
	  if((x.data()&0x00200000)!=0) rDataBot|=1<<(7-k);
	  if((x.data()&0x00100000)!=0) rDataTop|=1<<(7-k);

	  /*
	    if((x.data()&0x00200000)!=0) {
	    std::cout << "1";
	    } else {
	    std::cout << "0";
	    }
	    if(k==31) std::cout << std::endl;
	  */

	}
	// Write read data into output object
	// Changed from j to 17-j to reverse DAC order 28/4/05
	srrBot->dac(17-j,rDataBot);
	srrTop->dac(17-j,rDataTop);
      }

      // Do SR last
      unsigned short wDataBot(srwBot->shiftRegister().data());
      unsigned short wDataTop(srwTop->shiftRegister().data());
      
      unsigned short rDataBot(0);
      unsigned short rDataTop(0);
      
      // Only 16 bits for SR
      for(unsigned k(0);k<16;k++) {
	
	// Set clock off and next data bit
	send.bit(srClkInBot,false);
	send.bit(srClkInTop,false);
	send.bit(srDInBot,((wDataBot>>k)&0x1)==1);
	send.bit(srDInTop,((wDataTop>>k)&0x1)==1);
	s.data(send.word());
	if(!serialWrite(&s)) return false;
	
	// Set clock on
	send.bit(srClkInBot,true);
	send.bit(srClkInTop,true);
	s.data(send.word());
	if(!serialWrite(&s)) return false;
	
	// Get SROUT value and put in read data
	if(!serialRead(&x)) return false;
	if((x.data()&0x00200000)!=0) rDataBot|=1<<k;
	if((x.data()&0x00100000)!=0) rDataTop|=1<<k;

	/*
	  if((x.data()&0x00200000)!=0) {
	  std::cout << "1";
	  } else {
	  std::cout << "0";
	  }
	  if(k==31) std::cout << std::endl;
	*/
   }

      // Write read data into output object
      srrBot->shiftRegister(AhcVfeShiftRegister(rDataBot));
      srrTop->shiftRegister(AhcVfeShiftRegister(rDataTop));
    }
    
    // Set final lines
    //send.word(0);
    //send.bit(swDacInBot,true);
    //send.bit(ledSelBot,true);
    //send.bit(swDacInTop,true);
    //send.bit(ledSelTop,true);
    //s.data(send.word());
    s.data(w.vfeControl().data().word());
    if(!serialWrite(&s)) return false;

    r.print(std::cout);

    return true;
  }

  /*
  bool writeAhcFeConfigurationData(CrcLocation::CrcComponent c,
				   const AhcFeConfigurationData &w,
				   AhcFeConfigurationData &r) {

    if(!writeFeConfigurationData(c,w)) return false;
    if( !readFeConfigurationData(c,r)) return false;

    // Define arrays of data to be loaded
    UtlPack wDataA[4],wDataB[3],rDataA[4],rDataB[3];

    // Pack shift registers in reverse HAB order for board A
    for(unsigned i(0);i<6;i++) {
      wDataA[i/2].halfWord(i%2,w.shiftRegister(5-i));
      wDataB[i/2].halfWord(i%2,w.shiftRegister(6+i));
    }

    wDataA[3]=w.vfeControl();

    // Do data transfer
    if(!loadAhcVfeData(c,3,wDataA,wDataB,rDataA,rDataB,false,false) return false;

    // Unpack shift registers in reverse HAB order for board A
    for(unsigned i(0);i<6;i++) {
      r.shiftRegister(5-i,rDataA[i/2].halfWord(i%2));
      r.shiftRegister(6+i,rDataB[i/2].halfWord(i%2));
    }

    // Set and readback final lines
    CrcSerialCommandFeAddress a(t);
    a.address(24);
    if(!serialWrite(&a)) return false;
    
    CrcSerialCommandWord sw(t,CrcSerialHeader::feDataOut);
    sw.data(w.vfeControl());
    if(!serialWrite(&sw)) return false;

    CrcSerialCommandWord sr(t,CrcSerialHeader::feDataIn);
    if(!serialRead(&sr)) return false;
    r.vfeControl(sr.data());
    
    //r.print(std::cout);

    return true;
  }
*/
       /*

    // Define bits
    unsigned srResInBot(6);
    unsigned srDInBot(7);
    unsigned selOutBot(8);
    unsigned srClkInBot(9);
    unsigned swHoldInBot(10);
    //unsigned swDacInBot(11);
    //unsigned ledSelBot(15);
    
    unsigned srResInTop(0);
    unsigned srDInTop(1);
    unsigned selOutTop(2);
    unsigned srClkInTop(3);
    unsigned swHoldInTop(4);
    //unsigned swDacInTop(5);
    //unsigned ledSelTop(13);
    
    // Set up initial signals
    UtlPack send(0);
    send.bit(srDInBot,true);
    send.bit(srDInTop,true);
    send.bit(swHoldInBot,true);
    send.bit(swHoldInTop,true);
    s.data(send.word());
    if(!serialWrite(&s)) return false;

    // Reset shift register and select SROUT
    if(false) {
      send.bit(srResInBot,true);/////
      send.bit(srResInTop,true);/////
      s.data(send.word());
      if(!serialWrite(&s)) return false;
      
      send.bit(srResInBot,false);/////
      send.bit(srResInTop,false);/////
      s.data(send.word());
      if(!serialWrite(&s)) return false;
    }
    
    send.bit(selOutBot,true);
    send.bit(selOutTop,true);
    s.data(send.word());
    if(!serialWrite(&s)) return false;
    
    // Zero readback data
    unsigned short rDataA[6];
    unsigned short rDataB[6];
    unsigned rBit(0);

    for(unsigned i(0);i<6;i++) {
      rDataA[i]=0;
      rDataB[i]=0;
    }

    // Get SROUT value and put in read data
    if(!serialRead(&x)) return false;
    if((x.data()&0x00200000)!=0) rDataA[5-rBit/16]|=1<<(rBit%16);
    if((x.data()&0x00100000)!=0) rDataB[  rBit/16]|=1<<(rBit%16);
    rBit++;

    // Send initial verification bit
    send.bit(srClkInBot,true);
    send.bit(srClkInTop,true);
    s.data(send.word());
    if(!serialWrite(&s)) return false;

    // Get SROUT value and put in read data
    if(!serialRead(&x)) return false;
    if((x.data()&0x00200000)!=0) rDataA[5-rBit/16]|=1<<(rBit%16);
    if((x.data()&0x00100000)!=0) rDataB[  rBit/16]|=1<<(rBit%16);
    rBit++;

    for(unsigned i(0);i<6;i++) {
      
      // Changed from i to 5-i to reverse board order 28/4/05
      unsigned short wDataA(w.shiftRegister(CrcFeConfigurationData::bot,5-i));
      unsigned short wDataB(w.shiftRegister(CrcFeConfigurationData::top,  i));
      
      // Only 16 bits for SR
      for(unsigned k(0);k<16;k++) {
	
	// Set clock off and next data bit
	send.bit(srClkInBot,false);
	send.bit(srClkInTop,false);
	send.bit(srDInBot,((wDataA>>k)&0x1)==1);
	send.bit(srDInTop,((wDataB>>k)&0x1)==1);
	s.data(send.word());
	if(!serialWrite(&s)) return false;
	
	// Set clock on
	send.bit(srClkInBot,true);
	send.bit(srClkInTop,true);
	s.data(send.word());
	if(!serialWrite(&s)) return false;
	
	// Get SROUT value and put in read data
	if(!serialRead(&x)) return false;
	if((x.data()&0x00200000)!=0) rDataA[5-rBit/16]|=1<<(rBit%16);
	if((x.data()&0x00100000)!=0) rDataB[  rBit/16]|=1<<(rBit%16);
	rBit++;
      }
    }
      
    // Write read data into output object
    for(unsigned i(0);i<6;i++) {
      r.shiftRegister(CrcFeConfigurationData::bot,i,rDataA[i]);
      r.shiftRegister(CrcFeConfigurationData::top,i,rDataB[i]);
    }

    */



  bool writeAhcVfeConfigurationData(CrcLocation::CrcComponent c, const AhcVfeConfigurationDataFine &w, AhcVfeConfigurationDataFine &r) {

    //w.print(std::cout,"VME WRIT ");

    // Define arrays of data to be loaded
    UtlPack wData[7],rData[7]; // 7 x 4 bytes = 12 x 2 bytes + 4 bytes

    // Pack verification data which goes in first
    wData[0]=w.verificationData();

    // Pack shift registers in reverse HAB order for board A
    for(unsigned i(0);i<6;i++) {
      wData[1+i].halfWord(0,w.shiftRegister(5-i).data());
      wData[1+i].halfWord(1,w.shiftRegister(6+i).data());
    }

    // Do data transfer
    if(!loadAhcVfeData(c,7,wData,rData,false,false)) return false;
    
    // Unpack verification data which come out last
    r.verificationData(rData[6]);

    // Unpack shift registers in reverse HAB order for board A
    for(unsigned i(0);i<6;i++) {
      r.shiftRegister(5-i,AhcVfeShiftRegister(rData[i].halfWord(0)));
      r.shiftRegister(6+i,AhcVfeShiftRegister(rData[i].halfWord(1)));
    }

    //w.print(std::cout,"VME READ ");

    return true;
  }


  bool writeAhcVfeConfigurationData(CrcLocation::CrcComponent c, const AhcVfeConfigurationDataCoarse &w, AhcVfeConfigurationDataCoarse &r) {

    // Define arrays of data to be loaded
    UtlPack wData[5],rData[5]; // 5 x 4 bytes = 8 x 2 bytes + 4 bytes

    // Pack verification data which goes in first
    wData[0]=w.verificationData();

    // Pack shift registers in reverse HAB order for board A
    for(unsigned i(0);i<4;i++) {
      wData[1+i].halfWord(0,w.shiftRegister(3-i).data());
      wData[1+i].halfWord(1,w.shiftRegister(8+i).data());
    }

    // Do data transfer
    if(!loadAhcVfeData(c,5,wData,rData,false,false)) return false;
    
    // Unpack verification data which come out last
    r.verificationData(rData[4]);

    // Unpack shift registers in reverse HAB order for board A
    for(unsigned i(0);i<4;i++) {
      r.shiftRegister(3-i,AhcVfeShiftRegister(rData[i].halfWord(0)));
      r.shiftRegister(8+i,AhcVfeShiftRegister(rData[i].halfWord(1)));
    }

    return true;
  }


  bool writeAhcVfeStartUpData(CrcLocation::CrcComponent c, const AhcVfeStartUpDataFine &w, AhcVfeStartUpDataFine &r, bool reset=true) {

    // Zero VFE control word
    if(!writeVfeControlData(c,0)) return false;

    // Define arrays of data to be loaded
    UtlPack wData[55],rData[55]; // 55 x 4 bytes = 12 x 18 x 1 byte + 4 bytes

    // Pack verification data which goes in first
    wData[0]=w.verificationData();

    // Pack DACs in reverse HAB order for board A and reverse channel order
    for(unsigned i(0);i<6;i++) {
      for(unsigned j(0);j<18;j++) {
	UtlPack dacA(w.dac(5-i,17-j));
	UtlPack dacB(w.dac(6+i,17-j));
	
	// Pack DAC values in reverse bit order
	for(unsigned k(0);k<8;k++) {
	  wData[1+9*i+j/2].bit(8*(j%2)+k   ,dacA.bit(7-k));
	  wData[1+9*i+j/2].bit(8*(j%2)+k+16,dacB.bit(7-k));
	}
      }
    }
    
    // Do data transfer
    if(!loadAhcVfeData(c,55,wData,rData,reset,true)) return false;
    
    // Unpack verification data which come out last
    r.verificationData(rData[54]);

    for(unsigned i(0);i<6;i++) {
      for(unsigned j(0);j<18;j++) {
	UtlPack dacA;
	UtlPack dacB;
	
	// Unpack DAC values in reverse bit order
	for(unsigned k(0);k<8;k++) {
	  dacA.bit(7-k,rData[9*i+j/2].bit(8*(j%2)+k   ));
	  dacB.bit(7-k,rData[9*i+j/2].bit(8*(j%2)+k+16));
	}

	// Unpack DACs in reverse HAB order for board A and reverse channel order
	r.dac(5-i,17-j,dacA.byte(0));
	r.dac(6+i,17-j,dacB.byte(0));
      }
    }

    return true;
  }

  bool writeAhcVfeStartUpData(CrcLocation::CrcComponent c, const AhcVfeStartUpDataCoarse &w, AhcVfeStartUpDataCoarse &r, bool reset=true) {

    // Zero VFE control word
    if(!writeVfeControlData(c,0)) return false;

    // Define arrays of data to be loaded
    UtlPack wData[37],rData[37]; // 37 x 4 bytes = 8 x 18 x 1 byte + 4 bytes

    // Pack verification data which goes in first
    wData[0]=w.verificationData();

    // Pack DACs in reverse HAB order for board A and reverse channel order
    for(unsigned i(0);i<4;i++) {
      for(unsigned j(0);j<18;j++) {
	UtlPack dacA(w.dac(3-i,17-j));
	UtlPack dacB(w.dac(8+i,17-j));
	
	// Pack DAC values in reverse bit order
	for(unsigned k(0);k<8;k++) {
	  wData[1+9*i+j/2].bit(8*(j%2)+k   ,dacA.bit(7-k));
	  wData[1+9*i+j/2].bit(8*(j%2)+k+16,dacB.bit(7-k));
	}
      }
    }
    
    // Do data transfer
    if(!loadAhcVfeData(c,37,wData,rData,reset,true)) return false;
    
    // Unpack verification data which come out last
    r.verificationData(rData[36]);

    for(unsigned i(0);i<4;i++) {
      for(unsigned j(0);j<18;j++) {
	UtlPack dacA;
	UtlPack dacB;
	
	// Unpack DAC values in reverse bit order
	for(unsigned k(0);k<8;k++) {
	  dacA.bit(7-k,rData[9*i+j/2].bit(8*(j%2)+k   ));
	  dacB.bit(7-k,rData[9*i+j/2].bit(8*(j%2)+k+16));
	}

	// Unpack DACs in reverse HAB order for board A and reverse channel order
	r.dac(3-i,17-j,dacA.byte(0));
	r.dac(8+i,17-j,dacB.byte(0));
      }
    }

    return true;
  }

/*
  bool writeAhcVfeStartUpDataFine(CrcLocation::CrcComponent c, const AhcVfeStartUpDataFine &w, AhcVfeStartUpDataFine &r, bool reset=true) {

    CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;
    
    CrcSerialCommandFeAddress a(t);
    a.address(24);
    if(!serialWrite(&a)) return false;

    CrcSerialCommandWord s(t,CrcSerialHeader::feDataOut);
    CrcSerialCommandWord x(t,CrcSerialHeader::feSpyRegister);

    // Now do shift register

    // Define bits
    unsigned srResInBot(6);
    unsigned srDInBot(7);
    unsigned selOutBot(8);
    unsigned srClkInBot(9);
    unsigned swHoldInBot(10);
    unsigned swDacInBot(11);
    //unsigned ledSelBot(15);
    
    unsigned srResInTop(0);
    unsigned srDInTop(1);
    unsigned selOutTop(2);
    unsigned srClkInTop(3);
    unsigned swHoldInTop(4);
    unsigned swDacInTop(5);
    //unsigned ledSelTop(13);
    
    // Set up initial signals
    UtlPack send(0);
    send.bit(srDInBot,true);
    send.bit(srDInTop,true);
    send.bit(swHoldInBot,true);
    send.bit(swHoldInTop,true);
    send.bit(swDacInBot,true);
    send.bit(swDacInTop,true);
    s.data(send.word());
    if(!serialWrite(&s)) return false;

    // Reset shift register and select SROUT
    if(reset) {
      send.bit(srResInBot,true);/////
      send.bit(srResInTop,true);/////
      s.data(send.word());
      if(!serialWrite(&s)) return false;
      
      send.bit(srResInBot,false);/////
      send.bit(srResInTop,false);/////
      s.data(send.word());
      if(!serialWrite(&s)) return false;
    }
      
    send.bit(selOutBot,true);
    send.bit(selOutTop,true);
    s.data(send.word());
    if(!serialWrite(&s)) return false;
    
    // Send initial verification bit
    send.bit(srClkInBot,true);
    send.bit(srClkInTop,true);
    s.data(send.word());
    if(!serialWrite(&s)) return false;

    for(unsigned i(0);i<6;i++) {
      for(unsigned j(0);j<18;j++) {
	
	// Changed from i to 5-i to reverse board order 28/4/05
	// Changed from j to 17-j to reverse DAC order 28/4/05
	unsigned char wDataBot(w.dac(CrcFeConfigurationData::bot,5-i,17-j));
	unsigned char wDataTop(w.dac(CrcFeConfigurationData::top,  i,17-j));
	
	unsigned char rDataBot(0);
	unsigned char rDataTop(0);
      
	// 8 bits for DAC
	for(unsigned k(0);k<8;k++) {
	  
	  // Set clock off and next data bit
	  send.bit(srClkInBot,false);
	  send.bit(srClkInTop,false);	
	  // Changed from k to 7-k to reverse bit order 28/4/05
	  send.bit(srDInBot,((wDataBot>>(7-k))&0x1)==1);
	  send.bit(srDInTop,((wDataTop>>(7-k))&0x1)==1);
	  s.data(send.word());
	  if(!serialWrite(&s)) return false;
	  
	  // Set clock on
	  send.bit(srClkInBot,true);
	  send.bit(srClkInTop,true);
	  s.data(send.word());
	  if(!serialWrite(&s)) return false;
	  
	  // Get SROUT value and put in read data
	  if(!serialRead(&x)) return false;
	  // Changed from k to 7-k to reverse bit order 28/4/05
	  if((x.data()&0x00200000)!=0) rDataBot|=1<<(7-k);
	  if((x.data()&0x00100000)!=0) rDataTop|=1<<(7-k);
	}
	
	// Write read data into output object
	r.dac(CrcFeConfigurationData::bot,5-i,17-j,rDataBot);
	r.dac(CrcFeConfigurationData::bot,  i,17-j,rDataTop);
      }
    }
    
    // Set final lines
    s.data(0);
    if(!serialWrite(&s)) return false;

    //r.print(std::cout);

    return true;
  }


  // Write without returning previous data

  bool writeAhcFeConfigurationData(CrcLocation::CrcComponent c, const AhcFeConfigurationDataV0 &w) {
    AhcFeConfigurationDataV0 r;
    return writeAhcFeConfigurationData(c,w,r);
  }

  bool writeAhcFeConfigurationData(CrcLocation::CrcComponent c, const AhcFeConfigurationData &w) {
    AhcFeConfigurationData r;
    return writeAhcFeConfigurationData(c,w,r);
  }

  bool writeAhcVfeSlowControlsData(CrcLocation::CrcComponent c, const AhcVfeSlowControlsData &w) {
    AhcVfeSlowControlsData r;
    return writeAhcVfeSlowControlsData(c,w,r);
  }
*/
  bool readAhcFeConfigurationData(CrcLocation::CrcComponent c, AhcFeConfigurationDataV0 &w) {

    // DOES NOT RETURN THE SERIAL REGISTER AND DAC INFO!!! SET TO DEFAULT
    AhcVfeDataV0 def;
    AhcVfeDataV0 *p(w.srData((CrcFeConfigurationData::Connector)0,0));
    for(unsigned i(0);i<12;i++) p[i]=def;

    return readFeConfigurationData(c,w);
  }
  /*
  bool readAhcFeConfigurationData(CrcLocation::CrcComponent c, AhcFeConfigurationData &w) {

    // DOES NOT RETURN THE SERIAL REGISTER INFO!!! SET TO ZERO
    for(unsigned i(0);i<2;i++) {
      for(unsigned j(0);j<6;j++) {
	w.shiftRegister((CrcFeConfigurationData::Connector)i,j,0);
      }
    }

    return readFeConfigurationData(c,w);
  }

  bool readAhcVfeSlowControlsData(CrcLocation::CrcComponent c, AhcVfeSlowControlsData &w) {
    return false;
  }
*/
  bool writeFeFakeEventData(CrcLocation::CrcComponent c, const CrcFeFakeEventData &d) {

    CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;

    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataOut);

    const unsigned *p(d.data());
    for(unsigned i(0);i<d.numberOfWords();i++) {
      a.address(i+512);
      if(!serialWrite(&a)) return false;
      s.data(p[i]);
      if(!serialWrite(&s)) return false;
    }

    a.address(511);
    if(!serialWrite(&a)) return false;
    s.data(d.control());
    if(!serialWrite(&s)) return false;

    return true;
  }

  bool readFeFakeEventData(CrcLocation::CrcComponent c, CrcFeFakeEventData &d) {
    CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;

    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataIn);

    a.address(511);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.control(s.data());

    // ENABLE prevents readback!
    if(d.enable()) {
      CrcSerialCommandWord w(t,CrcSerialHeader::feDataOut);
      w.data(0);
      if(!serialWrite(&w)) return false;
    }

    unsigned *p(d.data());
    for(unsigned i(0);i<d.numberOfWords();i++) {
      a.address(i+512);
      if(!serialWrite(&a)) return false;
      if(!serialRead(&s)) return false;
      p[i]=s.data();
    }

    if(d.enable()) {
      CrcSerialCommandWord w(t,CrcSerialHeader::feDataOut);
      w.data(d.control());
      if(!serialWrite(&w)) return false;
    }

    return true;
  }


  /*
  bool enableFeFakeEvent(CrcLocation::CrcComponent c, bool e=true) {

    CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;

    CrcSerialCommandFeAddress a(t);
    a.address(511);
    if(!serialWrite(&a)) return false;

    CrcSerialCommandWord r(t,CrcSerialHeader::feDataIn);
    if(!serialRead(&r)) return false;

    CrcSerialCommandWord w(t,CrcSerialHeader::feDataOut);
    if(e) w.data(0x80000000 | r.data());
    else  w.data(0x7fffffff & r.data());
    return serialWrite(&w);
  }
  */

  bool readFeEventData(CrcLocation::CrcComponent c, CrcFeEventData &d) {

    CrcSerialHeader::Target t(crcComponentToTarget(c));
    if(!CrcSerialHeader::feTarget(t)) return false;

    CrcSerialCommandWord s(t,CrcSerialHeader::feTriggerCounter);
    if(!serialRead(&s)) return false;
    d.triggerCounter(s.data());

    s.designator(CrcSerialHeader::feSpyRegister);
    if(!serialRead(&s)) return false;
    d.spyRegister(s.data());

    return true;
  }


  // ***** Slow controls *****

  bool writeI2C(CrcLocation::CrcComponent c, unsigned char a, unsigned char d) {
    if(c==CrcLocation::vmeLm82   ) return writeVmeI2C("LM82",a,d);
    if(c==CrcLocation::vmeAdm1025) return writeVmeI2C("ADM1025A",a,d);

    CrcSerialHeader::Target t(crcComponentToTarget(c));
    CrcSerialHeader::Designator cd(CrcSerialHeader::beLm82Control);
    CrcSerialHeader::Designator sd(CrcSerialHeader::beLm82Status);

    if(c!=CrcLocation::be) {
      if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
      if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
      if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
      if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
      if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
      if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
      if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
      if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
      cd=CrcSerialHeader::feLm82Control;
      sd=CrcSerialHeader::feLm82Status;
    }

    if(t==CrcSerialHeader::null) return false;

    CrcSerialCommandI2CControl control(t,cd);
    control.set(a,d,false);
    serialWrite(&control);

    CrcSerialCommandShort status(t,sd);
    status.data(0x100);
    while((status.data()&0x100)!=0) {
      if(!serialRead(&status)) return false;
    }
    if((status.data()&0x300)!=0) return false;

    return true;
  }

  bool readI2C(CrcLocation::CrcComponent c, unsigned char a, unsigned char &d) {
    if(c==CrcLocation::vmeLm82   ) return readVmeI2C("LM82",a,d);
    if(c==CrcLocation::vmeAdm1025) return readVmeI2C("ADM1025A",a,d);

    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::be) t=CrcSerialHeader::be;
    CrcSerialHeader::Designator cd(CrcSerialHeader::beLm82Control);
    CrcSerialHeader::Designator sd(CrcSerialHeader::beLm82Status);

    if(c!=CrcLocation::be) {
      if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
      if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
      if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
      if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
      if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
      if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
      if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
      if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
      cd=CrcSerialHeader::feLm82Control;
      sd=CrcSerialHeader::feLm82Status;
    }

    if(t==CrcSerialHeader::null) return false;

    CrcSerialCommandI2CControl control(t,cd);
    control.set(a,0,true);
    if(!serialWrite(&control)) return false;

    CrcSerialCommandShort status(t,sd);
    status.data(0x100);
    while((status.data()&0x100)!=0) {
      if(!serialRead(&status)) return false;
    }
    if((status.data()&0x300)!=0) return false;

    d=status.data()&0xff;
    return true;
  }

  bool writeVmeI2C(std::string dev, unsigned char add, unsigned char dat) {
    /*unsigned long*/ uint32_t value;

    write(dev+"Control",(add<<8)+dat);
    pollItem(dev+"BusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);

    return value==0;
  }

  bool readVmeI2C(std::string dev, unsigned char add, unsigned char &dat) {
    /*unsigned long*/ uint32_t value;

    write(dev+"Control",0x10000+(add<<8));
    pollItem(dev+"BusyError",0x1,1000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);
    if(value!=0) return false;

    read(dev+"Status",&value);
    dat=value&0xff;
    return true;
  }

  bool readLm82Run(CrcLocation::CrcComponent c, CrcLm82RunData &d) {
    unsigned char val;
    if(!readI2C(c,0xfe,val)) return false;
    d.manufacturerId(val);
    if(!readI2C(c,0xff,val)) return false;
    d.steppingCode(val);
    return true;
  }

  bool writeLm82Configuration(CrcLocation::CrcComponent c, const CrcLm82ConfigurationData &d) {
    if(!writeI2C(c,0x09,0x28)) return false; // Set these bits first
    if(!writeI2C(c,0x0b,d.localLimit())) return false;
    if(!writeI2C(c,0x0d,d.remoteLimit())) return false;
    if(!writeI2C(c,0x5a,d.criticalLimit())) return false;

    if(!writeI2C(c,0x09,d.configuration())) return false;
    return true;
  }

  bool readLm82Configuration(CrcLocation::CrcComponent c, CrcLm82ConfigurationData &d) {
    unsigned char val;
    if(!readI2C(c,0x03,val)) return false;
    d.configuration(val);
    if(!readI2C(c,0x42,val)) return false;
    d.criticalLimit(val);
    if(!readI2C(c,0x05,val)) return false;
    d.localLimit(val);
    if(!readI2C(c,0x07,val)) return false;
    d.remoteLimit(val);
    return true;
  }

  bool readLm82SlowReadout(CrcLocation::CrcComponent c, CrcLm82SlowReadoutData &d) {
    unsigned char val;
    if(!readI2C(c,0x02,val)) return false;
    d.status(val);
    if(!readI2C(c,0x00,val)) return false;
    d.localTemperature(val);
    if(!readI2C(c,0x01,val)) return false;
    d.remoteTemperature(val);
    return true;
  }

  bool readAdm1025Run(CrcAdm1025RunData &d) {
    unsigned char val;
    if(!readI2C(CrcLocation::vmeAdm1025,0x3e,val)) return false;
    d.manufacturerId(val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x3f,val)) return false;
    d.steppingCode(val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x47,val)) return false;
    d.vid(val&0x0f);
    if(!readI2C(CrcLocation::vmeAdm1025,0x49,val)) return false;
    d.vid(d.vid()|((val&0x01)<<4));
    return true;
  }

  bool writeAdm1025Configuration(const CrcAdm1025ConfigurationData &d) {
    if(!writeI2C(CrcLocation::vmeAdm1025,0x40,0x00)) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x39,d.localHighLimit())) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x3a,d.localLowLimit())) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x37,d.remoteHighLimit())) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x38,d.remoteLowLimit())) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x47,d.vid())) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x15,d.test())) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x1f,d.offset())) return false;

    CrcAdm1025Voltages v;
    
    v=d.voltageHighLimits();
    if(!writeI2C(CrcLocation::vmeAdm1025,0x2b,v.voltage(CrcAdm1025Voltages::v25))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x2d,v.voltage(CrcAdm1025Voltages::vccp))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x2f,v.voltage(CrcAdm1025Voltages::v33))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x31,v.voltage(CrcAdm1025Voltages::v50))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x33,v.voltage(CrcAdm1025Voltages::v120))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x35,v.voltage(CrcAdm1025Voltages::vcc))) return false;

    v=d.voltageLowLimits();
    if(!writeI2C(CrcLocation::vmeAdm1025,0x2c,v.voltage(CrcAdm1025Voltages::v25))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x2e,v.voltage(CrcAdm1025Voltages::vccp))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x30,v.voltage(CrcAdm1025Voltages::v33))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x32,v.voltage(CrcAdm1025Voltages::v50))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x34,v.voltage(CrcAdm1025Voltages::v120))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x36,v.voltage(CrcAdm1025Voltages::vcc))) return false;

    // Finally, write configuration including setting monitor bit
    if(!writeI2C(CrcLocation::vmeAdm1025,0x40,d.configuration()|0x01)) return false;
    return true;
  }

  bool readAdm1025Configuration(CrcAdm1025ConfigurationData &d) {
    unsigned char val;
    if(!readI2C(CrcLocation::vmeAdm1025,0x40,val)) return false;
    d.configuration(val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x39,val)) return false;
    d.localHighLimit(val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x3a,val)) return false;
    d.localLowLimit(val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x37,val)) return false;
    d.remoteHighLimit(val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x38,val)) return false;
    d.remoteLowLimit(val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x47,val)) return false;
    d.vid(val&0xc0);
    if(!readI2C(CrcLocation::vmeAdm1025,0x15,val)) return false;
    d.test(val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x1f,val)) return false;
    d.offset(val);

    CrcAdm1025Voltages v;

    if(!readI2C(CrcLocation::vmeAdm1025,0x2b,val)) return false;
    v.voltage(CrcAdm1025Voltages::v25,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x2d,val)) return false;
    v.voltage(CrcAdm1025Voltages::vccp,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x2f,val)) return false;
    v.voltage(CrcAdm1025Voltages::v33,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x31,val)) return false;
    v.voltage(CrcAdm1025Voltages::v50,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x33,val)) return false;
    v.voltage(CrcAdm1025Voltages::v120,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x35,val)) return false;
    v.voltage(CrcAdm1025Voltages::vcc,val);
    d.voltageHighLimits(v);

    if(!readI2C(CrcLocation::vmeAdm1025,0x2c,val)) return false;
    v.voltage(CrcAdm1025Voltages::v25,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x2e,val)) return false;
    v.voltage(CrcAdm1025Voltages::vccp,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x30,val)) return false;
    v.voltage(CrcAdm1025Voltages::v33,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x32,val)) return false;
    v.voltage(CrcAdm1025Voltages::v50,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x34,val)) return false;
    v.voltage(CrcAdm1025Voltages::v120,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x36,val)) return false;
    v.voltage(CrcAdm1025Voltages::vcc,val);
    d.voltageLowLimits(v);

    return true;
  }

  bool readAdm1025SlowReadout(CrcAdm1025SlowReadoutData &d) {
    /*unsigned long*/ uint32_t value;

    write("ADM1025AControl",0x14100);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    d.status1(value&0xff);

    write("ADM1025AControl",0x14200);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    d.status2(value&0xff);

    write("ADM1025AControl",0x12700);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    d.localTemperature(value&0xff);

    write("ADM1025AControl",0x12600);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    d.remoteTemperature(value&0xff);
    

    CrcAdm1025Voltages v;
    
    write("ADM1025AControl",0x12000);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    v.voltage(CrcAdm1025Voltages::v25,value&0xff);

    write("ADM1025AControl",0x12100);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    v.voltage(CrcAdm1025Voltages::vccp,value&0xff);

    write("ADM1025AControl",0x12200);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    v.voltage(CrcAdm1025Voltages::v33,value&0xff);

    write("ADM1025AControl",0x12300);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    v.voltage(CrcAdm1025Voltages::v50,value&0xff);

    write("ADM1025AControl",0x12400);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    v.voltage(CrcAdm1025Voltages::v120,value&0xff);

    write("ADM1025AControl",0x12500);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    v.voltage(CrcAdm1025Voltages::vcc,value&0xff);

    d.voltages(v);

    return true;
  }

  CrcSerialHeader::Target crcComponentToTarget(CrcLocation::CrcComponent c) const {
    CrcSerialHeader::Target t(CrcSerialHeader::null);

    if(c==CrcLocation::fe0        ) t=CrcSerialHeader::fe0;
    if(c==CrcLocation::fe1        ) t=CrcSerialHeader::fe1;
    if(c==CrcLocation::fe2        ) t=CrcSerialHeader::fe2;
    if(c==CrcLocation::fe3        ) t=CrcSerialHeader::fe3;
    if(c==CrcLocation::fe4        ) t=CrcSerialHeader::fe4;
    if(c==CrcLocation::fe5        ) t=CrcSerialHeader::fe5;
    if(c==CrcLocation::fe6        ) t=CrcSerialHeader::fe6;
    if(c==CrcLocation::fe7        ) t=CrcSerialHeader::fe7;
    if(c==CrcLocation::feBroadcast) t=CrcSerialHeader::feBroadcast;
    if(c==CrcLocation::beTrg      ) t=CrcSerialHeader::beTrg;
    if(c==CrcLocation::be         ) t=CrcSerialHeader::be;

    return t;
  }

  void print(std::ostream &o) {
    /*unsigned long*/ uint32_t value;
    o << "CrcVmeDevice::print()" << std::endl;

    read("FirmwareId",&value);
    o << std::hex << "  FirmwareId = 0x" << value << std::dec << std::endl;
    read("ClockSelect",&value);
    o << std::hex << "  ClockSelect = 0x" << value << std::dec << std::endl;
    read("VmeStatus",&value);
    o << std::hex << "  VmeStatus = 0x" << value << std::dec << std::endl;
    read("SerialStatus",&value);
    o << std::hex << "  SerialStatus = 0x" << value << std::dec << std::endl;
    read("TTCClockCounter",&value);
    o << std::hex << "  TTCClockCounter = 0x" << value << std::dec << std::endl;
    read("BackplaneClockCounter",&value);
    o << std::hex << "  BackplaneClockCounter = 0x" << value << std::dec << std::endl;

    o << std::endl;
  }

private:
  unsigned _printLevel;
  unsigned _slot;
  unsigned vlinkTotalLengthSum;

#ifdef FAKE_SPILL
#ifdef SPILL_INPUT
  static bool _spillInvert;
#endif
#endif

};

#ifdef FAKE_SPILL
#ifdef SPILL_INPUT
bool CrcVmeDevice::_spillInvert(false);
#endif
#endif

#endif
