#ifndef CrcVmeDevice_HH
#define CrcVmeDevice_HH

#include <iostream>
#include <vector>

// hal
#include "VMEDevice.hh"
#include "VMEAddressTable.hh"
#include "VMEBusAdapterInterface.hh"

// dual/inc/crc
#include "CrcLocation.hh"
#include "CrcVmeRunData.hh"
#include "CrcVmeEventData.hh"
#include "CrcBeRunData.hh"
#include "CrcBeConfigurationData.hh"
#include "CrcBeEventData.hh"
#include "CrcBeTrgRunData.hh"
#include "CrcBeTrgConfigurationData.hh"
#include "CrcBeTrgEventData.hh"
#include "CrcBeTrgPollData.hh"
#include "CrcFeRunData.hh"
#include "CrcFeConfigurationData.hh"
#include "EmcFeConfigurationData.hh"
#include "AhcFeConfigurationData.hh"
#include "CrcFeFakeEventData.hh"
#include "CrcFeEventData.hh"
#include "CrcVlinkEventData.hh"
#include "CrcLm82StartupData.hh"
#include "CrcLm82SlowReadoutData.hh"
#include "CrcLm82SlowControlsData.hh"
#include "CrcAdm1025StartupData.hh"
#include "CrcAdm1025SlowReadoutData.hh"
#include "CrcAdm1025SlowControlsData.hh"

// online/inc/crc
#include "CrcSerialHeader.hh"
#include "CrcSerialCommandI2CControl.hh"
#include "CrcSerialCommandWord.hh"
#include "CrcSerialCommandShort.hh"
#include "CrcSerialCommandByte.hh"
#include "CrcSerialCommandBit.hh"
#include "CrcSerialCommandFeAddress.hh"
#include "CrcSerialCommandBeCounter.hh"
#include "CrcSerialCommandBeBxCounter.hh"
#include "CrcSerialCommandBeDaqId.hh"
#include "CrcSerialCommandBeQdrDataCounter.hh"
#include "CrcSerialCommandBeControl.hh"
#include "CrcSerialCommandBeMode.hh"
#include "CrcSerialCommandBeTriggerSelect.hh"


bool serialPrint(false);
bool vlinkPrint(false);


class CrcVmeDevice : public VMEDevice {
public:
  CrcVmeDevice(VMEAddressTable &t, VMEBusAdapterInterface &b, unsigned a)
    : VMEDevice(t,b,a*0x10000), _slot(a), vlinkTotalLengthSum(0) {
  }

  unsigned slot() const {
    return _slot;
  }

  bool alive() {
    long unsigned value;
    try {
      read("FirmwareId",&value);
      return true;
    } catch ( std::exception e ) {
      return false;
    }
  }

  void clockSelect(unsigned clock=0) {
    write("ClockSelect",clock);
  }

  bool reset(unsigned clock=0) {
    //for(unsigned i(0);i<10000;i++) 
    writePulse("SoftwareReset");
    write("ClockSelect",clock);
    vlinkTotalLengthSum=0;
    sleep(1);
    feSoftReset();
    return true;
  }

  bool serialWrite(const CrcSerialHeader *h) {
    if(h==0) return false;

    if(serialPrint) {
      cout << std::endl << "serialWrite ";
      h->print(std::cout);
      cout << endl;
    }

    assert(!h->readback());

    unsigned long *b((unsigned long*)h);
    if(serialPrint) std::cout << "serialWrite  SerialBuffer[0] = 0x" 
			      << hex << b[0] << dec << endl;
    write("SerialBuffer",b[0]);

    unsigned nWords((h->length()+31)/32);
    for(unsigned i(0);i<nWords;i++) {
      if(serialPrint) std::cout << "serialWrite  SerialBuffer[" << i+1 << "] = 0x" 
				<< hex << b[i+1] << dec << endl;
      write("SerialBuffer",b[i+1],HAL_NO_VERIFY,4*(i+1));
    }
    write("SerialBuffer",0,HAL_NO_VERIFY,4*(nWords+1));
    write("SerialWrite",nWords+1);

    unsigned long ss(0);
    pollItem("SerialStatus",0xffffffff,1000,&ss);
    if(serialPrint) std::cout << "serialWrite  SerialStatus = 0x" 
			      << hex << ss << dec << endl << endl;
    return ss==0xffffffff;
  }

  bool serialWriteVector(const vector<const CrcSerialHeader*> &v) {
    for(unsigned i(0);i<v.size();i++) {
      if(!serialWrite(v[i])) return false;
    }
    return true;
  }

  bool serialRead(CrcSerialHeader *h) {
    if(h==0) return false;

    h->readback(true);
    if(serialPrint) {
      cout << std::endl << "serialRead ";
      h->print(std::cout);
      cout << endl;
    }

    unsigned long *b((unsigned long*)h);
    unsigned header(b[0]);
    header++; // Increase length
    header|=0x400000; // Set readback bit

    if(h->target()==CrcSerialHeader::be && (h->designator()&0x10)!=0) {
      header++; // Increase length again for RO
      header<<=1; // Shift sideways for RO

    } else if(h->feTarget()) {
      if(h->designator()==30) {
	header++; // Increase length again for RO
	header++; // Increase length again for RO
	header<<=1; // Shift sideways for trigger counter
	header<<=1; // Shift sideways for trigger counter
      }
      if(h->designator()==27) {
	header++; // Increase length again for RO
	header<<=1; // Shift sideways for trigger counter
      }

      if(h->designator()== 3 || 
	 h->designator()== 9 ||
	 h->designator()==10 ||
	 h->designator()==28) {
	header++; // Increase length again for RO
	header++; // Increase length again for RO
	header<<=1; // Shift sideways for RO
      }

      if(h->designator()==20 
	 || h->designator()==29) {
	header++; // Increase length again for RO
	header++; // Increase length again for RO
	//header++; // Increase length again for RO
	//header+=2; // Increase length again for RO
	header<<=1; // Shift sideways for RO
	header<<=1; // Shift sideways for RO
	//header<<=1; // Shift sideways for RO
	//header<<=2; // Shift sideways for RO
      }
    }

    if(serialPrint) cout << "serialRead  Header = 0x" << hex << header << dec << endl;
    write("SerialRead",header);

    unsigned long ss(0);
    pollItem("SerialStatus",0xffffffff,1000,&ss);
    if(serialPrint) cout << "serialRead  SerialStatus = 0x" << hex << ss << dec << endl;
    if(ss!=0xffffffff) return false;

    unsigned nWords((h->length()+31)/32);
    for(unsigned i(0);i<nWords;i++) {
      read("SerialBuffer",b+i+1,4*i);
      if(serialPrint) cout << "serialRead  SerialBuffer[" << i+1 << "] = 0x" 
			   << hex << b[i+1] << dec << endl << endl;
    }

    return true;
  }

  bool serialReadVector(const vector<CrcSerialHeader*> &v) {
    for(unsigned i(0);i<v.size();i++) {
      if(!serialRead(v[i])) return false;
    }
    return true;
  }

  bool readVlinkEventData(CrcVlinkEventData &d, bool blt=false) {
    return vlinkRead(d,blt);
  }

  bool vlinkRead(CrcVlinkEventData &d, bool blt=false) {
    d.numberOfWords(vlinkRead(d.data(),blt));
    return true;
  }

  unsigned vlinkRead(unsigned *d, bool blt=false) {
    if(d==0) return 0;

    if(vlinkPrint) cout << "Entered vlinkRead" << endl;

    if(!isSet("VlinkBufferReady")) {
      if(vlinkPrint) cout << "vlinkRead  VlinkBufferReady not set"<< endl;
      return 0;
    }

    if(isSet("VlinkLastBuffer")) {
      if(vlinkPrint) cout << "Last buffer flag set" << endl;
    }

    unsigned nWords(0);
    unsigned long n(0);
    read("VlinkEventNumber",&n);
    if(vlinkPrint) cout << "VlinkEventNumber " << n << endl;
    d[nWords++]=n;

    bool lastBuffer(false);
    while(isSet("VlinkBufferReady") && !lastBuffer) {
      read("VlinkBufferLength",&n);
      if(vlinkPrint) cout << "vlinkRead  VlinkBufferLength = " << n << endl;

      if(!blt) {
	for(unsigned i(0);i<n;i++) {
	  read("VlinkBuffer",(unsigned long*)(d+nWords+i),4*i);
	  //cout << "vlinkRead  d[" << nWords+i << "] = " << hex << d[nWords+i] << dec << endl;
	}
      } else {
	//readBlock("VlinkBufferBlt",4*n,(char*)(d+nWords));
	for(unsigned i(0);i<n;i+=64) {
	  unsigned m(4*(n-i));
	  if(m>256) m=256;
	  readBlock("VlinkBufferBlt",m,(char*)(d+nWords+i),HAL_DO_INCREMENT,4*i);
	}
      }
      nWords+=n;

      if(isSet("VlinkLastBuffer")) {
	if(vlinkPrint) cout << "Last buffer flag set" << endl;

	lastBuffer=true;
	unsigned long lWords(0);
	read("VlinkTotalLength",&lWords);
	vlinkTotalLengthSum+=nWords-1;
	if(vlinkTotalLengthSum!=lWords) 
	  cout << "vlinkRead ERROR in number of words: read "
	       << nWords-1 << ", VlinkTotalLengthSum " 
	       << vlinkTotalLengthSum << " != VlinkTotalLength " 
	       << lWords << std::endl;

	d[nWords++]=lWords;
	writePulse("VlinkNextEvent");
      } else {
	resetBit("VlinkBufferReady");
		sleep(1);
	if(isSet("VlinkBufferReady")) {
	  if(vlinkPrint) cout << "Buffer ready flag set" << endl;
	} else {
	  if(vlinkPrint) cout << "Buffer ready flag not set" << endl;
	}
      }
    }

    return nWords;
  }


  // ***** EPROM *****

  bool initialiseEprom(const unsigned char s) {
    long unsigned value;

    // Remove protection
    write("EpromControl",(0x7ff<<8)+2);
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    write("EpromControl",(0x7ff<<8)+6);
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    write("EpromControl",(0x7ff<<8)+2);
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    sleep(1);


    // Write the data
    std::cout << "Writing SN = " << (unsigned)s << " to board" << std::endl;

    write("EpromControl",(0x000<<8)+s);
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    sleep(1);

    write("EpromControl",(0x001<<8)+0);
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    sleep(1);

    write("EpromControl",(0x002<<8)+0);
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    sleep(1);

    write("EpromControl",(0x003<<8)+0);
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    sleep(1);


    // Reset protection
    write("EpromControl",(0x7ff<<8)+6);
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    write("EpromControl",(0x7ff<<8)+0x9a);
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    sleep(1);

    write("EpromControl",(0x7ff<<8)+0);
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    return true;
  }

  bool writeEpromMessage(std::string s) {
    long unsigned value;
    
    unsigned n(0);

    write("EpromControl",0x80000+(2<<8));
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    if(value!=0) return false;
    read("EpromStatus",&value);
    n+=(value&0xff);

    write("EpromControl",0x80000+(3<<8));
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    if(value!=0) return false;
    read("EpromStatus",&value);
    n+=(value&0xff)<<8;

    cout << "n = " << n << endl;


    // Remove protection
    write("EpromControl",(0x7ff<<8)+2);
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    write("EpromControl",(0x7ff<<8)+6);
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    write("EpromControl",(0x7ff<<8)+2);
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    sleep(1);


    // Write the data
    UtlPack t(time(0));
    for(unsigned i(0);i<4;i++) {
      write("EpromControl",((n+i+4)<<8)+t.byte(i));
      pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
      std::cout << "Value = " << value << std::endl;
      if(value!=0) return false;
      
      sleep(1);
    }

    UtlPack b(s.size());
    for(unsigned i(0);i<4;i++) {
      write("EpromControl",((n+i+8)<<8)+b.byte(i));
      pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
      std::cout << "Value = " << value << std::endl;
      if(value!=0) return false;
      
      sleep(1);
    }

    const unsigned char *c((const unsigned char*)s.c_str());
    for(unsigned i(0);i<s.size();i++) {
      write("EpromControl",((n+i+12)<<8)+c[i]);
      pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
      std::cout << "Value = " << value << std::endl;
      if(value!=0) return false;
      
      sleep(1);
    }

    n+=8+s.size();
    cout << "n = " << n << endl;


    write("EpromControl",(2<<8)+(n&0xff));
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    sleep(1);

    write("EpromControl",(3<<8)+(n>>8));
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    sleep(1);


    // Reset protection
    write("EpromControl",(0x7ff<<8)+6);
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    write("EpromControl",(0x7ff<<8)+0x9a);
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    sleep(1);

    write("EpromControl",(0x7ff<<8)+0);
    pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
    
    return true;
  }

  /*
  bool writeVmeEprom(unsigned n, const unsigned *p) {
    long unsigned value;

    // Remove protection
    write("EpromControl",(0x7ff<<8)+2);
    pollItem("EpromBusyError",0x1,10000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;


    //write("EpromControl",0x80000+(0x7ff<<8));
	write("EpromControl",0x80000);
	pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
	if(value!=0) return false;
	read("EpromStatus",&value);
    std::cout << "Register = " << value << std::endl;



    write("EpromControl",(0x7ff<<8)+6);
    pollItem("EpromBusyError",0x1,10000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    //write("EpromControl",0x80000+(0x7ff<<8));
	write("EpromControl",0x80100);
	pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
	if(value!=0) return false;
	read("EpromStatus",&value);
    std::cout << "Register = " << value << std::endl;

    write("EpromControl",(0x7ff<<8)+2);
    pollItem("EpromBusyError",0x1,10000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;
 
	write("EpromControl",0x80000+(0x7ff<<8));
	pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
	if(value!=0) return false;
	read("EpromStatus",&value);
    std::cout << "Register = " << value << std::endl;

    for(unsigned i(0);i<=n;i++) {
      UtlPack pack;
      if(i==0) pack.word(n);
      else     pack.word(p[i-1]);

      std::cout << "Word " << setw(4) << i << " = "
		<< printHex(pack.word()) << std::endl;

      for(unsigned j(0);j<4;j++) {
	write("EpromControl",((4*i+j)<<8)+pack.byte(j));
	pollItem("EpromBusyError",0x1,1000000,&value,HAL_POLL_UNTIL_DIFFERENT);
	std::cout << "Value = " << value << std::endl;
	//if(value!=0) return false;
      }
    }

    // Reset protection
    write("EpromControl",(0x7ff<<8)+0x1a);
    pollItem("EpromBusyError",0x1,10000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    //if(value!=0) return false;

    read("FirmwareId",&value);
    read("ClockSelect",&value);

    for(unsigned i(0);i<=n;i++) {
      UtlPack pack;
      for(unsigned j(0);j<4;j++) {
	//write("EpromControl",0x80000+((4*i+j)<<8));
	//pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
	//if(value!=0) return false;
	read("EpromStatus",&value);
	pack.byte(j,value&0xff);
      }
      std::cout << "Word " << setw(4) << i << " = "
		<< printHex(pack.word()) << std::endl;
    }
    
    return true;
 }
*/


  // ***** VME *****

  bool readVmeRunData(CrcVmeRunData &d) {
    long unsigned value;

    read("FirmwareId",&value);
    d.firmwareId(value);
    read("ClockSelect",&value);
    d.clockSelect(value);

    /*
    write("EpromControl",(1<<8)+0xec);
    pollItem("EpromBusyError",0x1,10000,&value,HAL_POLL_UNTIL_DIFFERENT);
    if(value!=0) return false;
    */

    /*
    // Remove protection
    write("EpromControl",(0x7ff<<8)+2);
    pollItem("EpromBusyError",0x1,10000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    write("EpromControl",(0x7ff<<8)+6);
    pollItem("EpromBusyError",0x1,10000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    write("EpromControl",(0x7ff<<8)+2);
    pollItem("EpromBusyError",0x1,10000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    sleep(1);

    write("EpromControl",(0x7fe<<8)+2);
    pollItem("EpromBusyError",0x1,10000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    sleep(1);

    write("EpromControl",(0x000<<8)+0x78);
    pollItem("EpromBusyError",0x1,10000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    sleep(1);

    write("EpromControl",(0x001<<8)+0x56);
    pollItem("EpromBusyError",0x1,10000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    sleep(1);

    write("EpromControl",(0x002<<8)+0x34);
    pollItem("EpromBusyError",0x1,10000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    sleep(1);

    write("EpromControl",(0x003<<8)+0x12);
    pollItem("EpromBusyError",0x1,10000,&value,HAL_POLL_UNTIL_DIFFERENT);
    std::cout << "Value = " << value << std::endl;
    if(value!=0) return false;

    sleep(1);
    */

    /*

    for(unsigned i(0);i<512;i++) {
      for(unsigned j(0);j<4;j++) {
	write("EpromControl",((4*i+j)<<8)+0xaa);
    pollItem("EpromBusyError",0x1,10000,&value,HAL_POLL_UNTIL_DIFFERENT);
    if(value==0) {
      std::cout << "Add " << setw(4) << 4*i+j << " Value = " << value << std::endl;
    }
      }
    }
    */

    write("EpromControl",0x80000+((0x7ff)<<8));
    for(unsigned j(0);j<16;j++) {
      read("EpromStatus",&value);
      //cout << "WPR Byte = " << printHex((unsigned)value) << endl;
    }
    
    sleep(1); // Gives 0xff for first char otherwise?!?

    for(unsigned i(0);i<4;i++) {
      write("EpromControl",0x80000+(i<<8));
      pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
      if(value!=0) return false;

      read("EpromStatus",&value);
      //cout << "Header[" << i << "] = " << printHex((unsigned)value) << endl;

      d.epromHeader(i,value&0xff);
    }


    unsigned bytes(d.epromHeader()>>16);
    unsigned rBytes(0);
    //cout << "Number of bytes = " << bytes << endl;

    //unsigned char *c(d.data());

    //for(unsigned i(0);i<(d.epromHeader()>>16) || i<4;i++) {
    //for(unsigned i(0);i<(d.epromHeader()>>16) && i<8;i++) { // LIMIT FOR NOW!!!

      while(rBytes<bytes) {
	UtlPack time;
	for(unsigned j(0);j<4;j++) {
	  write("EpromControl",0x80000+((4+j)<<8));
	  pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
	  if(value!=0) return false;
	  read("EpromStatus",&value);
	  time.byte(j,value&0xff);
	}
	rBytes+=4;

	UtlPack leng;
	for(unsigned j(0);j<4;j++) {
	  write("EpromControl",0x80000+((8+j)<<8));
	  pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
	  if(value!=0) return false;
	  read("EpromStatus",&value);
	  leng.byte(j,value&0xff);
	}
	rBytes+=4+leng.word();
      }

      /*

      //cout << "Word " << setw(3) << i << " = " << printHex(pack.word()) << endl;      UtlPack time;
      for(unsigned j(0);j<4;j++) {
	write("EpromControl",0x80000+((4*i+4+j)<<8));
	pollItem("EpromBusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
	if(value!=0) return false;
	read("EpromStatus",&value);
	if(i<(d.epromHeader()>>16)) c[4*i+4+j]=value&0xff;

	pack.byte(j,value&0xff);
      }
      //cout << "Word " << setw(3) << i << " = " << printHex(pack.word()) << endl;
    }

      */
    
    return true;
  }

  bool readVmeEventData(CrcVmeEventData &d) {
    long unsigned value;

    read("VmeStatus",&value);
    d.status(value);

    read("TTCClockCounter",&value);
    d.ttcCounter(value);

    read("BackplaneClockCounter",&value);
    d.backplaneCounter(value);

    return true;
  }


  // ***** BE *****

  bool beSoftReset() {
    return true;
  }

  bool clearBeFifos() {
    CrcSerialCommandBit s(CrcSerialHeader::be,
			  CrcSerialHeader::beSoftReset);
    s.bit(false);
    return serialWrite(&s);
  }

  bool beSoftTrigger() {
    CrcSerialCommandBit s(CrcSerialHeader::be,
			  CrcSerialHeader::beSoftTrigger);
    s.bit(false);
    return serialWrite(&s);
  }

  bool readBeRunData(CrcBeRunData &d) {
    CrcSerialCommandWord w(CrcSerialHeader::be,CrcSerialHeader::beFirmwareId);
    if(!serialRead(&w)) return false;
    d.firmwareId(w.data());

    return true;
  }

  bool writeBeConfigurationData(const CrcBeConfigurationData &d) {
    CrcSerialCommandBeTriggerSelect b3(CrcSerialHeader::be,CrcSerialHeader::beTriggerSelect);
    b3.data(d.triggerSelect());
    if(!serialWrite(&b3)) return false;

    CrcSerialCommandBeMode b4;
    b4.data(d.mode());
    if(!serialWrite(&b4)) return false;
    
    CrcSerialCommandBeControl b2(CrcSerialHeader::be,CrcSerialHeader::beReadoutControl);
    b2.data(d.readoutControl());
    if(!serialWrite(&b2)) return false;    

    b2.designator(CrcSerialHeader::beRunControl);
    b2.data(d.runControl());
    if(!serialWrite(&b2)) return false;
    
    CrcSerialCommandWord w(CrcSerialHeader::be,CrcSerialHeader::beTest);
    w.data(d.test());
    if(!serialWrite(&w)) return false;    

    CrcSerialCommandByte de(CrcSerialHeader::be,CrcSerialHeader::beFeDataEnable);
    de.data(d.feDataEnable());
    if(!serialWrite(&de)) return false;

    CrcSerialCommandBeDaqId id;
    id.data(d.daqId());
    if(!serialWrite(&id)) return false;    

    w.designator(CrcSerialHeader::beFeTrgEnable);
    w.data(d.trgEnables());
    if(!serialWrite(&w)) return false;

    CrcSerialCommandWord te(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgTestLength); // N.B. from beTrg!
    //te.designator(CrcSerialHeader::beTrgTestLength); // N.B. from beTrg!
    te.data(d.testLength());
    if(!serialWrite(&te)) return false;

    te.designator(CrcSerialHeader::beTrgQdrAddress); // N.B. from beTrg!
    te.data(d.qdrAddress());
    if(!serialWrite(&te)) return false;


    // OLD !!!
    CrcSerialCommandWord tr(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgFirmwareDate);
    if(!serialRead(&tr)) return false;
    if(tr.data()<=1100536550) {
      te.designator(CrcSerialHeader::beTrgOutputEnable);
      te.data(0xff);

      //cout << "OLD!!! SETTING ENABLES TO 0xff" << endl;

      if(!serialWrite(&te)) return false;
    }


    return true;
  }

  bool readBeConfigurationData(CrcBeConfigurationData &d) {
    CrcSerialCommandBeTriggerSelect b3(CrcSerialHeader::be,CrcSerialHeader::beTriggerSelect);
    if(!serialRead(&b3)) return false;
    d.triggerSelect(b3.data());

    CrcSerialCommandBeMode b4;
    if(!serialRead(&b4)) return false;
    d.mode(b4.data());

    CrcSerialCommandBeControl b2(CrcSerialHeader::be,CrcSerialHeader::beReadoutControl);
    if(!serialRead(&b2)) return false;
    d.readoutControl(b2.data());

    b2.designator(CrcSerialHeader::beRunControl);
    if(!serialRead(&b2)) return false;
    d.runControl(b2.data());

    CrcSerialCommandWord w(CrcSerialHeader::be,CrcSerialHeader::beTest);
    if(!serialRead(&w)) return false;
    d.test(w.data());

    CrcSerialCommandByte de(CrcSerialHeader::be,CrcSerialHeader::beFeDataEnable);
    if(!serialRead(&de)) return false;
    d.feDataEnable(de.data());

    CrcSerialCommandBeDaqId id;
    if(!serialRead(&id)) return false;
    d.daqId(id.data());

    w.designator(CrcSerialHeader::beFeTrgEnable);
    if(!serialRead(&w)) return false;
    d.trgEnables(w.data());

    CrcSerialCommandWord te(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgTestLength); // N.B. from beTrg!
    //te.designator(CrcSerialHeader::beTrgTestLength); // N.B. from beTrg!
    if(!serialRead(&te)) return false;
    d.testLength(te.data());

    te.designator(CrcSerialHeader::beTrgQdrAddress); // N.B. from beTrg!
    if(!serialRead(&te)) return false;
    d.qdrAddress(te.data());

    return true;
  }

  bool readBeEventData(CrcBeEventData &d) {
    CrcSerialCommandWord w(CrcSerialHeader::be,CrcSerialHeader::beStatus);
    if(!serialRead(&w)) return false;
    d.status(w.data());

    CrcSerialCommandBeCounter c(CrcSerialHeader::beL1aCounter);
    if(!serialRead(&c)) return false;
    d.l1aCounter(c.data());

    CrcSerialCommandBeBxCounter b12;
    if(!serialRead(&b12)) return false;
    d.bxCounter(b12.data());

    c.designator(CrcSerialHeader::beQdrFrameCounter);
    if(!serialRead(&c)) return false;
    d.qdrFrameCounter(c.data());

    CrcSerialCommandBeQdrDataCounter b18;
    if(!serialRead(&b18)) return false;
    d.qdrDataCounter(b18.data());

    c.designator(CrcSerialHeader::beTotalFrameCounter);
    if(!serialRead(&c)) return false;
    d.totalFrameCounter(c.data());

    CrcSerialCommandWord tc(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgSignalCatch); // N.B. from beTrg!
    if(!serialRead(&tc)) return false;
    d.signalCatch(tc.data());

    return true;
  }


  // ***** BE-Trigger *****

  bool beTrgSoftReset() {
    CrcSerialCommandWord r(CrcSerialHeader::beTrg,
			   CrcSerialHeader::beTrgCommands);
    if(!serialRead(&r)) return false;

    CrcSerialCommandWord w(CrcSerialHeader::beTrg,
			   CrcSerialHeader::beTrgCommands);
    w.data(r.data() | 0x2);
    if(!serialWrite(&w)) return true;

    //if(!serialRead(&r)) return false;
    //std::cout << "***** beTrgSoftReset **** data = " << r.data() << std::endl;

    return true;
  }


  bool clearBeTrgFifos() {
    CrcSerialCommandWord w(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgPrebusyTriggerCounter);
    w.data(0);
    if(!serialWrite(&w)) return false;

    w.designator(CrcSerialHeader::beTrgTriggerCounter);
    w.data(0);
    if(!serialWrite(&w)) return false;

    // Flush out FIFOs by reading them
    unsigned array[1024];
    CrcBeTrgEventData *d((CrcBeTrgEventData*)array);
    readBeTrgEventData(*d,false); // NOT squirt as no trigger!
    //d->print(cout);

    //CrcSerialCommand<32*CrcBeTrgEventData::maxFifoLength>
    //  f(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgFifoSquirt);
    //if(!serialRead(&f)) return false;

    return true;
  }

  bool beTrgSoftTrigger() {
    CrcSerialCommandWord r(CrcSerialHeader::beTrg,
			   CrcSerialHeader::beTrgCommands);
    if(!serialRead(&r)) return false;
    //std::cout << "***** beTrgSoftTrigger **** data = " << r.data() << std::endl;

    CrcSerialCommandWord w(CrcSerialHeader::beTrg,
			   CrcSerialHeader::beTrgCommands);

    w.data(r.data() & 0xfffffffe);
    if(!serialWrite(&w)) return false;

    w.data(r.data() | 0x00000001);
    if(!serialWrite(&w)) return false;

    w.data(r.data() & 0xfffffffe);
    return serialWrite(&w);
  }

  bool pollBeTrgTrigger(CrcBeTrgPollData &d, unsigned n=1000) {
    d.update();

    CrcSerialCommandWord r(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgControl);
    d.tries(0);
    d.maximumTries(true);

    for(unsigned i(0);i<n;i++) {
      if(!serialRead(&r)) return false;

      if((r.data()&1)==0x1) {
	d.tries(i);
	d.maximumTries(false);
	return true;
      }
    }

    cout << "TrgControl = " << r.data() << endl;

    d.tries(n);
    d.maximumTries(true);
    return false;
  }

  bool pollBeTrgTrigger(unsigned n=1000) {
    CrcSerialCommandWord r(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgControl);

    bool notBusy(true);
    for(unsigned i(0);(i<n || n==0) && notBusy;i++) {
      if(!serialRead(&r)) return false;
      notBusy=(r.data()&0x1)==0;
      if(!notBusy) n=i;
    }

    if(notBusy)
    std::cout << "pollBeTrgTrigger failed after " << n << " iterations" << std::endl;
    else
    std::cout << "pollBeTrgTrigger took " << n << " iterations" << std::endl;

    CrcSerialCommandWord tc(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgRamAddress);
    tc.data(0);
    if(!serialWrite(&tc)) return false;

    tc.designator(CrcSerialHeader::beTrgRamData);
    if(notBusy) tc.data(n);
    else        tc.data((1<<31)+n);
    if(!serialWrite(&tc)) return false;

    return true;
  }

  bool beTrgForceBusy(bool b) {
    CrcSerialCommandWord r(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgControl);
    if(!serialRead(&r)) return false;

    CrcSerialCommandWord w(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgControl);
    if(b) w.data(r.data()|0x00000002);
    else  w.data(r.data()&0xfffffffd);
    if(!serialWrite(&w)) return false;

    return true;
  }

  bool clearBeTrgTrigger() {
    CrcSerialCommandWord r(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgControl);
    if(!serialRead(&r)) return false;
    //std::cout << "**** trig read = " << r.data() << " *****" << std::endl;

    CrcSerialCommandWord tc(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgSignalCatch); // N.B. from beTrg!
    tc.data(0);
    if(!serialWrite(&tc)) return false;

    tc.designator(CrcSerialHeader::beTrgInputCatch);
    tc.data(0);
    if(!serialWrite(&tc)) return false;

    CrcSerialCommandWord w(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgControl);
    w.data(r.data()&0xfffffffe);
    if(!serialWrite(&w)) return false;

    //if(!serialRead(&r)) return false;
    //std::cout << "**** trig read = " << r.data() << " *****" << std::endl;

    return true;
  }

  bool readBeTrgRunData(CrcBeTrgRunData &d) {
    CrcSerialCommandWord w(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgFirmwareId);
    if(!serialRead(&w)) return false;
    d.firmwareId(w.data());

    w.designator(CrcSerialHeader::beTrgFirmwareDate);
    if(!serialRead(&w)) return false;
    d.firmwareDate(w.data());

    return true;
  }

  bool writeBeTrgConfigurationData(const CrcBeTrgConfigurationData &d) {
    // Hold off all triggers
    beTrgForceBusy(true);

    CrcSerialCommandWord w(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgInputEnable);
    w.data(d.inputEnable());
    if(!serialWrite(&w)) return false;

    // NEW !!!
    CrcSerialCommandWord tr(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgFirmwareDate);
    if(!serialRead(&tr)) return false;
    if(tr.data()>1100536550) {

      //cout << "NEW!!! SETTING ENABLES TO 0xff" << endl;

      w.designator(CrcSerialHeader::beTrgOutputEnable);
      w.data(d.outputEnable());
      if(!serialWrite(&w)) return false;
    }

    w.designator(CrcSerialHeader::beTrgOscillationPeriod);
    w.data(d.oscillationPeriod());
    if(!serialWrite(&w)) return false;

    return true;
  }

  bool readBeTrgConfigurationData(CrcBeTrgConfigurationData &d) {
    CrcSerialCommandWord w(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgInputEnable);
    if(!serialRead(&w)) return false;
    d.inputEnable(w.data());

    w.designator(CrcSerialHeader::beTrgOutputEnable);
    if(!serialRead(&w)) return false;
    d.outputEnable(w.data());

    w.designator(CrcSerialHeader::beTrgOscillationPeriod);
    if(!serialRead(&w)) return false;
    d.oscillationPeriod(w.data());

    /*
    w.designator((CrcSerialHeader::Designator)24);
    if(!serialRead(&w)) return false;
    std::cout << "readBeTrgEventData  Desig24 = " << printHex(w.data()) << std::endl;

    w.designator((CrcSerialHeader::Designator)25);
    if(!serialRead(&w)) return false;
    std::cout << "readBeTrgEventData  Desig25 = " << printHex(w.data()) << std::endl;
    */

    return true;
  }

  bool readBeTrgEventData(CrcBeTrgEventData &d, bool squirt=false) {
    d.numberOfFifoWords(0);

    CrcSerialCommandWord w(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgInputStatus);
    if(!serialRead(&w)) return false;
    d.inputStatus(w.data());

    w.designator(CrcSerialHeader::beTrgInputCatch);
    if(!serialRead(&w)) return false;
    d.inputCatch(w.data());

    /*
    CrcSerialCommandWord tc(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgRamAddress);
    tc.data(0);
    if(!serialWrite(&tc)) return false;

    w.designator(CrcSerialHeader::beTrgRamData);
    if(!serialRead(&w)) return false;
    d.triggerBusyStatus(w.data());
    */

    w.designator(CrcSerialHeader::beTrgControl);
    if(!serialRead(&w)) return false;
    d.triggerBusyStatus(w.data());

    w.designator(CrcSerialHeader::beTrgPrebusyTriggerCounter);
    if(!serialRead(&w)) return false;
    d.prebusyTriggerCounter(w.data());

    w.designator(CrcSerialHeader::beTrgTriggerCounter);
    if(!serialRead(&w)) return false;
    d.triggerCounter(w.data());

    w.designator(CrcSerialHeader::beTrgFifoStatus);
    if(!serialRead(&w)) return false;
    d.initialFifoStatus(w.data());

    unsigned fWords(w.data()&0x3ff);
    if((w.data()&0x200)!=0) fWords++; // Full has one more word!

    unsigned *pf(d.fifoData());

    if(squirt) {
      CrcSerialCommand<32*CrcBeTrgEventData::maxFifoLength>
	f(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgFifoSquirt);

      //serialPrint=true;
      if(!serialRead(&f)) return false;
      //serialPrint=false;

      const unsigned *qf(f.data());
      for(unsigned i(0);i<fWords && i<CrcBeTrgEventData::maxFifoLength;i++) {
	pf[i]=qf[i];
      }

      if(fWords<256) d.numberOfFifoWords(fWords);
      else           d.numberOfFifoWords(CrcBeTrgEventData::maxFifoLength);

    } else {
      CrcSerialCommandWord f(CrcSerialHeader::beTrg,CrcSerialHeader::beTrgFifo);

      for(unsigned i(0);i<fWords;i++) {
	if(!serialRead(&f)) return false;
	//if(f.data()!=0)
	//std::cout << "readBeTrgEventData  FIFO word " << setw(4) << i
	//	  << " = " << printHex(f.data()) << std::endl;
	
	if(i<CrcBeTrgEventData::maxFifoLength) {
	  pf[i]=f.data();
	  d.numberOfFifoWords(i+1);
	}
	
	if(i<(fWords-1)) {
	  if(!serialRead(&w)) return false;
	  if(w.data()!=(fWords-1-i)) {
	    std::cout << "readBeTrgEventData  FIFO status   "
		      << " = " << printHex(w.data()) << std::endl;
	    return false;
	  }
	}
      }
    }

    w.designator(CrcSerialHeader::beTrgFifoStatus);
    if(!serialRead(&w)) return false;
    d.finalFifoStatus(w.data());

    return true;
  }


  // ***** FE *****

  bool feSoftReset() {
    CrcSerialCommandBit s(CrcSerialHeader::feBroadcast,
			  CrcSerialHeader::feSoftReset);
    s.bit(false);
    if(!serialWrite(&s)) return false;

    sleep(1);

    // Ensure HOLD is left open
    CrcSerialCommandFeAddress a(CrcSerialHeader::feBroadcast);
    CrcSerialCommandWord w(CrcSerialHeader::feBroadcast,
			   CrcSerialHeader::feDataOut);

    // Need to set HOLD not to be inverted
    a.address(6);
    if(!serialWrite(&a)) return false;
    w.data(0);
    if(!serialWrite(&w)) return false;

    // Need to send one mplex clock
    a.address(23);
    if(!serialWrite(&a)) return false;
    w.data(1);
    if(!serialWrite(&w)) return false;

    // Trigger is necessary to get HOLD un-invert to work
    //cout << "ABOUT TO BROADCAST TRIGGER " << endl;
    feSoftTrigger();

    return true;
  }

  bool feSoftReset(CrcLocation::CrcComponent c) {
    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
    if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
    if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
    if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
    if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
    if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
    if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
    if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
    if(t==CrcSerialHeader::null) return false;

    CrcSerialCommandBit s(t,CrcSerialHeader::feSoftReset);
    s.bit(false);
    if(!serialWrite(&s)) return false;

    // Ensure HOLD is left open
    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord w(t,CrcSerialHeader::feDataOut);

    //CrcFeConfigurationData cd;
    //cd.holdInvert(false);
    //writeFeConfigurationData(c,cd);

    // Need to set HOLD not to be inverted
    a.address(6);
    if(!serialWrite(&a)) return false;
    w.data(0);
    if(!serialWrite(&w)) return false;

    // Need to send one mplex clock
    a.address(23);
    if(!serialWrite(&a)) return false;
    w.data(1);
    if(!serialWrite(&w)) return false;

    // Trigger is necessary to get HOLD un-invert to work
    //cout << "ABOUT TO TRIGGER " << endl;
    feSoftTrigger(c);

    return true;
  }

  bool clearFeFifos() {
    CrcSerialCommandWord w(CrcSerialHeader::feBroadcast,
			   CrcSerialHeader::feTriggerCounter);
    w.data(0);
    if(!serialWrite(&w)) return false;

    CrcSerialCommandBit s(CrcSerialHeader::feBroadcast,
			  CrcSerialHeader::feFifoReset);
    s.bit(false);
    if(!serialWrite(&s)) return false;

    return true;
  }

  bool feSoftTrigger() {
    CrcSerialCommandBit s(CrcSerialHeader::feBroadcast,
			  CrcSerialHeader::feSoftTrigger);
    s.bit(false);
    return serialWrite(&s);
  }

  bool feSoftTrigger(CrcLocation::CrcComponent c) {
    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
    if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
    if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
    if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
    if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
    if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
    if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
    if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
    if(t==CrcSerialHeader::null) return false;

    //std::cout << "FE" << (unsigned)c << " SOFT TRIGGER" << std::endl;

    CrcSerialCommandBit s(t,CrcSerialHeader::feSoftTrigger);
    s.bit(false);
    return serialWrite(&s);
  }

  bool feZeroTriggerCounter() {
    CrcSerialCommandWord s(CrcSerialHeader::feBroadcast,
			   CrcSerialHeader::feTriggerCounter);
    s.data(0);
    return serialWrite(&s);
  }

  bool feZeroTriggerCounter(CrcLocation::CrcComponent c) {
    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
    if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
    if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
    if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
    if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
    if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
    if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
    if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
    if(t==CrcSerialHeader::null) return false;

    CrcSerialCommandWord s(t,CrcSerialHeader::feTriggerCounter);
    s.data(0);
    return serialWrite(&s);
  }


  bool readFeRunData(CrcLocation::CrcComponent c, CrcFeRunData &d) {
    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
    if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
    if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
    if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
    if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
    if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
    if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
    if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
    if(t==CrcSerialHeader::null) return false;

    CrcSerialCommandWord w(t,CrcSerialHeader::feFirmwareId);
    if(!serialRead(&w)) return false;
    d.firmwareId(w.data());

    w.designator(CrcSerialHeader::feLinkArray);
    if(!serialRead(&w)) return false;
    d.linkArray(w.data());

    w.designator(CrcSerialHeader::feVfeType);
    if(!serialRead(&w)) return false;
    d.vfeType(w.data());

    return true;
  }

  /*
  bool writeFeConfigurationData(CrcLocation::CrcComponent c, const CrcFeConfigurationData &d) {
    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
    if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
    if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
    if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
    if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
    if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
    if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
    if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
    if(t==CrcSerialHeader::null) return false;
    
    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataOut);

    const unsigned *p(d.data());
    for(unsigned i(0);i<(sizeof(CrcFeConfigurationData)+3)/4;i++) {
      a.address(i);
      if(!serialWrite(&a)) return false;
      s.data(p[i]);
      if(!serialWrite(&s)) return false;
    }

    return true;
  }

  bool readFeConfigurationData(CrcLocation::CrcComponent c, CrcFeConfigurationData &d) {
    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
    if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
    if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
    if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
    if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
    if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
    if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
    if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
    if(t==CrcSerialHeader::null) return false;
    
    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataIn);

    unsigned *p(d.data());
    for(unsigned i(0);i<(sizeof(CrcFeConfigurationData)+3)/4;i++) {
      a.address(i);
      if(!serialWrite(&a)) return false;
      if(!serialRead(&s)) return false;
      p[i]=s.data();
    }

    return true;
  }
*/

private:
  bool writeFeConfigurationData(CrcLocation::CrcComponent c, const CrcFeConfigurationData &d) {
    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
    if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
    if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
    if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
    if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
    if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
    if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
    if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
    if(t==CrcSerialHeader::null) return false;
    
    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataOut);

    a.address(0);
    if(!serialWrite(&a)) return false;
    s.data(d.calibWidth());
    if(!serialWrite(&s)) return false;

    a.address(1);
    if(!serialWrite(&a)) return false;
    s.data(d.calibControl());
    if(!serialWrite(&s)) return false;

    a.address(4);
    if(!serialWrite(&a)) return false;
    s.data(d.holdStart());
    if(!serialWrite(&s)) return false;

    a.address(5);
    if(!serialWrite(&a)) return false;
    s.data(d.holdWidth());
    if(!serialWrite(&s)) return false;

    a.address(6);
    if(!serialWrite(&a)) return false;
    s.data(d.holdControl());
    if(!serialWrite(&s)) return false;

    a.address(12);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeResetStart());
    if(!serialWrite(&s)) return false;

    a.address(13);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeResetEnd());
    if(!serialWrite(&s)) return false;

    a.address(16);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeSrinStart());
    if(!serialWrite(&s)) return false;

    a.address(17);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeSrinEnd());
    if(!serialWrite(&s)) return false;

    a.address(20);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeMplexClockStart());
    if(!serialWrite(&s)) return false;

    a.address(21);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeMplexClockMark());
    if(!serialWrite(&s)) return false;

    a.address(22);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeMplexClockSpace());
    if(!serialWrite(&s)) return false;

    a.address(23);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeMplexClockPulses());
    if(!serialWrite(&s)) return false;

    a.address(28);
    if(!serialWrite(&a)) return false;
    s.data(d.adcStart());
    if(!serialWrite(&s)) return false;

    a.address(29);
    if(!serialWrite(&a)) return false;
    s.data(d.adcEnd());
    if(!serialWrite(&s)) return false;

    a.address(30);
    if(!serialWrite(&a)) return false;
    s.data(d.adcControl());
    if(!serialWrite(&s)) return false;

    a.address(31);
    if(!serialWrite(&a)) return false;
    s.data(d.adcDelay());
    if(!serialWrite(&s)) return false;

    a.address(32);
    if(!serialWrite(&a)) return false;
    s.data(d.dacData(CrcFeConfigurationData::bot));
    if(!serialWrite(&s)) return false;

    a.address(33);
    if(!serialWrite(&a)) return false;
    s.data(d.dacData(CrcFeConfigurationData::top));
    if(!serialWrite(&s)) return false;

    a.address(34);
    if(!serialWrite(&a)) return false;
    s.data(d.frameSyncDelay());
    if(!serialWrite(&s)) return false;

    a.address(35);
    if(!serialWrite(&a)) return false;
    s.data(d.qdrDataDelay());
    if(!serialWrite(&s)) return false;

    return true;
  }

  bool readFeConfigurationData(CrcLocation::CrcComponent c, CrcFeConfigurationData &d) {
    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
    if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
    if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
    if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
    if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
    if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
    if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
    if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
    if(t==CrcSerialHeader::null) return false;
    
    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataIn);

    a.address(0);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.calibWidth(s.data());

    a.address(1);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.calibControl(s.data());

    a.address(4);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.holdStart(s.data());

    a.address(5);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.holdWidth(s.data());

    a.address(6);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.holdControl(s.data());

    a.address(12);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeResetStart(s.data());

    a.address(13);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeResetEnd(s.data());

    a.address(16);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeSrinStart(s.data());

    a.address(17);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeSrinEnd(s.data());

    a.address(20);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeMplexClockStart(s.data());

    a.address(21);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeMplexClockMark(s.data());

    a.address(22);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeMplexClockSpace(s.data());

    a.address(23);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeMplexClockPulses(s.data());

    a.address(28);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.adcStart(s.data());

    a.address(29);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.adcEnd(s.data());

    a.address(30);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.adcControl(s.data()&0xff);

    a.address(31);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.adcDelay(s.data());

    a.address(32);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.dacData(CrcFeConfigurationData::bot,s.data()&0xffff);

    a.address(33);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.dacData(CrcFeConfigurationData::top,s.data()&0xffff);

    a.address(34);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.frameSyncDelay(s.data());

    a.address(35);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.qdrDataDelay(s.data());

    return true;
  }

public:
  bool writeEmcFeConfigurationData(CrcLocation::CrcComponent c, const EmcFeConfigurationData &d) {
    if(!writeFeConfigurationData(c,d)) return false;

    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
    if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
    if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
    if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
    if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
    if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
    if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
    if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
    if(t==CrcSerialHeader::null) return false;
    
    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataOut);

    a.address(24);
    if(!serialWrite(&a)) return false;
    s.data(d.vfeInformation());
    if(!serialWrite(&s)) return false;

    return true;
  }

  bool readEmcFeConfigurationData(CrcLocation::CrcComponent c, EmcFeConfigurationData &d) {
    if(!readFeConfigurationData(c,d)) return false;

    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
    if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
    if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
    if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
    if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
    if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
    if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
    if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
    if(t==CrcSerialHeader::null) return false;
    
    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataIn);

    a.address(24);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.vfeInformation(s.data());

    return true;
  }

  // Write returning previous data from shift register

  bool writeAhcFeConfigurationData(CrcLocation::CrcComponent c, const AhcFeConfigurationData &w, AhcFeConfigurationData &r) {
    if(!writeFeConfigurationData(c,w)) return false;

    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
    if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
    if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
    if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
    if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
    if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
    if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
    if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
    if(t==CrcSerialHeader::null) return false;
    
    CrcSerialCommandFeAddress a(t);
    a.address(24);
    if(!serialWrite(&a)) return false;

    CrcSerialCommandWord s(t,CrcSerialHeader::feDataOut);
    CrcSerialCommandWord x(t,CrcSerialHeader::feSpyRegister);

    // Now do shift register

    // Define bits
    unsigned srResInBot(6);
    unsigned srDInBot(7);
    unsigned selOutBot(8);
    unsigned srClkInBot(9);
    unsigned swHoldInBot(10);
    //unsigned swDacInBot(11);
    //unsigned ledSelBot(15);
    
    unsigned srResInTop(0);
    unsigned srDInTop(1);
    unsigned selOutTop(2);
    unsigned srClkInTop(3);
    unsigned swHoldInTop(4);
    //unsigned swDacInTop(5);
    //unsigned ledSelTop(13);
    
    // Set up initial signals
    UtlPack send(0);
    send.bit(srDInBot,true);
    send.bit(srDInTop,true);
    send.bit(swHoldInBot,true);
    send.bit(swHoldInTop,true);
    s.data(send.word());
    if(!serialWrite(&s)) return false;

    // Reset shift register and select SROUT
    send.bit(srResInBot,true);/////
    send.bit(srResInTop,true);/////
    s.data(send.word());
    if(!serialWrite(&s)) return false;
    
    send.bit(srResInBot,false);/////
    send.bit(srResInTop,false);/////
    s.data(send.word());
    if(!serialWrite(&s)) return false;
    
    send.bit(selOutBot,true);
    send.bit(selOutTop,true);
    s.data(send.word());
    if(!serialWrite(&s)) return false;
    
    // Send initial verification bit
    send.bit(srClkInBot,true);
    send.bit(srClkInTop,true);
    s.data(send.word());
    if(!serialWrite(&s)) return false;

    for(unsigned i(0);i<6;i++) {
      const AhcFeSrConfigurationData* srwBot(w.srData(CrcFeConfigurationData::bot,i));
      const AhcFeSrConfigurationData* srwTop(w.srData(CrcFeConfigurationData::top,i));
      AhcFeSrConfigurationData* srrBot(r.srData(CrcFeConfigurationData::bot,i));
      AhcFeSrConfigurationData* srrTop(r.srData(CrcFeConfigurationData::top,i));
      
      for(unsigned j(0);j<5;j++) {
	unsigned wDataBot(srwBot->word(j));
	unsigned wDataTop(srwTop->word(j));

	unsigned rDataBot(0);
	unsigned rDataTop(0);
	
	for(unsigned k(0);k<32;k++) {
	  
	  // Set clock off and next data bit
	  send.bit(srClkInBot,false);
	  send.bit(srClkInTop,false);
	  send.bit(srDInBot,((wDataBot>>k)&0x1)==1);
	  send.bit(srDInTop,((wDataTop>>k)&0x1)==1);
	  s.data(send.word());
	  if(!serialWrite(&s)) return false;
	  
	  // Set clock on
	  send.bit(srClkInBot,true);
	  send.bit(srClkInTop,true);
	  s.data(send.word());
	  if(!serialWrite(&s)) return false;
	  
	  // Get SROUT value and put in read data
	  if(!serialRead(&x)) return false;
	  if((x.data()&0x00200000)!=0) rDataBot|=1<<k;
	  if((x.data()&0x00100000)!=0) rDataTop|=1<<k;

	  /*
	  if((x.data()&0x00200000)!=0) {
	    cout << "1";
	  } else {
	    cout << "0";
	  }
	  if(k==31) cout << endl;
	  */
	}
	
	// Write read data into output object
	srrBot->word(j,rDataBot);
	srrTop->word(j,rDataTop);
      }
    }

    // Set final lines
    //send.word(0);
    //send.bit(swDacInBot,true);
    //send.bit(ledSelBot,true);
    //send.bit(swDacInTop,true);
    //send.bit(ledSelTop,true);
    //s.data(send.word());
    s.data(w.vfeInformation());
    if(!serialWrite(&s)) return false;

    //r.print(std::cout);

    return true;
  }


  // Write without returning previous data

  bool writeAhcFeConfigurationData(CrcLocation::CrcComponent c, const AhcFeConfigurationData &w) {
    AhcFeConfigurationData r;
    return writeAhcFeConfigurationData(c,w,r);
  }

  bool readAhcFeConfigurationData(CrcLocation::CrcComponent c, AhcFeConfigurationData &w) {

    // DOES NOT RETURN THE SERIAL REGISTER AND DAC INFO!!! SET TO DEFAULT
    AhcFeSrConfigurationData def;
    AhcFeSrConfigurationData *p(w.srData((CrcFeConfigurationData::Connector)0,0));
    for(unsigned i(0);i<12;i++) p[i]=def;

    return readFeConfigurationData(c,w);
  }

  bool writeFeFakeEventData(CrcLocation::CrcComponent c, const CrcFeFakeEventData &d) {
    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
    if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
    if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
    if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
    if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
    if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
    if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
    if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
    if(t==CrcSerialHeader::null) return false;
    
    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataOut);

    const unsigned *p(d.data());
    for(unsigned i(0);i<d.numberOfWords();i++) {
      a.address(i+512);
      if(!serialWrite(&a)) return false;
      s.data(p[i]);
      if(!serialWrite(&s)) return false;
    }

    a.address(511);
    if(!serialWrite(&a)) return false;
    s.data(d.control());
    if(!serialWrite(&s)) return false;

    return true;
  }

  bool readFeFakeEventData(CrcLocation::CrcComponent c, CrcFeFakeEventData &d) {
    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
    if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
    if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
    if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
    if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
    if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
    if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
    if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
    if(t==CrcSerialHeader::null) return false;
    
    CrcSerialCommandFeAddress a(t);
    CrcSerialCommandWord s(t,CrcSerialHeader::feDataIn);

    a.address(511);
    if(!serialWrite(&a)) return false;
    if(!serialRead(&s)) return false;
    d.control(s.data());

    // ENABLE prevents readback!
    if(d.enable()) {
      CrcSerialCommandWord w(t,CrcSerialHeader::feDataOut);
      w.data(0);
      if(!serialWrite(&w)) return false;
    }

    unsigned *p(d.data());
    for(unsigned i(0);i<d.numberOfWords();i++) {
      a.address(i+512);
      if(!serialWrite(&a)) return false;
      if(!serialRead(&s)) return false;
      p[i]=s.data();
    }

    if(d.enable()) {
      CrcSerialCommandWord w(t,CrcSerialHeader::feDataOut);
      w.data(d.control());
      if(!serialWrite(&w)) return false;
    }

    return true;
  }


  /*
  bool enableFeFakeEvent(CrcLocation::CrcComponent c, bool e=true) {
    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
    if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
    if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
    if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
    if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
    if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
    if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
    if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
    if(t==CrcSerialHeader::null) return false;
    
    CrcSerialCommandFeAddress a(t);
    a.address(511);
    if(!serialWrite(&a)) return false;

    CrcSerialCommandWord r(t,CrcSerialHeader::feDataIn);
    if(!serialRead(&r)) return false;

    CrcSerialCommandWord w(t,CrcSerialHeader::feDataOut);
    if(e) w.data(0x80000000 | r.data());
    else  w.data(0x7fffffff & r.data());
    return serialWrite(&w);
  }
  */

  bool readFeEventData(CrcLocation::CrcComponent c, CrcFeEventData &d) {
    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
    if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
    if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
    if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
    if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
    if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
    if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
    if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
    if(t==CrcSerialHeader::null) return false;

    CrcSerialCommandWord s(t,CrcSerialHeader::feTriggerCounter);
    if(!serialRead(&s)) return false;
    d.triggerCounter(s.data());

    s.designator(CrcSerialHeader::feSpyRegister);
    if(!serialRead(&s)) return false;
    d.spyRegister(s.data());

    return true;
  }


  // ***** Slow controls *****

  bool writeI2C(CrcLocation::CrcComponent c, unsigned char a, unsigned char d) {
    if(c==CrcLocation::vmeLm82   ) return writeVmeI2C("LM82",a,d);
    if(c==CrcLocation::vmeAdm1025) return writeVmeI2C("ADM1025A",a,d);

    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::be) t=CrcSerialHeader::be;
    CrcSerialHeader::Designator cd(CrcSerialHeader::beLm82Control);
    CrcSerialHeader::Designator sd(CrcSerialHeader::beLm82Status);

    if(c!=CrcLocation::be) {
      if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
      if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
      if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
      if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
      if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
      if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
      if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
      if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
      cd=CrcSerialHeader::feLm82Control;
      sd=CrcSerialHeader::feLm82Status;
    }

    CrcSerialCommandI2CControl control(t,cd);
    control.set(a,d,false);
    serialWrite(&control);

    CrcSerialCommandShort status(t,sd);
    status.data(0x100);
    while((status.data()&0x100)!=0) {
      if(!serialRead(&status)) return false;
    }
    if((status.data()&0x300)!=0) return false;

    return true;
  }

  bool readI2C(CrcLocation::CrcComponent c, unsigned char a, unsigned char &d) {
    if(c==CrcLocation::vmeLm82   ) return readVmeI2C("LM82",a,d);
    if(c==CrcLocation::vmeAdm1025) return readVmeI2C("ADM1025A",a,d);

    CrcSerialHeader::Target t(CrcSerialHeader::null);
    if(c==CrcLocation::be) t=CrcSerialHeader::be;
    CrcSerialHeader::Designator cd(CrcSerialHeader::beLm82Control);
    CrcSerialHeader::Designator sd(CrcSerialHeader::beLm82Status);

    if(c!=CrcLocation::be) {
      if(c==CrcLocation::fe0) t=CrcSerialHeader::fe0;
      if(c==CrcLocation::fe1) t=CrcSerialHeader::fe1;
      if(c==CrcLocation::fe2) t=CrcSerialHeader::fe2;
      if(c==CrcLocation::fe3) t=CrcSerialHeader::fe3;
      if(c==CrcLocation::fe4) t=CrcSerialHeader::fe4;
      if(c==CrcLocation::fe5) t=CrcSerialHeader::fe5;
      if(c==CrcLocation::fe6) t=CrcSerialHeader::fe6;
      if(c==CrcLocation::fe7) t=CrcSerialHeader::fe7;
      cd=CrcSerialHeader::feLm82Control;
      sd=CrcSerialHeader::feLm82Status;
    }

    if(t==CrcSerialHeader::null) return false;

    CrcSerialCommandI2CControl control(t,cd);
    control.set(a,0,true);
    if(!serialWrite(&control)) return false;

    CrcSerialCommandShort status(t,sd);
    status.data(0x100);
    while((status.data()&0x100)!=0) {
      if(!serialRead(&status)) return false;
    }
    if((status.data()&0x300)!=0) return false;

    d=status.data()&0xff;
    return true;
  }

  bool writeVmeI2C(std::string dev, unsigned char add, unsigned char dat) {
    long unsigned value;

    write(dev+"Control",(add<<8)+dat);
    pollItem(dev+"BusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);

    return value==0;
  }

  bool readVmeI2C(std::string dev, unsigned char add, unsigned char &dat) {
    long unsigned value;

    write(dev+"Control",0x10000+(add<<8));
    pollItem(dev+"BusyError",0x1,1000,&value,HAL_POLL_UNTIL_DIFFERENT);
    if(value!=0) return false;

    read(dev+"Status",&value);
    dat=value&0xff;
    return true;
  }

  bool readLm82Startup(CrcLocation::CrcComponent c, CrcLm82StartupData &d) {
    unsigned char val;
    if(!readI2C(c,0xfe,val)) return false;
    d.manufacturerId(val);
    if(!readI2C(c,0xff,val)) return false;
    d.steppingCode(val);
    return true;
  }

  bool writeLm82SlowControls(CrcLocation::CrcComponent c, const CrcLm82SlowControlsData &d) {
    if(!writeI2C(c,0x09,0x28)) return false; // Set these bits first
    if(!writeI2C(c,0x0b,d.localLimit())) return false;
    if(!writeI2C(c,0x0d,d.remoteLimit())) return false;
    if(!writeI2C(c,0x5a,d.criticalLimit())) return false;

    if(!writeI2C(c,0x09,d.configuration())) return false;
    return true;
  }

  bool readLm82SlowControls(CrcLocation::CrcComponent c, CrcLm82SlowControlsData &d) {
    unsigned char val;
    if(!readI2C(c,0x03,val)) return false;
    d.configuration(val);
    if(!readI2C(c,0x42,val)) return false;
    d.criticalLimit(val);
    if(!readI2C(c,0x05,val)) return false;
    d.localLimit(val);
    if(!readI2C(c,0x07,val)) return false;
    d.remoteLimit(val);
    return true;
  }

  bool readLm82SlowReadout(CrcLocation::CrcComponent c, CrcLm82SlowReadoutData &d) {
    unsigned char val;
    if(!readI2C(c,0x02,val)) return false;
    d.status(val);
    if(!readI2C(c,0x00,val)) return false;
    d.localTemperature(val);
    if(!readI2C(c,0x01,val)) return false;
    d.remoteTemperature(val);
    return true;
  }

  bool readAdm1025Startup(CrcAdm1025StartupData &d) {
    unsigned char val;
    if(!readI2C(CrcLocation::vmeAdm1025,0x3e,val)) return false;
    d.manufacturerId(val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x3f,val)) return false;
    d.steppingCode(val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x47,val)) return false;
    d.vid(val&0x0f);
    if(!readI2C(CrcLocation::vmeAdm1025,0x49,val)) return false;
    d.vid(d.vid()|((val&0x01)<<4));
    return true;
  }

  bool writeAdm1025SlowControls(const CrcAdm1025SlowControlsData &d) {
    if(!writeI2C(CrcLocation::vmeAdm1025,0x40,0x00)) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x39,d.localHighLimit())) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x3a,d.localLowLimit())) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x37,d.remoteHighLimit())) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x38,d.remoteLowLimit())) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x47,d.vid())) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x15,d.test())) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x1f,d.offset())) return false;

    CrcAdm1025Voltages v;
    
    v=d.voltageHighLimits();
    if(!writeI2C(CrcLocation::vmeAdm1025,0x2b,v.voltage(CrcAdm1025Voltages::v25))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x2d,v.voltage(CrcAdm1025Voltages::vccp))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x2f,v.voltage(CrcAdm1025Voltages::v33))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x31,v.voltage(CrcAdm1025Voltages::v50))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x33,v.voltage(CrcAdm1025Voltages::v120))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x35,v.voltage(CrcAdm1025Voltages::vcc))) return false;

    v=d.voltageLowLimits();
    if(!writeI2C(CrcLocation::vmeAdm1025,0x2c,v.voltage(CrcAdm1025Voltages::v25))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x2e,v.voltage(CrcAdm1025Voltages::vccp))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x30,v.voltage(CrcAdm1025Voltages::v33))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x32,v.voltage(CrcAdm1025Voltages::v50))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x34,v.voltage(CrcAdm1025Voltages::v120))) return false;
    if(!writeI2C(CrcLocation::vmeAdm1025,0x36,v.voltage(CrcAdm1025Voltages::vcc))) return false;

    // Finally, write configuration including setting monitor bit
    if(!writeI2C(CrcLocation::vmeAdm1025,0x40,d.configuration()|0x01)) return false;
    return true;
  }

  bool readAdm1025SlowControls(CrcAdm1025SlowControlsData &d) {
    unsigned char val;
    if(!readI2C(CrcLocation::vmeAdm1025,0x40,val)) return false;
    d.configuration(val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x39,val)) return false;
    d.localHighLimit(val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x3a,val)) return false;
    d.localLowLimit(val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x37,val)) return false;
    d.remoteHighLimit(val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x38,val)) return false;
    d.remoteLowLimit(val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x47,val)) return false;
    d.vid(val&0xc0);
    if(!readI2C(CrcLocation::vmeAdm1025,0x15,val)) return false;
    d.test(val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x1f,val)) return false;
    d.offset(val);

    CrcAdm1025Voltages v;

    if(!readI2C(CrcLocation::vmeAdm1025,0x2b,val)) return false;
    v.voltage(CrcAdm1025Voltages::v25,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x2d,val)) return false;
    v.voltage(CrcAdm1025Voltages::vccp,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x2f,val)) return false;
    v.voltage(CrcAdm1025Voltages::v33,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x31,val)) return false;
    v.voltage(CrcAdm1025Voltages::v50,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x33,val)) return false;
    v.voltage(CrcAdm1025Voltages::v120,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x35,val)) return false;
    v.voltage(CrcAdm1025Voltages::vcc,val);
    d.voltageHighLimits(v);

    if(!readI2C(CrcLocation::vmeAdm1025,0x2c,val)) return false;
    v.voltage(CrcAdm1025Voltages::v25,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x2e,val)) return false;
    v.voltage(CrcAdm1025Voltages::vccp,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x30,val)) return false;
    v.voltage(CrcAdm1025Voltages::v33,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x32,val)) return false;
    v.voltage(CrcAdm1025Voltages::v50,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x34,val)) return false;
    v.voltage(CrcAdm1025Voltages::v120,val);
    if(!readI2C(CrcLocation::vmeAdm1025,0x36,val)) return false;
    v.voltage(CrcAdm1025Voltages::vcc,val);
    d.voltageLowLimits(v);

    return true;
  }

  bool readAdm1025SlowReadout(CrcAdm1025SlowReadoutData &d) {
    long unsigned value;

    write("ADM1025AControl",0x14100);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    d.status1(value&0xff);

    write("ADM1025AControl",0x14200);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    d.status2(value&0xff);

    write("ADM1025AControl",0x12700);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    d.localTemperature(value&0xff);

    write("ADM1025AControl",0x12600);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    d.remoteTemperature(value&0xff);
    

    CrcAdm1025Voltages v;
    
    write("ADM1025AControl",0x12000);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    v.voltage(CrcAdm1025Voltages::v25,value&0xff);

    write("ADM1025AControl",0x12100);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    v.voltage(CrcAdm1025Voltages::vccp,value&0xff);

    write("ADM1025AControl",0x12200);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    v.voltage(CrcAdm1025Voltages::v33,value&0xff);

    write("ADM1025AControl",0x12300);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    v.voltage(CrcAdm1025Voltages::v50,value&0xff);

    write("ADM1025AControl",0x12400);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    v.voltage(CrcAdm1025Voltages::v120,value&0xff);

    write("ADM1025AControl",0x12500);
    do read("ADM1025AStatus",&value);
    while((value&0x300)==0x100);
    if((value&0x300)!=0) return false;
    v.voltage(CrcAdm1025Voltages::vcc,value&0xff);

    d.voltages(v);

    return true;
  }

  void print(std::ostream &o) {
    long unsigned value;
    o << "CrcVmeDevice::print()" << std::endl;

    read("FirmwareId",&value);
    o << hex << "  FirmwareId = 0x" << value << dec << std::endl;
    read("ClockSelect",&value);
    o << hex<< "  ClockSelect = 0x" << value << dec << std::endl;
    read("VmeStatus",&value);
    o << hex<< "  VmeStatus = 0x" << value << dec << std::endl;
    read("SerialStatus",&value);
    o << hex<< "  SerialStatus = 0x" << value << dec << std::endl;
    read("TTCClockCounter",&value);
    o << hex<< "  TTCClockCounter = 0x" << value << dec << std::endl;
    read("BackplaneClockCounter",&value);
    o << hex<< "  BackplaneClockCounter = 0x" << value << dec << std::endl;

    o << std::endl;
  }

private:
  unsigned _printLevel;
  unsigned _slot;
  unsigned vlinkTotalLengthSum;
};

#endif
