#ifndef __DaqDummyBusAdapter
#define __DaqDummyBusAdapter

#include <iomanip>
#include <iostream>
#include "VMEBusAdapterInterface.hh"
#include "VMEDummyDeviceIdentifier.hh"
#include "HardwareProblemException.hh"

#include "CrcDummyDevice.hh"

/**
*
*
*     @short A dummy VMEBusAdapter in order to work without hardware.
*            
*            This  BusAdapter  merely  serves  to  let  you  test  you 
*            software  without actually  having hardware  connected to 
*            your  computer.  All actions  which  in  real life  would 
*            result in  read or write  operations will be  reported on 
*            the standard output.
*
*       @see BusAdapterInterface
*    @author Christoph Schwick
* $Revision: 1.1 $
*     $Date: 2008/06/27 10:34:05 $
*
*
**/
class DaqDummyBusAdapter : public VMEBusAdapterInterface {

public:

  /**
   * This enumerator is used in the constructor. It switches 
   * the versbose behaviour of the BusAdapter on or off. Note
   * that the enumerator is a member of the class PCIDummyBusAdapter.
   * So it must be used. This means that a Scope identifier is
   * needed when calling the constructor of the class. A valid 
   * example would be:
   * <pre>
   * PCIDummyBusAdapter myBa( PCIDummyBusAdapter::VERBOSE_ON,
   *                          PCIDummyBusAdapter::MEMORY_MAP_OFF );
   * </pre>
   * The enumerators have been put into the class in order not
   * to contaminate the global namesapce with objects merely used 
   * for debugging purposes.
   */
  enum VerboseMode { VERBOSE_OFF, VERBOSE_ON };

  /**
   * This enumerator is used in the constructor. It switches 
   * the memory-mapping behaviour of the BusAdapter on or off. 
   * If switched on every address-space  corresponding to a 
   * specific BAR or to the configuration space is mapped into
   * the local memory of the host. It is then used to perform
   * read and write operations into the mapped memory. This 
   * allows to play in a more realistic way with the device.
   * See VerboseMode comments for a note on how to use this 
   * enumerator.
   */
  enum MemoryMode  { MEMORY_MAP_OFF, MEMORY_MAP_ON };

  /**
   * The constructor takes two arguments:
   * @param verbose allows you to monitor all data transfers: If 
   *        set to true every action is reported on the standard
   *        output.
   * @param memoryMode allows you to create memory regions which
   *        fake the hardware device. For every openDevice call
   *        a memory region is created into which or from which 
   *        data is transfered. With this you can write and read
   *        back data from you dummy-device. The size of the 
   *        mapped region depend on the VMEAddressTable given
   *        to the openDevice call. 
   */
  DaqDummyBusAdapter( unsigned p = 0,
		      enum VerboseMode verbose    = VERBOSE_ON,
                      enum MemoryMode  memoryMode = MEMORY_MAP_OFF,
		      ostream& os = cout ) : os_( os ), _crc(12) {

    if(p!=0) throw std::exception();

    deviceNumberCounter = 0;
    this->verbose = verbose;
    this->memoryMode = memoryMode;
    os_  << "DaqDummyBusAdapter : constructor has been called :\n" 
	 << "                     The \"verbose\" flag is set to " << (int)verbose << "\n"
	 << "                     The \"memoryMode\" flag is set to " << (int)memoryMode  
	 << endl;
  }

  /**
   * There is nothing to do in the desctructor.
   */
  virtual ~DaqDummyBusAdapter() {
    os_  << "DaqDummyBusAdapter : destructor called" << endl;
  }

  /**
   * If the VMEDummyBusAdapter was created with the memoryMode
   * option set to true, the call scans the VMEAddressTable in
   * order to find out how much memory has to be reserved for 
   * the dummy Hardware device. The pointer to this memory is 
   * then contained in the deviceIdentifier. 
   */
  void openDevice(const VMEAddressTable& VMEAddressTable,
                  unsigned long vmeBaseaddress,
                  DeviceIdentifier** deviceIdentifierPtr,
                  unsigned long* baseAddressPtr)
    throw();
  
  void openDevice(const VMEAddressTable& vmeAddressTable,
		  vector<unsigned long>& vmeBaseaddresses,
		  DeviceIdentifier** deviceIdentifierPtr,
		  vector<unsigned long>* baseAddressPtr) 
    throw();
  /**
   * Here the memory regions which might have beeen reserved
   * in the openDevice call, are deleted.
   */
  void closeDevice( DeviceIdentifier* deviceIdentifier )
    throw();
  
  void write( DeviceIdentifier* deviceIdentifier,
              unsigned long address,
              unsigned long addressModifier,
              unsigned long dataWidth,
              unsigned long data)
    throw();
  
  void read( DeviceIdentifier* DeviceIdentifier,
             unsigned long address,
             unsigned long addressModifier,
             unsigned long dataWidth,
             unsigned long* result)
    throw();

  void writeBlock( DeviceIdentifier* deviceIdentifierPtr,
                   unsigned long startAddress,
                   unsigned long length,      // in bytes
                   unsigned long addressModifier,
                   unsigned long dataWidth,
                   char *buffer,
                   HalAddressIncrement addressBehaviour = HAL_DO_INCREMENT )
    throw();
  
  void readBlock( DeviceIdentifier* deviceIdentifierPtr,
                  unsigned long startAddress,
                  unsigned long length,      // in bytes
                  unsigned long addressModifier,
                  unsigned long dataWidth,
                  char *buffer,
                  HalAddressIncrement addressBehaviour = HAL_DO_INCREMENT )
    throw();

  void resetBus( )
    throw();
  
private:
  ostream& os_;
  unsigned long deviceNumberCounter;
  enum VerboseMode verbose;
  enum MemoryMode  memoryMode;

  CrcDummyDevice _crc;
};


void
DaqDummyBusAdapter::openDevice(const VMEAddressTable& vmeAddressTable,
                               unsigned long vmeBaseaddress,
                               DeviceIdentifier** deviceIdentifierPtr,
                               unsigned long* baseAddressPtr) 
  throw() {
  vector<char *> memoryRegions(NUMBER_OF_VME64XFUNCTIONS);
  for ( int i=0; i<NUMBER_OF_VME64XFUNCTIONS; i++ ) memoryRegions.push_back( (char*) 0 );

  unsigned long ic;
  os_  << "DaqDummyBusAdapter : opening Device number "<< deviceNumberCounter 
       << " with baseAddress : " 
       << hex << setw(8) << setfill('0') << vmeBaseaddress
       << "\n                     memory-mapping-mode : "
       << (int)this->memoryMode 
       << endl;

  if(vmeBaseaddress!=0x000c0000) throw std::exception();


  vector<unsigned long> minAddresses; 
  vector<unsigned long> maxAddresses;

  if ( memoryMode == MEMORY_MAP_ON ) {
    vmeAddressTable.getAddressBoundaries( minAddresses, maxAddresses );
    memoryRegions[0] = new char[ maxAddresses[0] + 1 ];
    for (ic=0; ic<=maxAddresses[0]; ic++) memoryRegions[0][ic] = (char)0x00;
    *baseAddressPtr = (unsigned long)memoryRegions[0];

    os_  << "                     mapped the address space to memory address " 
	 << hex << ((unsigned long)memoryRegions[0]) 
	 << "\n                     reserved 0x" << hex << maxAddresses[0] 
         << " (dec: " << dec << maxAddresses[0] << " )" 
         << " bytes for memory mapped operation" << endl; 
  } else {
    *baseAddressPtr = vmeBaseaddress;
  }

  *deviceIdentifierPtr = new VMEDummyDeviceIdentifier( deviceNumberCounter,
                                                       memoryRegions );
  deviceNumberCounter++;
}

// the call for VME64x modules
void
DaqDummyBusAdapter::openDevice(const VMEAddressTable& vmeAddressTable,
                               vector<unsigned long>& vmeBaseaddresses,
                               DeviceIdentifier** deviceIdentifierPtr,
                               vector<unsigned long>* baseAddressPtr) 
  throw() {
  vector<char *> memoryRegions(NUMBER_OF_VME64XFUNCTIONS);
  for ( int i=0; i<NUMBER_OF_VME64XFUNCTIONS; i++ ) memoryRegions.push_back( (char*) 0 );

  unsigned long ic;
  os_  << "DaqDummyBusAdapter : opening VME64x Device number "<< deviceNumberCounter 
       << "\n                     memory-mapping-mode : "
       << (int)this->memoryMode 
       << endl;




  vector<unsigned long> minAddresses; 
  vector<unsigned long> maxAddresses;

  if ( memoryMode == MEMORY_MAP_ON ) {
    vmeAddressTable.getAddressBoundaries( minAddresses, maxAddresses );
    for ( int ifunc=0; ifunc<NUMBER_OF_VME64XFUNCTIONS; ifunc++ ) {
      if ( maxAddresses[ifunc] > 0 ) {
	memoryRegions[ifunc] = new char[ maxAddresses[ifunc] ];
	for (ic=0; ic<maxAddresses[ifunc]; ic++) memoryRegions[ifunc][ic] = (char)0x00;
	(*baseAddressPtr)[ifunc] = (unsigned long)memoryRegions[ifunc];
	
	os_  << "                     mapped the address space of function" << ifunc 
	     << " to memory address " << hex << ((unsigned long)memoryRegions[ifunc]) 
	     << "\n                     reserved 0x" << hex << maxAddresses[ifunc] 
	     << " (dec: " << dec << maxAddresses[ifunc] << " )" 
	     << " bytes for memory mapped operation" << endl;
      } 
    }
  } else {
    for ( int ifunc=0; ifunc<NUMBER_OF_VME64XFUNCTIONS; ifunc++ ) {
      (*baseAddressPtr)[ifunc] = vmeBaseaddresses[ifunc];
    }
  }

  *deviceIdentifierPtr = new VMEDummyDeviceIdentifier( deviceNumberCounter,
                                                       memoryRegions );
  deviceNumberCounter++;
}

void
DaqDummyBusAdapter::closeDevice( DeviceIdentifier* vmeDevice ) 
  throw() {
    os_  << "DaqDummyBusAdapter : closing Device  \n" 
         << "                     " << vmeDevice->printString() << endl;
    delete( vmeDevice );
}

void
DaqDummyBusAdapter::read( DeviceIdentifier* vmeDevice, 
                          unsigned long address,
                          unsigned long addressModifier,
                          unsigned long dataWidth,
                          unsigned long *resultPtr ) 
  throw() {

  *resultPtr = 0;
  if ( memoryMode == MEMORY_MAP_ON ) {
    char *memoryAddress = (char*) address;
    memcpy( resultPtr, memoryAddress, dataWidth);
  }


  unsigned result(0);
  if(!_crc.read((unsigned)address,result)) 
    throw HardwareProblemException("UNKNOWN HARDWARE LOCATION");
  *resultPtr=result;


  if ( verbose == VERBOSE_ON ) {
    os_  << endl;
    os_  << "DaqDummyBusAdapter : read from Device number " << vmeDevice->printString() << endl;
    os_  << "                     address : " << hex << setw(8) << setfill('0') << address << endl;
    os_  << "                          AM :       " << hex << setw(2) << setfill('0') << addressModifier << endl;
    os_  << "                   dataWidth : " << dec << setw(8) << setfill(' ') << dataWidth << endl;
    os_  << "                   returning : 0x" << hex << setw(8) << setfill('0') << *resultPtr 
         << " (dec) " << dec << *resultPtr << endl;
    os_  << endl;
  }
}

void
DaqDummyBusAdapter::write( DeviceIdentifier* vmeDevice, 
                           unsigned long address, 
                           unsigned long addressModifier,
                           unsigned long dataWidth,
                           unsigned long data) 
  throw() {

  if ( memoryMode == MEMORY_MAP_ON ) {
    char *memoryAddress = (char*) address;
    memcpy( memoryAddress, &data, dataWidth);
  }

  if(!_crc.write((unsigned)address,data)) 
    throw HardwareProblemException("UNKNOWN HARDWARE LOCATION");


  if ( verbose == VERBOSE_ON ) {
    os_  << endl;
    os_  << "DaqDummyBusAdapter : write to Device number " << vmeDevice->printString() << endl;
    os_  << "                     address : " << hex << setw(8) << setfill('0') << address << endl;
    os_  << "                          AM :       " << hex << setw(2) << setfill(' ') << addressModifier << endl;
    os_  << "                   dataWidth : " << dec << setw(8) << setfill(' ') << dataWidth << endl;
    os_  << "                   dataValue : 0x" << hex << setw(8) << setfill('0') << data 
         << " (dec) " << dec << data << endl;
    os_  << endl;
  }
}


void
DaqDummyBusAdapter::resetBus( ) 
  throw() {
  if ( verbose == VERBOSE_ON ) {
    os_  << endl;
    os_  << "DaqDummyBusAdapter : resetting bus" << endl;
    os_  << endl;
  }
}

void
DaqDummyBusAdapter::readBlock( DeviceIdentifier *vmeDevice,
                               unsigned long startAddress,
                               unsigned long length,      // in bytes
                               unsigned long addressModifier,
                               unsigned long dataWidth,
                               char *buffer,
                               HalAddressIncrement addressBehaviour ) 
  throw() {

  if ( memoryMode == MEMORY_MAP_ON ) {
    char *memoryAddress = (char*)startAddress;
    if ( addressBehaviour == HAL_DO_INCREMENT ) {
      memcpy( buffer, memoryAddress, length);
    } else {
      os_  << "DaqDummyBusAdapter::readBlock : No Fifo functionality in DaqDummyBusAdapter. Doing nothing!" << endl;
    }
  }

  if( verbose == VERBOSE_ON ) {
    os_  << endl;
    os_  << "DaqDummyBusAdapter : readBlock from Device \n                     " << vmeDevice->printString() << endl;
    os_  << "                startaddress (hex) : " << hex << setw(8) << setfill('0') << startAddress << endl;
    os_  << "                      length (hex) : " << hex << setw(8) << setfill('0') << length << endl;
    os_  << "                          AM (hex) :       " << hex << setw(2) << setfill('0') << addressModifier << endl;
    os_  << "                         dataWidth : " << dec << setw(8) << setfill(' ') << dataWidth << endl;
    os_  << "                  addressBehaviour : ";
    if ( addressBehaviour == HAL_DO_INCREMENT ) {
      os_  << "HAL_DO_INCREMENT" << endl;
    } else if ( addressBehaviour == HAL_NO_INCREMENT ) {
      os_  << "HAL_NO_INCREMENT" << endl;
    } else { // should never happen
      os_  << "XXXXXXXXXXXXXXXX" << endl;
    }
    os_  << "       pointer to readbuffer (hex) : " << hex << setw(8) << setfill('0') << (unsigned long)buffer << endl;
    os_  << endl;
  }
}
  
void
DaqDummyBusAdapter::writeBlock( DeviceIdentifier* vmeDevice,
                                unsigned long startAddress,
                                unsigned long length,      // in bytes
                                unsigned long addressModifier,
                                unsigned long dataWidth,
                                char *buffer,
                                HalAddressIncrement addressBehaviour ) 
  throw() {
  

  if ( memoryMode == MEMORY_MAP_ON ) {
    char *memoryAddress = (char*) startAddress;
    if ( addressBehaviour == HAL_DO_INCREMENT ) {
      memcpy( memoryAddress, buffer, length);
     } else {
       os_  << "DaqDummyBusAdapter::writeBlock : No Fifo functionality in DaqDummyBusAdapter. Doing nothing!" << endl;
     }
  }

  if ( verbose == VERBOSE_ON ) {
    os_  << endl;
    os_  << "DaqDummyBusAdapter : writeBlock to Device \n                     " << vmeDevice->printString() << endl;
    os_  << "                startaddress (hex) : " << hex << setw(8) << setfill('0') << startAddress << endl;
    os_  << "                      length (hex) : " << hex << setw(8) << setfill('0') << length << endl;
    os_  << "                          AM (hex) :       " << hex << setw(2) << setfill('0') << addressModifier << endl;
    os_  << "                         dataWidth : " << dec << setw(8) << setfill(' ') << dataWidth << endl;
    os_  << "                  addressBehaviour : ";
    if ( addressBehaviour == HAL_DO_INCREMENT ) {
      os_  << "HAL_DO_INCREMENT" << endl;
    } else if ( addressBehaviour == HAL_NO_INCREMENT ) {
      os_  << "HAL_NO_INCREMENT" << endl;
    } else { // should never happen
      os_  << "XXXXXXXXXXXXXXXX" << endl;
    }
    os_  << "     pointer to sourcebuffer (hex) : " << hex << setw(8) << setfill('0') << (unsigned long)buffer << endl;
    os_  << endl;
  }
}

/*
class DaqDummyBusAdapter : public VMEDummyBusAdapter {

public:
  DaqDummyBusAdapter(unsigned p) :
    VMEDummyBusAdapter(p,VMEDummyBusAdapter::VERBOSE_ON,VMEDummyBusAdapter::MEMORY_MAP_ON,std::cout) {
    if(p!=0) throw std::exception();
  }
};
*/


#endif /* __DaqDummyBusAdapter */
