//
// $Id: DhcReadout.hh,v 1.1 2008/06/27 10:34:05 meyern Exp $
//

#ifndef DhcReadout_HH
#define DhcReadout_HH

#include <iostream>
#include <fstream>

#include "VMEAddressTable.hh"
#include "VMEAddressTableASCIIReader.hh"
#include "DaqBusAdapter.hh"
#include "HardwareAccessException.hh"

#include "RcdHeader.hh"
#include "RcdUserRW.hh"

#include "SubAccessor.hh"
#include "SubInserter.hh"

#include "DhcVmeDevice.hh"
#include "DhcReadoutConfigurationData.hh"


class DhcReadout : public RcdUserRW {

public:
  DhcReadout(unsigned b, unsigned char c) : 
    _interface(b), _location(c,0,0,0), _crateNumber(c), _busAdapter(0),
    _addressTableReader("online/hal/DCOL_20070614.hal"),
    _addressTable("DCOL VME Address Table",_addressTableReader) {

    for(unsigned i(0);i<=21;i++) _device[i]=0;

    // Catch exceptions locally here in case PCI card not installed
    try {
      _busAdapter=new DaqBusAdapter(HAL::CAENLinuxBusAdapter::V2718, b);
    } catch ( std::exception e ) {
      _busAdapter=0;
    }

    // Set crate number of readout configuration
    _config.crateNumber(c);

  }

  virtual ~DhcReadout() {

    if(_busAdapter!=0) delete _busAdapter;
  }

  virtual bool record(RcdRecord &r) {
    if(doPrint(r.recordType())) {
      std::cout << "DhcReadout::record()" << std::endl;
      r.RcdHeader::print(std::cout," ") << std::endl;
    }
 
    // Catch exceptions from HAL code
    try {

      // Check record type
      switch (r.recordType()) {

      case RcdHeader::startUp: {
	if(_busAdapter!=0) {
	  for(unsigned s(2);s<=21;s++) {
	    _device[s]=new DhcVmeDevice(_addressTable,*_busAdapter,s);
	    if(_device[s]->alive()) {
	      std::cout << "DhcReadout::ctor()  PCI card " << _interface
			<< " Unit " << s
			<< " crate 0x" << std::hex << (unsigned)_crateNumber
			<< std::dec
			<< " slot " << std::setw(2)
			<< s << " found alive" << std::endl;
	    } else {
	      delete _device[s];
	      _device[s]=0;
	    }
	  }
	}
	break;
      }

      case RcdHeader::runStart: {
	// Access the DaqRunStart to get print level 
	SubAccessor accessor(r);
	std::vector<const DaqRunStart*> v(accessor.extract<DaqRunStart>());
	if(v.size()==1)  _printLevel=v[0]->runType().printLevel();

	assert(reset());
	assert(readRunData(r));
	assert(readSlowRunData(r));
	break;
      }

      case RcdHeader::runEnd:
      case RcdHeader::runStop: {
	assert(readRunData(r));
	assert(readSlowRunData(r));
	break;
      }

	// Configuration start is used to set up system
      case RcdHeader::configurationStart: {
	assert(writeSlowConfigurationData(r));
	assert(readSlowConfigurationData(r));

	assert(writeConfigurationData(r));
	assert(readConfigurationData(r));

	break;
      }

      case RcdHeader::acquisitionStart: {
	_nEvents=0;

	// Flush any data out
	unsigned char *array(new unsigned char[16*1024*1024]);

	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0) {

	    // Really check for no data
	    assert(_device[i]->readEventData(*((DhcEventData*)array)));
	  }
	}

	delete [] array;

	break;
      }

      case RcdHeader::acquisitionEnd:
      case RcdHeader::acquisitionStop: {
	break;
      }

      case RcdHeader::spillStart: {
	break;
      }

      case RcdHeader::spillEnd: {
	break;
      }
	
      case RcdHeader::spill: {
	_nEvents=0;
	break;
      }

      case RcdHeader::transferStart: {
	break;
      }

      case RcdHeader::transferEnd: {
	break;
      }
      
      case RcdHeader::transfer: {
	while(readEventData(r));
	break;
      }

      case RcdHeader::trigger: {
	_nEvents++;
	break;
      }

      case RcdHeader::event: {
	assert(readEventData(r));
	break;
      }

      case RcdHeader::shutDown: {
	for(unsigned i(2);i<=21;i++) {
	  if(_device[i]!=0) delete _device[i];
	}
	break;
      }

      case RcdHeader::slowReadout: {
	assert(readSlowReadoutData(r));
	break;
      }

      default: {
	break;
      }
      };

      return true;

    } catch ( HAL::HardwareAccessException& e ) {
      std::cout << "*** Exception occurred : " << e.what() << std::endl;
      std::cerr << "*** Exception occurred : " << e.what() << std::endl;
      
    } catch ( std::exception e ) {
        std::cout << "*** Unknown exception occurred" << std::endl;
      std::cerr << "*** Unknown exception occurred" << std::endl;
    }
  
    return false;
  }

  bool reset() {
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0) {
	assert(_device[i]->reset());
      }
    }
    return true;
  }

  bool readEventData(RcdRecord &r) {
    SubInserter inserter(r);

    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0 && _config.slotEnable(i)) {

	_location.slotNumber(_device[i]->slot());
	_location.label(0);
	    
	// Make the object in the record
	DhcLocationData<DhcEventData>
	  *p(inserter.insert< DhcLocationData<DhcEventData> >());
	p->location(_location);

	// Do the read
	assert(_device[i]->readEventData(*(p->data())));
	inserter.extend(p->data()->numberOfWords()*4);
	if(doPrint(r.recordType(),1)) p->print(std::cout) << std::endl;
	if(doPrint(r.recordType(),2)) p->data()->print(std::cout,"",true);

	// Check data structure
	if(!p->data()->verify(true)) {
	  p->print(std::cout,"NOT VERIFIED  ") << std::endl;
	}
      }
    }

    return true;
  }

  bool writeConfigurationData(RcdRecord &r) {
    SubAccessor accessor(r);

    std::vector<const DhcReadoutConfigurationData*>
      bc(accessor.access< DhcReadoutConfigurationData>());
    
    if(bc.size()>0) {
      for(unsigned j(0);j<bc.size();j++) {
	if(bc[j]->crateNumber()==_crateNumber) {
    	  _config=*(bc[j]);
	  if(doPrint(r.recordType(),1)) _config.print(std::cout) << std::endl;
	}
      }
    }

    // Handle BE
    std::vector<const DhcLocationData<DhcBeConfigurationData>*>
      be(accessor.access< DhcLocationData<DhcBeConfigurationData> >());

    // Check for consistency and store values for this crate
    std::vector<const DhcLocationData<DhcBeConfigurationData>*> bCrate;

    for(unsigned j(0);j<be.size();j++) {
      assert(be[j]->slotBroadcast() || be[j]->slotNumber()<=21);
      assert(be[j]->dhcComponent()==DhcLocation::be);

      if(be[j]->crateNumber()==_location.crateNumber())
	bCrate.push_back(be[j]);
    }

    if(doPrint(r.recordType(),1)) {
      std::cout << " Number of DhcBeConfigurationData write subrecords accessed for crate = "
		<< bCrate.size() << " (of " << be.size() << " total)" << std::endl << std::endl;
      for(unsigned i(0);i<bCrate.size();i++) {
	bCrate[i]->print(std::cout,"  ") << std::endl;
      }
    }

    // Look for configuration for all slots first
    for(unsigned j(0);j<bCrate.size();j++) {
      if(bCrate[j]->slotBroadcast()) {
	  
	// Loop over all the DCOLs
	for(unsigned i(2);i<=21;i++) {
	  if(_device[i]!=0 && _config.slotEnable(i)) {

	    DhcBeConfigurationData cbcd(*bCrate[j]->data());

	    if(doPrint(r.recordType(),1)) {
	      std::cout << " Writing to slot " << std::setw(2) << i << " ";
	      cbcd.print(std::cout,"  ") << std::endl;
	    } 
	    assert(_device[i]->writeBeConfigurationData(cbcd));
	  }
	}
      }
    }

    // Now look for individual slot configurations to override the broadcast
    for(unsigned j(0);j<bCrate.size();j++) {
      if(!bCrate[j]->slotBroadcast()) {
	  
	unsigned i(bCrate[j]->slotNumber());
	if(_device[i]!=0 && _config.slotEnable(i)) {

	  DhcBeConfigurationData cbcd(*bCrate[j]->data());

	  if(doPrint(r.recordType(),1)) {
	    std::cout << " Writing to slot " << std::setw(2) << i << " ";
	    cbcd.print(std::cout,"  ") << std::endl;
	  } 
	  assert(_device[i]->writeBeConfigurationData(cbcd));
	}
      }
    }

    // Handle FE
    std::vector<const DhcLocationData<DhcFeConfigurationData>*>
      fe(accessor.access< DhcLocationData<DhcFeConfigurationData> >());

    if(doPrint(r.recordType(),1))
      std::cout << " Number of DhcFeConfigurationData write subrecords accessed = "
		<< fe.size() << std::endl << std::endl;

    // First do slot broadcast
    for(unsigned j(0);j<fe.size();j++) {
      if(fe[j]->crateNumber()==_crateNumber && fe[j]->slotBroadcast()) {
	if(doPrint(r.recordType(),1)) fe[j]->print(std::cout) << std::endl;

	// Loop over all the DCOLs
	for(unsigned i(0);i<=21;i++) {
	  if(_device[i]!=0) {
	    if(_config.slotEnable(i)) {
	      if(fe[j]->dhcComponent()==DhcLocation::feBroadcast) {
		for(unsigned f(0);f<12;f++) {
		  if(_config.slotFeEnable(i,f)) {
		    assert(_device[i]->writeFeConfigurationData((DhcLocation::DhcComponent)f,
								*fe[j]->data()));
		  }
		}
	      } else {
		assert(_device[i]->writeFeConfigurationData(fe[j]->dhcComponent(),
							    *fe[j]->data()));
	      }
	    }
	  }
	}
      }
    }

    // Next do FE broadcast 
    for(unsigned j(0);j<fe.size();j++) {
      if(fe[j]->crateNumber()==_crateNumber && !fe[j]->slotBroadcast()) {
	assert(fe[j]->slotNumber()<=21);
	
	if(fe[j]->dhcComponent()==DhcLocation::feBroadcast) {
	  
	  if(_device[fe[j]->slotNumber()]!=0) {
	    if(doPrint(r.recordType(),1)) fe[j]->print(std::cout) << std::endl;
	    for(unsigned f(0);f<12;f++) {
	      if(_config.slotFeEnable(fe[j]->slotNumber(),f)) {
		assert(_device[fe[j]->slotNumber()]
		       ->writeFeConfigurationData((DhcLocation::DhcComponent)f,
						  *fe[j]->data()));
	      }
	    }
	  }
	}
      }
    }


    // Finally do individual FE configurations
    for(unsigned j(0);j<fe.size();j++) {
      if(fe[j]->crateNumber()==_crateNumber) {

	if(!fe[j]->slotBroadcast() && fe[j]->dhcComponent()!=DhcLocation::feBroadcast) {

	  assert(fe[j]->slotNumber()<=21);
	  if(_device[fe[j]->slotNumber()]!=0) {

	    if(_config.slotFeEnable(fe[j]->slotNumber(),fe[j]->dhcComponent())) {
	      
	    if(doPrint(r.recordType(),1)) fe[j]->print(std::cout) << std::endl;
	    assert(_device[fe[j]->slotNumber()]
		   ->writeFeConfigurationData(fe[j]->dhcComponent(),
					      *fe[j]->data()));
	    }
	  }
	}
      }
    }

    return true;
  }

  bool readConfigurationData(RcdRecord &r) {
    SubInserter inserter(r);
    
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0 && _config.slotEnable(i)) {
	DhcLocation cl(_crateNumber,_device[i]->slot(),DhcLocation::be,0);
	
	DhcLocationData<DhcBeConfigurationData>
	  *b(inserter.insert< DhcLocationData<DhcBeConfigurationData> >());
	b->location(cl);
	assert(_device[i]->readBeConfigurationData(cl.dhcComponent(),*b->data()));
	if(doPrint(r.recordType(),1)) b->print(std::cout,"  ") << std::endl;

	for(unsigned f(0);f<12;f++) {
	  if(_config.slotFeEnable(i,f)) {
	    for (unsigned c(0);c<4;c++) {
	      DhcLocationData<DhcFeConfigurationData>
		*e(inserter.insert< DhcLocationData<DhcFeConfigurationData> >());

	      cl.dhcComponent((DhcLocation::DhcComponent)f);
	      e->location(cl);
	      e->data()->chipid(c);
	      assert(_device[i]->readFeConfigurationData(cl.dhcComponent(),*e->data()));

	      if(doPrint(r.recordType(),1)) e->print(std::cout) << std::endl;
	    }
	  }
	}
      }
    }
    
    return true;
  }

  bool readRunData(RcdRecord &r) {

    return true;
  }

  bool readSlowRunData(RcdRecord &r) {

    return true;
  }

  bool readSlowReadoutData(RcdRecord &r) {

    return true;
  }

  bool writeSlowConfigurationData(RcdRecord &r) {

    return true;
  }

  bool readSlowConfigurationData(RcdRecord &r) {

    return true;
  }


protected:
  unsigned _interface;
  DhcLocation _location;
  unsigned char _crateNumber;
  DaqBusAdapter *_busAdapter;
  HAL::VMEAddressTableASCIIReader _addressTableReader;
  HAL::VMEAddressTable _addressTable;

  DhcVmeDevice *_device[22];
  DhcReadoutConfigurationData _config;
  unsigned _nEvents;

};

#endif // DhcReadout_HH
