//
// $Id: TtmReadout.hh,v 1.1 2008/06/27 10:34:05 meyern Exp $
//

#ifndef TtmtReadout_HH
#define TtmReadout_HH

#include <iostream>
#include <fstream>

#include "VMEAddressTable.hh"
#include "VMEAddressTableASCIIReader.hh"
#include "DaqBusAdapter.hh"
#include "HardwareAccessException.hh"

#include "RcdHeader.hh"
#include "RcdUserRW.hh"

#include "SubAccessor.hh"
#include "SubInserter.hh"

#include "TtmLocation.hh"
#include "TtmLocationData.hh"
#include "TtmConfigurationData.hh"
#include "TtmVmeDevice.hh"

#include "DhcReadoutConfigurationData.hh"

class TtmReadout : public RcdUserRW {

public:
  TtmReadout(unsigned b, unsigned char c, unsigned char s) : 
    _interface(b), _location(c,s), _crateNumber(c), _busAdapter(0),
    _addressTableReader("online/hal/TTM_20070828.hal"),
    _addressTable("TTM VME Address Table",_addressTableReader) {

    _device = 0;

    // Catch exceptions locally here in case PCI card not installed
    try {
      _busAdapter=new DaqBusAdapter(HAL::CAENLinuxBusAdapter::V2718, b);
    } catch ( std::exception e ) {
      _busAdapter=0;
    }
  }

  virtual ~TtmReadout() {

    if(_busAdapter!=0) delete _busAdapter;
  }

  virtual bool record(RcdRecord &r) {
    if(doPrint(r.recordType())) {
      std::cout << "TtmReadout::record()" << std::endl;
      r.RcdHeader::print(std::cout," ") << std::endl;
    }
 
    // Catch exceptions from HAL code
    try {

      // Check record type
      switch (r.recordType()) {

      case RcdHeader::startUp: {
	if(_busAdapter!=0) {
	  _device=new TtmVmeDevice(_addressTable,*_busAdapter,_location.slotNumber());
	  if(_device->alive()) {
	    std::cout << "TtmReadout::ctor()  PCI card " << _interface
		      << " crate 0x" << std::hex << (unsigned)_crateNumber
		      << std::dec
		      << " slot " << std::setw(2)
		      << (unsigned)_location.slotNumber() << " found alive" << std::endl;
	  } else {
	    delete _device;
	    _device=0;
	  }
	}
	break;
      }
      case RcdHeader::runStart: {
	// Access the DaqRunStart to get print level 
	SubAccessor accessor(r);
	std::vector<const DaqRunStart*> v(accessor.extract<DaqRunStart>());
	if(v.size()==1)  _printLevel=v[0]->runType().printLevel();

	assert(reset());
	assert(readRunData(r));
	break;
      }

      case RcdHeader::runEnd:
      case RcdHeader::runStop: {
	assert(reset());
	assert(readRunData(r));
	break;
      }

	// Configuration start is used to set up system
      case RcdHeader::configurationStart: {
	assert(writeConfigurationData(r));
	assert(readConfigurationData(r));

	break;
      }

      case RcdHeader::acquisitionStart: {
	_nEvents=0;
	break;
      }

      case RcdHeader::acquisitionEnd:
      case RcdHeader::acquisitionStop: {
	break;
      }

      case RcdHeader::spillStart: {
	break;
      }

      case RcdHeader::spillEnd: {
	break;
      }

      case RcdHeader::spill: {
	_nEvents=0;
	break;
      }

      case RcdHeader::transferStart: {
	break;
      }

      case RcdHeader::transferEnd: {
	break;
      }

      case RcdHeader::transfer: {
	break;
      }

      case RcdHeader::trigger: {
	if (takeTrigger(r))
	  _nEvents++;
	else
	  return false;
	break;
      }

      case RcdHeader::event: {
	break;
      }

      case RcdHeader::shutDown: {
	if(_device!=0) delete _device;
	break;
      }

      case RcdHeader::slowReadout: {
	break;
      }

      default: {
	break;
      }
      };

      return true;

    } catch ( HAL::HardwareAccessException& e ) {
      std::cout << "*** Exception occurred : " << e.what() << std::endl;
      std::cerr << "*** Exception occurred : " << e.what() << std::endl;
      
    } catch ( std::exception e ) {
      std::cout << "*** Unknown exception occurred" << std::endl;
      std::cerr << "*** Unknown exception occurred" << std::endl;
    }
    return false;
  }

  bool reset() {
    if(_device!=0) {
	assert(_device->reset());
    }
    return true;
  }

  bool writeConfigurationData(RcdRecord &r) {
    SubAccessor accessor(r);

    std::vector<const DhcReadoutConfigurationData*>
      bc(accessor.access< DhcReadoutConfigurationData>());
    
    if(bc.size()>0) {
      for(unsigned j(0);j<bc.size();j++) {
	if(bc[j]->crateNumber()==_crateNumber) {
    	  _config=*(bc[j]);
	  if(doPrint(r.recordType(),1)) _config.print(std::cout) << std::endl;
	}
      }
    }

    _delay.tv_sec = _config.pollInterval().seconds();
    _delay.tv_nsec = 1000*_config.pollInterval().microseconds();

    std::vector<const TtmLocationData<TtmConfigurationData>*>
      tt(accessor.access< TtmLocationData<TtmConfigurationData> >());

    if(doPrint(r.recordType(),1))
      std::cout << " Number of TtmConfigurationData write subrecords accessed = "
		<< tt.size() << std::endl << std::endl;

    for(unsigned j(0);j<tt.size();j++) {
      if(tt[j]->crateNumber()==_crateNumber) {

	assert(tt[j]->slotNumber()<=21);
	if(doPrint(r.recordType(),1)) tt[j]->print(std::cout) << std::endl;
	if(_device!=0) {
	  assert(_device->writeConfigurationData(*tt[j]->data()));
	}
      }
    }

    return true;
  }

  bool readConfigurationData(RcdRecord &r) {
    SubInserter inserter(r);

    TtmLocationData<TtmConfigurationData>
      *tt(inserter.insert< TtmLocationData<TtmConfigurationData> >());
    tt->location(_location);

    if(_device!=0) {
      assert(_device->readConfigurationData(*tt->data()));
      if(doPrint(r.recordType(),1)) tt->print(std::cout) << std::endl;
    }

    return true;
  }

  bool readRunData(RcdRecord &r) {

    return true;
  }

  bool takeTrigger(const RcdRecord &r) {

    if (_config.triggerByWaiting()) {

      nanosleep(&_delay,0);
    }

    if (_config.triggerInternally()) {

      if(_device!=0)
	assert(_device->takeTrigger());

      nanosleep(&_delay,0);
    }

    if (_config.triggerExternally()) {

      if(_device!=0)
	assert(_device->pollTrigger(10000));

      nanosleep(&_delay,0);
    }

    return true;
  }

protected:
  unsigned _interface;
  TtmLocation _location;
  unsigned char _crateNumber;
  DaqBusAdapter *_busAdapter;
  HAL::VMEAddressTableASCIIReader _addressTableReader;
  HAL::VMEAddressTable _addressTable;

  TtmVmeDevice *_device;
  DhcReadoutConfigurationData _config;
  timespec _delay;
  unsigned _nEvents;

};
 
#endif // TtmReadout_HH
