//
// $Id: TtmVmeDevice.hh,v 1.1 2008/06/27 10:34:05 meyern Exp $
//

#ifndef TtmVmeDevice_HH
#define TtmVmeDevice_HH

#include <iostream>
#include <vector>

// hal
#include "VMEDevice.hh"
#include "VMEAddressTable.hh"
#include "VMEBusAdapterInterface.hh"

#include "TtmConfigurationData.hh"

const uint32_t TTM_RAM_SIZE=2048;


class TtmVmeDevice : public HAL::VMEDevice {

public:
  TtmVmeDevice(HAL::VMEAddressTable &t,
	       HAL::VMEBusAdapterInterface &b, unsigned a)
    : HAL::VMEDevice(t,b,a<<27), _slot(a) {
  }

  unsigned slot() const {
    return _slot;
  }

  bool alive() {
    uint32_t value;
    try {
      read("VME_Firmware",&value);
      read("TD_Firmware",&value);
      return true;

    } catch ( HAL::HardwareAccessException& e ) {
      return false;

    } catch ( std::exception e ) {
      return false;
    }
  }

  bool reset() {

    try {
      write("TCS_TriggerSource", 1, HAL::HAL_NO_VERIFY);
      write("TCS_SignalSource", 0, HAL::HAL_NO_VERIFY);
      write("RAM_Depth", TTM_RAM_SIZE - 1, HAL::HAL_NO_VERIFY);
      for (unsigned i(0); i<TTM_RAM_SIZE; i++)
	write("RAM_Data", 0, HAL::HAL_NO_VERIFY, i*4);

      write("TCS_ResetTimeStampFifo", 1, HAL::HAL_NO_VERIFY);
      write("TCS_ResetTimeStampFifo", 0, HAL::HAL_NO_VERIFY);

      return true;

    } catch ( HAL::HardwareAccessException& e ) {
      return false;

    } catch ( std::exception e ) {
      return false;
    }
  }

  bool writeConfigurationData(const TtmConfigurationData &d) {

    write("TCS_SignalSource", 0, HAL::HAL_NO_VERIFY);
    if (d.ramDepth()>0) {
      const unsigned* ram(d.ramData());
      write("RAM_Depth", d.ramDepth(), HAL::HAL_NO_VERIFY);
      for (unsigned i(0); i<d.ramDepth(); i++)
	write("RAM_Data", ram[i], HAL::HAL_NO_VERIFY, i*4);
    }

    write("TCS_EnableTriggerDelay", d.tcsEnableTriggerDelay(), HAL::HAL_NO_VERIFY);
    write("TriggerDelay", d.triggerDelay(), HAL::HAL_NO_VERIFY);
    write("TCS_SignalSource", d.tcsRamSignal(), HAL::HAL_NO_VERIFY);
    write("TCS_ResetSource", d.tcsInternalReset(), HAL::HAL_NO_VERIFY);
    write("TCS_TcalSource", d.tcsInternalTcal(), HAL::HAL_NO_VERIFY);
    write("TCS_TriggerSource", d.tcsInternalTrigger(), HAL::HAL_NO_VERIFY);

    _useRam=d.tcsRamSignal();

    return true;
  }

  bool readConfigurationData(TtmConfigurationData &d) {
    uint32_t value;

    read("RAM_Depth",&value);
    d.ramDepth(value);

    if (d.ramDepth()>0) {
      read("TCS_SignalSource", &value);
      write("TCS_SignalSource", 0, HAL::HAL_NO_VERIFY);

      unsigned* ram(d.ramData());
      for (unsigned i(0); i<d.ramDepth(); i++)
	read("RAM_Data", &ram[i], i*4);

      write("TCS_SignalSource", value, HAL::HAL_NO_VERIFY);
    }

    read("VME_Firmware",&value);
    d.vmeFirmware(value);
    read("TD_Firmware",&value);
    d.tdFirmware(value);

    read("VCS",&value);
    d.vcs(value);
    read("TCS",&value);
    d.tcs(value);

    return true;
  }

  bool takeTrigger() {

    if (_useRam) {
      write("TCS_EnableRAM", 0, HAL::HAL_NO_VERIFY);
      write("TCS_EnableRAM", 1, HAL::HAL_NO_VERIFY);
    }
    else {
      write("ManualSignals", 3, HAL::HAL_NO_VERIFY);
      write("ManualSignals", 0, HAL::HAL_NO_VERIFY);
    }
    return true;
  }

  bool pollTrigger(unsigned timeout) {

    try {
      unsigned value;

      pollItem("TCS_TimeStampFifoEmpty", 1, timeout, &value, HAL::HAL_POLL_UNTIL_DIFFERENT);

      while (1) {
	read("TimeStampFifo", &value);
	read("TCS_TimeStampFifoEmpty",&value);
	if (value) break;
      }

      return true;

    } catch ( HAL::HardwareAccessException& e ) {
      return false;

    } catch ( std::exception e ) {
      return false;
    }
  }

  void print(std::ostream &o) {

    o << "TtmVmeDevice::print()" << std::endl;

    o << std::endl;
  }

private:
  unsigned _printLevel;
  unsigned _slot;
  bool _useRam;

};

#endif // TtmVmeDevice_HH
