#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <fcntl.h>
#include <unistd.h>

#include <iostream>
#include <sstream>
#include <vector>
#include <cstdio>

#include "VMEAddressTable.hh"
#include "VMEAddressTableASCIIReader.hh"
#include "SBS620x86LinuxBusAdapter.hh"
//#include "VMEDummyBusAdapter.hh"
#include "HardwareAccessException.hh"

#include "UtlTime.hh"
#include "UtlArguments.hh"
#include "RcdArena.hh"
#include "RcdWriterAsc.hh"
#include "RcdWriterBin.hh"
#include "RcdWriterDmy.hh"
#include "DaqRunStart.hh"

#include "SubAccessor.hh"

#include "CrcLocationData.hh"
#include "EmcReadout.hh"
#include "CrcReadoutConfigurationData.hh"
#include "OnlCounter.hh"
#include "HstNoise.hh"
#include "HstChipNoise.hh"
#include "HstChanNoise.hh"

#include "BmlLc1176VmeDevice.hh"
#include "BmlLc1176Readout.hh"

#include "DvrEmcVfeDac.hh"

#include "OnlRunNumber.hh"
#include "ShmObject.hh"

using std::cin;
using std::exception;
using namespace std;

#define CERCVMEADDRESSTABLE "online/hal/CrcVmeAddress.hal"
#define TDCVMEADDRESSTABLE "online/hal/Lc1176VmeAddress.hal"
#define SEQUENCE_SETTINGS "Sequences.dat"


bool continueJob=true;
bool continueRun=true;
bool continueConfiguration=true;
bool continueSpill=true;

void signalHandler(int signal) {
  std::cerr << "Process " << getpid() << " received signal "
	    << signal << std::endl;
  continueJob=false;
}

int main(int argc, const char **argv) {
  try{

    UtlArguments argh(argc,argv);
    //argh.print(cout);

    const bool doHistograms(argh.option('s',"Display histograms"));
    const bool useWriteDmy(argh.option('w',"Dummy output file"));
    const bool useWriteBin(argh.option('b',"Binary output file"));
    
    const unsigned printLevel(argh.optionArgument('p',0,"Print level"));
    const unsigned runType(argh.optionArgument('t',0,"Run type"));
    
    if(argh.help()) return 0;
    
    if(doHistograms) cout << "Histograms display selected" << endl;
    else           cout << "Histograms display not selected" << endl;
    if(useWriteDmy) cout << "Dummy output selected" << endl;
    else {
      if(useWriteBin) cout << "Binary file output selected" << endl;
      else            cout << "Acii file output selected" << endl;
    }

    cout << "Print level set to " << printLevel << endl;
    cout << "Run type set to " << runType << endl;
    
    // Define the CERC locations
    unsigned char theCrate(0xec);

    //const unsigned nSlots(1);
    //unsigned theSlots[nSlots]={12};

    //const unsigned nSlots(2);
    //unsigned theSlots[nSlots]={12,5};

    const unsigned nSlots(2);
    unsigned theSlots[nSlots]={12,17};
    
    SBS620x86LinuxBusAdapter busAdapter(0);
    VMEAddressTableASCIIReader addressTableReader( CERCVMEADDRESSTABLE );
    VMEAddressTable addressTable( "Test address table", addressTableReader );
    
    CrcVmeDevice *dev[nSlots];
    for(unsigned i(0);i<nSlots;i++) {
      dev[i]=new CrcVmeDevice(addressTable,busAdapter,theSlots[i]);
      assert(dev[i]->alive());
    }

    // Register CERCs
    EmcReadout oe(theCrate);
    oe.printLevel(printLevel);
    for(unsigned i(0);i<nSlots;i++) oe.device(dev[i],i==0);
    

    VMEAddressTableASCIIReader addressTableReader2( TDCVMEADDRESSTABLE );
    VMEAddressTable addressTable2( "Test address table", addressTableReader2 );

    BmlLc1176VmeDevice tdcVmeDevice(addressTable2,busAdapter,0x98); // DESY
    assert(tdcVmeDevice.alive());

    BmlLc1176Readout tdc(theCrate,tdcVmeDevice);
    tdc.printLevel(printLevel);

    HstBase *hn(0);
    //if(doHistograms) hn=new HstChipNoise(true);
    if(doHistograms) hn=new HstChanNoise(true);

    ShmObject<ChkCount> shmCt(123456786);
    ChkCount *pCt(shmCt.payload());

    RcdArena arena;
    SubAccessor extracter(arena);
    RcdArena *aArray(new RcdArena[256]);

    DrvEmcVfeDac dvr;
    dvr.printLevel(printLevel);

    // Create the poll object in advance
    CrcLocationData<CrcBeTrgPollData> poll;
    poll.crateNumber(theCrate);
    poll.slotNumber(theSlots[0]);
    poll.crcComponent(CrcLocation::beTrg);
    poll.label(0);
    if(printLevel>0) poll.print(std::cout) << std::endl;

    // Define output file
    RcdWriter *writer;
    if(useWriteDmy) writer=new RcdWriterDmy();
    else            writer=new RcdWriterAsc();
    //else            writer=new RcdWriterBin();
    

    // Loop over runs
    for(unsigned iRun(0);continueJob;iRun++) {
      continueRun=true;

      pCt->reset();

      const unsigned rnum(onlReadRunNumber());
      cout << "Counting run number " << rnum << endl;
      if(!useWriteDmy) assert(onlWriteRunNumber(rnum+1));

      // Set run number
      if(printLevel>0) cout << "Run number = " << rnum << endl;
      ostringstream oss;
      oss << "data/dat/Run" << rnum;

      // Ignor Ctrl^C from when we open the file
      signal(SIGINT,SIG_IGN);
      writer->open(oss.str());

      // Send run start
      arena.deleteData();
      arena.updateRecordTime();
      arena.recordType(RcdHeader::runStart);
      dvr.readout(arena);
      
      std::vector<const DaqRunStart*> vr(extracter.extract<DaqRunStart>());
      assert(vr.size()==1);
      ((DaqRunStart*)vr[0])->runNumber(rnum);

      // Readout and write
      oe.record(arena);
      tdc.record(arena);
      if(hn!=0) hn->record(arena);
      writer->write(arena);
      pCt->record(arena);

      // Now catch Ctrl^C
      signal(SIGINT,signalHandler);

      // Loop over configurations
      
      const unsigned nConfiguration(vr[0]->maximumNumberOfConfigurationsInRun());
      for(unsigned iConfiguration(0);iConfiguration<nConfiguration && continueJob && continueRun;iConfiguration++) {
	continueConfiguration=true;

	arena.deleteData();
	arena.updateRecordTime();
	arena.recordType(RcdHeader::configurationStart);
	dvr.readout(arena);

	oe.record(arena);
	if(hn!=0) hn->record(arena);
	writer->write(arena);
	pCt->record(arena);

	std::vector<const DaqConfigurationStart*> vc(extracter.extract<DaqConfigurationStart>());
	assert(vc.size()==1);

	const unsigned nSpill(vc[0]->maximumNumberOfSpillsInConfiguration());
	for(unsigned iSpill(0);iSpill<nSpill && continueJob && continueRun && continueConfiguration;iSpill++) {
	  continueSpill=true;

	  arena.deleteData();
	  arena.updateRecordTime();
	  arena.recordType(RcdHeader::spillStart);
	  dvr.record(arena);

	  oe.record(arena);
	  tdc.record(arena);
	  if(hn!=0) hn->record(arena);
	  writer->write(arena);
	  pCt->record(arena);

	  std::vector<const DaqSpillStart*> vs(extracter.extract<DaqSpillStart>());
	  assert(vs.size()==1);
	  
	  const unsigned nEvent(vc[0]->maximumNumberOfRunsInSpill());
	  for(iEvent=0;iEvent<nEvent && continueJob && continueRun && continueConfiguration && continueSpill;iEvent++) {

	    assert(dev[0]->clearBeTrgTrigger());
	    //assert(dev[0]->beTrgSoftTrigger());
	    //assert(dev[0]->beSoftTrigger());

	    /*
	    for(unsigned i(0);i<nSlots;i++) {
	      assert(dev[i]->clearBeTrgTrigger());
	      //assert(dev[i]->beSoftTrigger());
	      assert(dev[i]->beTrgSoftTrigger());
	      //assert(dev[i]->feSoftTrigger());
	    }
	    */

	    //for(unsigned i(1);i<nSlots;i++) {
	      //assert(dev[i]->beTrgSoftTrigger());
	      //assert(dev[i]->beTrgSoftTrigger());
	      //assert(dev[i]->beSoftTrigger()); //QWERTY
	      //assert(dev[i]->feSoftTrigger()); //QWERTY
	    //}
	    
	    dev[0]->pollBeTrgTrigger(*(poll.data()),100000);
	    aArray[iEvent].updateRecordTime();
	    if(printLevel>4) poll.print(std::cout) << std::endl;
	    SubInserter inserter(aArray[iEvent]);
	    inserter.insert< CrcLocationData<CrcBeTrgPollData> >(poll);


	    oe.record(aArray[iEvent]);
	    //tdc.record(aArray[iEvent]);
	  }
      
	  arena.deleteData();
	  arena.updateRecordTime();
	  if(continueConfiguration) arena.recordType(RcdHeader::spillEnd);
	  else                      arena.recordType(RcdHeader::spillStart);
	  dvr.record(arena);

	  oe.record(arena);

	  for(unsigned g(0);g<iEvent;g++) {
	    if(hn!=0) hn->record(aArray[g]);
	    writer->write(aArray[g]);
	    pCt->record(aArray[g]);
	  }

	  if(hn!=0) {
	    hn->record(arena);
	    hn->update();
	    //if(iSpill>1) hn->postscript("junk");
	  }
	  writer->write(arena);
	  pCt->record(arena);

	  if(iSpill==0) {
	    arena.deleteData();
	    arena.updateRecordTime();
	    arena.recordType(RcdHeader::slowRecord);

	    oe.record(arena);
	    if(hn!=0) hn->record(arena);
	    writer->write(arena);
	    pCt->record(arena);
	  }

	  if(printLevel>4) cout << "Number of MBytes written = " << 0.000001*writer->numberOfBytes() << endl;
	  if(writer->numberOfBytes()>1800000000) continueRun=false;
 	}
	
	arena.deleteData();
	arena.updateRecordTime();
	if(continueJob) arena.recordType(RcdHeader::configurationEnd);
	else            arena.recordType(RcdHeader::configurationStop);

	oe.record(arena);
	if(hn!=0) hn->record(arena);
	writer->write(arena);
	pCt->record(arena);
      }

      signal(SIGINT,SIG_IGN);

      arena.deleteData();
      arena.updateRecordTime();
      if(continueJob) arena.recordType(RcdHeader::runEnd);
      else            arena.recordType(RcdHeader::runStop);

      oe.record(arena);
      if(hn!=0) hn->record(arena);
      writer->write(arena);
      pCt->record(arena);

      writer->close();
    }

    if(hn!=0) delete hn;

  } catch ( HardwareAccessException& e ) {
    cout << "*** Exception occurred : " << e.what() << endl;
    
  } catch ( exception e ) {
    cout << "*** Unknown exception occurred" << endl;
  }
}
