#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <fcntl.h>
#include <unistd.h>

#include <iostream>
#include <sstream>
#include <vector>
#include <cstdio>

#include "VMEAddressTable.hh"
#include "VMEAddressTableASCIIReader.hh"
#include "SBS620x86LinuxBusAdapter.hh"
//#include "VMEDummyBusAdapter.hh"
#include "HardwareAccessException.hh"

#include "UtlTime.hh"
#include "UtlArguments.hh"
#include "RcdArena.hh"
#include "RcdWriterAsc.hh"
#include "RcdWriterBin.hh"
#include "RcdWriterDmy.hh"
#include "DaqRunStart.hh"

#include "SubAccessor.hh"

#include "CrcLocationData.hh"
#include "EmcReadout.hh"
#include "CrcReadoutConfigurationData.hh"
#include "OnlCounter.hh"
#include "HstNoise.hh"
#include "HstChipNoise.hh"
#include "HstChanNoise.hh"

using std::cin;
using std::exception;
using namespace std;

#define CERCVMEADDRESSTABLE "online/hal/CrcVmeAddress.hal"
#define SEQUENCE_SETTINGS "Sequences.dat"


bool continueJob=true;
bool continueRun=true;
bool continueConfiguration=true;

void signalHandler(int signal) {
  std::cerr << "Process " << getpid() << " received signal "
	    << signal << std::endl;
  continueJob=false;
}

int main(int argc, const char **argv) {
  try{

    UtlArguments argh(argc,argv);
    //argh.print(cout);

    const bool doCheck(argh.option('c',"Run data check"));
    const bool doHistograms(argh.option('s',"Display histograms"));
    const bool doGraphics(argh.option('g',"Display graphics"));
    const bool useWriteDmy(argh.option('w',"Dummy output file"));
    
    const unsigned printLevel(argh.optionArgument('p',5,"Print level"));
    const unsigned nEvents(argh.optionArgument('n',10,"Number of events between updates"));
    
    std::string logFile(argh.optionArgument('l',"log/msg.log","RS232 message log file"));
    
    if(argh.help()) return 0;
    
    if(doCheck) cout << "Data check selected" << endl;
    else           cout << "Data check not selected" << endl;
    if(doHistograms) cout << "Histograms display selected" << endl;
    else           cout << "Histograms display not selected" << endl;
    if(doGraphics) cout << "Graphics display selected" << endl;
    else           cout << "Graphics display not selected" << endl;
    if(useWriteDmy) cout << "Dummy output selected" << endl;
    else              cout << "File output selected" << endl;
    
    cout << "Print level set to " << printLevel << endl;
    cout << "Number of events set to " << nEvents << endl;
    
    // Define the CERC locations
    unsigned char theCrate(0xec);

    //const unsigned nSlots(4);
    //unsigned theSlots[nSlots]={12,5,7,17};
    
    SBS620x86LinuxBusAdapter busAdapter(0);
    VMEAddressTableASCIIReader addressTableReader( CERCVMEADDRESSTABLE );
    VMEAddressTable addressTable( "Test address table", addressTableReader );
    
    unsigned nSlots(0);
    unsigned theSlots[22];
    
    // Find CRCs automatically
    CrcVmeDevice *dev[22];
    for(unsigned i(4);i<=21;i++) {
      unsigned j(i);
      if(i==4) j=12;
      if(i!=12) { // Special trigger slot
	dev[nSlots]=new CrcVmeDevice(addressTable,busAdapter,j);
	if(dev[nSlots]->alive()) {
	  theSlots[nSlots]=j;
	  cout << "Slot " << j << " found alive" << endl;
	  nSlots++;
	} else {
	  delete dev[nSlots];
	}
      }
    }

    assert(nSlots>0);

    // Register CERCs
    EmcReadout oe(theCrate);
    oe.printLevel(printLevel);
    for(unsigned i(0);i<nSlots;i++) oe.device(dev[i],i==0);
    
    HstBase *hn(0);
    //if(doHistograms) hn=new HstNoise(theSlots[0]);
    //if(doHistograms) hn=new HstChipNoise(true);
    if(doHistograms) hn=new HstChipNoise(true);

    RcdArena arena;
    RcdArena *aArray(new RcdArena[256]);

    // Send Startup record
    OnlCounter od;
    od.readout(arena,RcdHeader::startUp);
    oe.record(arena);

    // Send Slow controls record
    od.readout(arena,RcdHeader::slowControl);
    SubInserter inserter(arena);

    for(unsigned i(0);i<nSlots;i++) {
      CrcLocationData<CrcAdm1025SlowControlsData>
	*b(inserter.insert< CrcLocationData<CrcAdm1025SlowControlsData> >());
      b->crateNumber(theCrate);
      b->slotNumber(theSlots[i]);
      b->crcComponent(CrcLocation::vme);
      b->label(1);

      *b->data()=CrcAdm1025SlowControlsData();
      b->print(std::cout) << std::endl;

      CrcLocationData<CrcLm82SlowControlsData>
	*v(inserter.insert< CrcLocationData<CrcLm82SlowControlsData> >());
      v->crateNumber(theCrate);
      v->slotNumber(theSlots[i]);
      v->crcComponent(CrcLocation::vmeLm82);
      v->label(1);
      
      *v->data()=CrcLm82SlowControlsData();
      v->print(std::cout) << std::endl;

      CrcLocationData<CrcLm82SlowControlsData>
	*l(inserter.insert< CrcLocationData<CrcLm82SlowControlsData> >());
      l->crateNumber(theCrate);
      l->slotNumber(theSlots[i]);
      l->crcComponent(CrcLocation::be);
      l->label(1);
      
      *l->data()=CrcLm82SlowControlsData();
      l->print(std::cout) << std::endl;

      for(unsigned f(0);f<8;f++) {
	CrcLocationData<CrcLm82SlowControlsData>
	  *l(inserter.insert< CrcLocationData<CrcLm82SlowControlsData> >());
	l->crateNumber(theCrate);
	l->slotNumber(theSlots[i]);
	l->crcComponent((CrcLocation::CrcComponent)f);
	l->label(1);
	
	*l->data()=CrcLm82SlowControlsData();
	l->print(std::cout) << std::endl;
      }
    }
  
    oe.record(arena);

    // Create the poll object in advance
    CrcLocationData<CrcBeTrgPollData> poll;
    poll.crateNumber(theCrate);
    poll.slotNumber(theSlots[0]);
    poll.crcComponent(CrcLocation::beTrg);
    poll.label(0);
    poll.print(std::cout) << std::endl;

    // Loop over runs
    for(unsigned iRun(0);iRun<1 && continueJob;iRun++) {

      // Send Run start
      od.readout(arena,RcdHeader::runStart);
      SubAccessor extracter(arena);
      std::vector<const DaqRunStart*> v(extracter.extract<DaqRunStart>());
      assert(v.size()==1);

      // Set run number
      unsigned runNumber=v[0]->runNumber();
      cout << "Run number = " << runNumber << endl;
      ostringstream oss;
      oss << "data/dat/Run" << runNumber;

      // Ignor Ctrl^C for a while
      signal(SIGINT,SIG_IGN);

      // Open output file
      RcdWriter *writer;
      if(useWriteDmy) writer=new RcdWriterDmy();
      else            writer=new RcdWriterAsc();
      //RcdWriterBin writer;
      writer->open(oss.str());

      // Readout and write
      oe.record(arena);
      if(hn!=0) hn->record(arena);
      writer->write(arena);
    
      // Loop over configurations
      for(unsigned iConfiguration(0);iConfiguration<1024 && continueJob && continueRun;iConfiguration++) {
        unsigned dac0(0),dac1(0);
        if((iConfiguration%2)==0) dac0=128*(511-(iConfiguration/2)); // 1024 configurations
        else                      dac1=128*(511-(iConfiguration/2));
        //if((iConfiguration%2)==0) dac0=4096*(15-(iConfiguration/2)); // 32 configurations
        //else                      dac1=4096*(15-(iConfiguration/2));

	od.readout(arena,RcdHeader::configurationStart);

	SubInserter inserter(arena);

	// Add readout control
	CrcReadoutConfigurationData
	  *b(inserter.insert<CrcReadoutConfigurationData>());
	*b=CrcReadoutConfigurationData();
	b->vmePeriod(1);
	b->bePeriod(1);
	b->fePeriod(1);
	b->beTrgMode(1);
	b->vlinkMode(3);
	b->print(std::cout) << std::endl;

	for(unsigned i(0);i<nSlots;i++) {
	  //if(i==0) {
	  CrcLocationData<CrcBeConfigurationData>
	    *b(inserter.insert< CrcLocationData<CrcBeConfigurationData> >());
	  b->crateNumber(theCrate);
	  //b->slotNumber(theSlots[i]);
	  b->slotBroadcast(true);
	  b->crcComponent(CrcLocation::be);
	  b->label(1);

	  *b->data()=CrcBeConfigurationData();
	  b->data()->mode(0x02);
	  b->data()->test(0x00);
	  b->data()->testLength(0);
	  //if(i==0) 
	  b->data()->trgEnables(0x1ff); // 0x1ff for J0, 0x2ff for internal
	  //else b->data()->trgEnables(0x2ff); // 0x1ff for J0, 0x2ff for internal
	  b->print(std::cout) << std::endl;
	  //}

	  // Only do BE-Trg for first board
	  if(i==0) {
	    CrcLocationData<CrcBeTrgConfigurationData>
	      *bt(inserter.insert< CrcLocationData<CrcBeTrgConfigurationData> >());
	    bt->crateNumber(theCrate);
	    bt->slotNumber(theSlots[i]);
	    bt->crcComponent(CrcLocation::beTrg);
	    bt->label(1);

	    *bt->data()=CrcBeTrgConfigurationData();
	    //
	    //bt->data()->inputEnable(1<<0); // DevBoard
	    //bt->data()->inputEnable(1<<24); // Trg osc

	    //bt->data()->outputEnable(1); // Bit 0 enables J0 fanout
	    //bt->data()->oscillationPeriod(40000000); // 1 sec
	    //bt->data()->oscillationPeriod(4000000); // 0.1 sec
	    bt->data()->oscillationPeriod(400000); // 0.01 sec
	    //bt->data()->oscillationPeriod(40000); // 0.001 sec
	    bt->print(std::cout) << std::endl;
	  }

	  for(unsigned f(0);f<8;f++) {
	    CrcLocationData<EmcFeConfigurationData>
	      *b(inserter.insert< CrcLocationData<EmcFeConfigurationData> >());
	    b->crateNumber(theCrate);
	    b->slotNumber(theSlots[i]);
	    b->crcComponent((CrcLocation::CrcComponent)f);
	    b->label(1);

	    *b->data()=EmcFeConfigurationData();
	    unsigned nadc=18;
	    // VFE Calib only!
	    //b->data()->calibEnable(true);
	    //b->data()->holdStart(30);
	    //b->data()->vfeEnable(CrcFeConfigurationData::bot,0,true);
	    //b->data()->vfeEnable(CrcFeConfigurationData::top,0,true);

	    b->data()->holdStart(1);
	    b->data()->holdWidth(nadc*4*700);
	    b->data()->vfeMplexClockPulses(nadc);
	    b->data()->internalDacEnable(true);
	    //b->data()->vfeLowGain(CrcFeConfigurationData::bot,false);
	    //b->data()->vfeLowGain(CrcFeConfigurationData::top,false);
	    b->data()->dacData(CrcFeConfigurationData::bot,dac0);
	    b->data()->dacData(CrcFeConfigurationData::top,dac1);
	    b->print(std::cout) << std::endl;

	    CrcLocationData<CrcFeFakeEventData>
	      *be(inserter.insert< CrcLocationData<CrcFeFakeEventData> >());
	    be->crateNumber(theCrate);
	    be->slotNumber(theSlots[i]);
	    be->crcComponent((CrcLocation::CrcComponent)f);
	    be->label(1);

	    be->data()->numberOfWords(32);
	    be->data()->enable(false);
	    unsigned *fe(be->data()->data());
	    for(unsigned ie(0);ie<be->data()->numberOfWords();ie++) {
	      if((ie%2)==0) fe[ie]=((~((  f)*8192+ie))<<16)&0xffff0000|((7-f)*8192+ie);
	      else          fe[ie]=((~((7-f)*8192+ie))<<16)&0xffff0000|((  f)*8192+ie);
	    }
	    inserter.extend(be->data()->numberOfWords()*4);

	    be->print(std::cout) << std::endl;
	  }
	}

	oe.record(arena);
	if(hn!=0) hn->record(arena);
	writer->write(arena);

	// Now catch Ctrl^C
	signal(SIGINT,signalHandler);

	// Loop over spills
	for(unsigned iSpill(0);iSpill<1 && continueJob && continueRun && continueConfiguration;iSpill++) {
	  od.readout(arena,RcdHeader::spillStart);
	  oe.record(arena);
	  if(hn!=0) hn->record(arena);
	  writer->write(arena);

	  unsigned iEvent(0);
	  for(iEvent=0;iEvent<32 && continueConfiguration;iEvent++) {
	    od.readout(aArray[iEvent],RcdHeader::event);

	    assert(dev[0]->clearBeTrgTrigger());
	    assert(dev[0]->beTrgSoftTrigger());
	    //assert(dev[0]->beSoftTrigger());

	    /*
	    for(unsigned i(0);i<nSlots;i++) {
	      assert(dev[i]->clearBeTrgTrigger());
	      //assert(dev[i]->beSoftTrigger());
	      assert(dev[i]->beTrgSoftTrigger());
	      //assert(dev[i]->feSoftTrigger());
	    }
	    */

	    //for(unsigned i(1);i<nSlots;i++) {
	      //assert(dev[i]->beTrgSoftTrigger());
	      //assert(dev[i]->beTrgSoftTrigger());
	      //assert(dev[i]->beSoftTrigger()); //QWERTY
	      //assert(dev[i]->feSoftTrigger()); //QWERTY
	    //}
	    
	    dev[0]->pollBeTrgTrigger(*(poll.data()),1000);
	    aArray[iEvent].updateRecordTime();
	    if(printLevel>4) poll.print(std::cout) << std::endl;
	    SubInserter inserter(aArray[iEvent]);
	    inserter.insert< CrcLocationData<CrcBeTrgPollData> >(poll);

	    oe.record(aArray[iEvent]);
	  }
      
	  if(continueConfiguration) od.readout(arena,RcdHeader::spillEnd);
	  else                      od.readout(arena,RcdHeader::spillStop);
	  oe.record(arena);

	  for(unsigned g(0);g<iEvent;g++) {
	    if(hn!=0) hn->record(aArray[g]);
	    writer->write(aArray[g]);
	  }

	  if(hn!=0) {
	    hn->record(arena);
	    hn->update();
	    //if(iSpill>1) hn->postscript("junk");
	  }
	  writer->write(arena);

	  od.readout(arena,RcdHeader::slowReadout);
	  oe.record(arena);
	  if(hn!=0) hn->record(arena);
	  writer->write(arena);
	}

	signal(SIGINT,SIG_IGN);
	
	if(continueJob) od.readout(arena,RcdHeader::configurationEnd);
	else            od.readout(arena,RcdHeader::configurationStop);
	oe.record(arena);
	if(hn!=0) hn->record(arena);
	writer->write(arena);
      }

      if(continueJob) od.readout(arena,RcdHeader::runEnd);
      else            od.readout(arena,RcdHeader::runStop);
      oe.record(arena);
      if(hn!=0) hn->record(arena);
      writer->write(arena);

      writer->close();
    }

    if(hn!=0) delete hn;

  } catch ( HardwareAccessException& e ) {
    cout << "*** Exception occurred : " << e.what() << endl;
    
  } catch ( exception e ) {
    cout << "*** Unknown exception occurred" << endl;
  }
}
