#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <fcntl.h>
#include <unistd.h>

#include <iostream>
#include <sstream>
#include <vector>
#include <cstdio>

// hal
#include "VMEAddressTable.hh"
#include "VMEAddressTableASCIIReader.hh"
#include "SBS620x86LinuxBusAdapter.hh"
//nclude "VMEDummyBusAdapter.hh"
#include "HardwareAccessException.hh"

// dual/inc/utl
#include "UtlTime.hh"
#include "UtlArguments.hh"

// dual/inc/rcd
#include "RcdArena.hh"
#include "RcdWriterAsc.hh"
#include "RcdWriterBin.hh"
#include "RcdWriterDmy.hh"
#include "RcdMultiUserRW.hh"
#include "RcdMultiUserRO.hh"

// dual/inc/sub
#include "SubAccessor.hh"

// dual/inc/daq
#include "DaqRunStart.hh"
#include "DaqConfigurationStart.hh"
#include "DaqSpillStart.hh"
#include "DaqWriter.hh"

// dual/inc/crc
#include "CrcLocationData.hh"
#include "CrcReadoutConfigurationData.hh"

// dual/inc/hst
#include "HstNoise.hh"
#include "HstChipNoise.hh"
#include "HstChanNoise.hh"

// online/inc/daq
#include "DaqCounter.hh"

// online/inc/emc
#include "EmcReadout.hh"

// online/inc/dvr
#include "DvrEmcIntDac.hh"
#include "DvrCrcSlw.hh"

// online/inc/shm
#include "ShmObject.hh"

// online/inc/daq
#include "DaqSlwControl.hh"


using namespace std;

DaqSlwControl *pDsc(0);

void signalHandler(int signal) {
  std::cerr << "Process " << getpid() << " received signal "
	    << signal << std::endl;

  if(signal==SIGINT || signal==SIGTERM) pDsc->command(DaqSlwControl::shutdown);

  //  if(signal==SIGUSR1) *pDsc->command(DaqSlwControl::);
}

int main(int argc, const char **argv) {
  unsigned eTime(CALICE_DAQ_TIME);
  cout << argv[0] << " compiled at " << ctime((const time_t*)&eTime) << endl;

  UtlArguments argh(argc,argv);

  const bool useWriteDmy(argh.option('w',"Dummy output file"));
  const bool useWriteAsc(argh.option('a',"Ascii output file"));
  const bool doHistograms(argh.option('s',"Display histograms"));
  
  const unsigned nSlw(argh.optionArgument('r',1,"Number of slow runs"));
  const unsigned printLevel(argh.optionArgument('p',0,"Print level"));
  
  if(argh.help()) return 0;
  
  if(doHistograms)  cout << "Histograms display selected" << endl;
  else              cout << "Histograms display not selected" << endl;
  if(useWriteDmy)   cout << "Dummy output selected" << endl;
  else {
    if(useWriteAsc) cout << "Ascii output selected" << endl;
    else            cout << "Binary output selected" << endl;
  }
  
  cout << "Print level set to " << printLevel << endl;
  cout << "Number of slow runs set to " << nSlw << endl;
  cout << endl;

  /*
  ShmObject<unsigned> shmRequest(45454545);
  ShmObject<unsigned> shmStatus(45454546);
  pRequest=shmRequest.payload();
  pStatus=shmStatus.payload();
  *pRequest=999;
  *pStatus=0;
  */

  ShmObject<DaqSlwControl> shmSlwControl(45454547);
  DaqSlwControl *pDsc(shmSlwControl.payload());
  assert(pDsc!=0);
  pDsc->print(cout);
  //ChkCount *chk(pDsc->count());
  ChkCount *chk(new ChkCount);
  chk->print(cout);

  pDsc->processId();
  pDsc->command(DaqSlwControl::run,false);
  pDsc->status(DaqSlwControl::endOfStatusEnum);

  try{

    // Define the CERC locations
    unsigned char theCrate(0xec);

    //const unsigned nSlots(4);
    //unsigned theSlots[nSlots]={12,5,7,17};
    
    SBS620x86LinuxBusAdapter busAdapter(0);
    VMEAddressTableASCIIReader addressTableReader("online/hal/CrcVmeAddress.hal");
    VMEAddressTable addressTable("CRC VME Address Table",addressTableReader);
    
    unsigned nSlots(0);
    unsigned theSlots[22];
    
    // Find CRCs automatically
    CrcVmeDevice *dev[22];
    for(unsigned i(5);i<=21;i++) {
      dev[nSlots]=new CrcVmeDevice(addressTable,busAdapter,i);
      if(dev[nSlots]->alive()) {
	theSlots[nSlots]=i;
	cout << "Slot " << i << " found alive" << endl;
	nSlots++;
      } else {
	delete dev[nSlots];
      }
    }

    assert(nSlots>0);

    // Define lists of user modules
    //vector<RcdUserBase*> vrub;
    RcdMultiUserRW vRrw;
    vRrw.printLevel(printLevel);
    RcdMultiUserRO vrub;
    vrub.printLevel(printLevel);

    // Add driver module
    DvrCrcSlw dcs(theCrate);
    vRrw.addUser(dcs);

    // Add CRC readout module and register CRCs
    EmcReadout er(theCrate);
    for(unsigned i(0);i<nSlots;i++) er.device(dev[i],false);
    vRrw.addUser(er);
    
    // Add counter
    //vrub.addUser(*chk);

    // Add writer module
    //DaqWriter dw(useWriteDmy,!useWriteAsc);
    //vrub.addUser(dw);
    RcdWriterBin writer;

    RcdArena arena;

    // Catch TERM but ignor Ctrl^C for a while
    signal(SIGTERM,signalHandler);
    signal(SIGUSR1,signalHandler);
    signal(SIGINT,SIG_IGN);

    // Loop over runs
    for(unsigned iSlw(0);pDsc->command()>DaqSlwControl::shutdown;iSlw++) {
      pDsc->status(DaqSlwControl::waiting);
      while(pDsc->command()==DaqSlwControl::wait) sleep(60);

      if(pDsc->command()>DaqSlwControl::wait) {
	pDsc->status(DaqSlwControl::initialising);

	// Now catch Ctrl^C
	signal(SIGINT,signalHandler);
	
	std::ostringstream sout;
	sout << "data/slw/Slw" << time(0);
	assert(writer.open(sout.str()));
	
	// Send Startup record
	arena.initialiseRecord(RcdHeader::startUp);
	vRrw.record(arena);
	vrub.record(arena);
	assert(writer.write(arena));
	
	// Send slow controls record
	arena.initialiseRecord(RcdHeader::slowControl);
	vRrw.record(arena);
	vrub.record(arena);
	assert(writer.write(arena));
	
	// Loop over slow readouts
	unsigned nCfg(0xffffffff);
	for(unsigned iCfg(0);iCfg<nCfg && pDsc->command()>DaqSlwControl::wait;iCfg++) {
	  pDsc->status(DaqSlwControl::running);
	  
	  arena.initialiseRecord(RcdHeader::slowReadout);
	  vRrw.record(arena);
	  vrub.record(arena);
	  assert(writer.write(arena));




	  //chk->record(arena);
	  //chk->print(cout);




	  
	  sleep(60);
	}

	// Send end-of-file record
        if(pDsc->command()>DaqSlwControl::shutdown;iSlw++) {
          arena.initialiseRecord(RcdHeader::slowEnd);
        } else {
          arena.initialiseRecord(RcdHeader::shutdown);
        }
 
	vRrw.record(arena);
	vrub.record(arena);

	assert(writer.close());

	// Ignor Ctrl-C again
	signal(SIGINT,SIG_IGN);
      }
    }

  } catch ( HardwareAccessException& e ) {
    cout << "*** Exception occurred : " << e.what() << endl;
    
  } catch ( exception e ) {
    cout << "*** Unknown exception occurred" << endl;
  }

  pDsc->processId(0);
  pDsc->command(DaqSlwControl::endOfCommandEnum,false);
  pDsc->status(DaqSlwControl::endOfStatusEnum);
}
