#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <fcntl.h>
#include <unistd.h>

#include <iostream>
#include <sstream>
#include <vector>
#include <cstdio>

#include "VMEAddressTable.hh"
#include "VMEAddressTableASCIIReader.hh"
#include "SBS620x86LinuxBusAdapter.hh"
//#include "VMEDummyBusAdapter.hh"
#include "HardwareAccessException.hh"

#include "UtlTime.hh"
#include "UtlArguments.hh"
#include "RcdArena.hh"
#include "RcdWriterAsc.hh"
#include "RcdWriterBin.hh"
#include "RcdWriterDmy.hh"
#include "DaqRunStart.hh"

#include "SubExtracter.hh"

#include "CrcLocationData.hh"
#include "CrcReadout.hh"
#include "CrcReadoutConfigurationData.hh"
#include "OnlCounter.hh"
#include "HstNoise.hh"

using std::cin;
using std::exception;
using namespace std;

#define CERCVMEADDRESSTABLE "CercAddressMap.dat"
#define SEQUENCE_SETTINGS "Sequences.dat"


bool continueJob=true;
bool continueRun=true;
bool continueConfiguration=true;

void signalHandler(int signal) {
  std::cerr << "Process " << getpid() << " received signal "
	    << signal << std::endl;
  continueJob=false;
}

int main(int argc, const char **argv) {
  try{

    UtlArguments argh(argc,argv);
    //argh.print(cout);

    const bool doCheck(argh.option('c',"Run data check"));
    const bool doHistograms(argh.option('s',"Display histograms"));
    const bool doGraphics(argh.option('g',"Display graphics"));
    const bool useIntTrigger(argh.option('i',"Internal trigger selection"));
    
    const unsigned printLevel(argh.optionArgument('p',5,"Print level"));
    const unsigned feMask(argh.optionArgument('f',0xff,"FE mask"));
    const unsigned nEvents(argh.optionArgument('n',10,"Number of events between updates"));
    
    std::string logFile(argh.optionArgument('l',"log/msg.log","RS232 message log file"));
    
    if(argh.help()) return 0;
    
    if(doCheck) cout << "Data check selected" << endl;
    else           cout << "Data check not selected" << endl;
    if(doHistograms) cout << "Histograms display selected" << endl;
    else           cout << "Histograms display not selected" << endl;
    if(doGraphics) cout << "Graphics display selected" << endl;
    else           cout << "Graphics display not selected" << endl;
    if(useIntTrigger) cout << "Internal trigger selected" << endl;
    else              cout << "External trigger selected" << endl;
    
    cout << "Print level set to " << printLevel << endl;
    cout << "FE mask set to " << printHex((unsigned char)feMask) << endl;
    cout << "Number of events set to " << nEvents << endl;
    
    // Define the CERC locations
    unsigned char theCrate(0xec);
    //const unsigned nSlots(4);
    //unsigned theSlots[nSlots]={6,10,14,18};
    //const unsigned nSlots(3);
    //unsigned theSlots[nSlots]={6,14,18};
    const unsigned nSlots(1);
    unsigned theSlots[nSlots]={14};
    
    SBS620x86LinuxBusAdapter busAdapter(0);
    VMEAddressTableASCIIReader addressTableReader( CERCVMEADDRESSTABLE );
    VMEAddressTable addressTable( "Test address table", addressTableReader );
    
    CrcVmeDevice *dev[nSlots];
    for(unsigned i(0);i<nSlots;i++) {
      dev[i]=new CrcVmeDevice(addressTable,busAdapter,theSlots[i]);
      assert(dev[i]->alive());
    }

    // Register CERCs
    OnlCounter od;
    
    CrcReadout oe(theCrate);
    oe.printLevel(printLevel);
    for(unsigned i(0);i<nSlots;i++) oe.device(dev[i],feMask,i==0);

    RcdArena arena;

      // Send Run start
      od.readout(arena,SubHeader::runStart);
      oe.record(arena);

  } catch ( HardwareAccessException& e ) {
    cout << "*** Exception occurred : " << e.what() << endl;
    
  } catch ( exception e ) {
    cout << "*** Unknown exception occurred" << endl;
  }
}
