#ifndef SceSlowReadoutData_HH
#define SceSlowReadoutData_HH

#include <string>
#include <iostream>


class SceSlowReadoutData {

public:
  enum {
    versionNumber=0
  };

  SceSlowReadoutData();

  bool parse(std::string r);
  
  time_t timeStamp() const;
  void   timeStamp(time_t t);

  unsigned moduleNumber() const;

  std::vector<double> getCmbTemperatures() const;

  std::vector<double>  getCmbVoltages() const;

  unsigned getLedSetting() const; 

  std::vector<unsigned>  getCmbSizes() const;

  double getHbabVoltage() const;

  double getHbabCurrent() const;

  std::ostream& print(std::ostream &o, std::string s="") const;


private:
  time_t _timeStamp;
  unsigned _moduleNumber;
  double _cmbTemperatures[2];
  double _cmbVoltages[9];
  unsigned _ledSetting;
  unsigned _cmbSizes[2];
  double _hbabVoltage;
  double _hbabCurrent;
};


#ifdef CALICE_DAQ_ICC

#include <time.h>
#include <cstring>


SceSlowReadoutData::SceSlowReadoutData() {
  memset(this,0,sizeof(SceSlowReadoutData));
}

bool SceSlowReadoutData::parse(std::string r) {
  memset(this,0,sizeof(SceSlowReadoutData));
  _timeStamp=0xffffffff;
  _moduleNumber=255;

  std::istringstream sin(r);
  if(!sin) return false;

  sin >> _timeStamp;
  if(!sin) return false;

  sin >> _moduleNumber;
  if(!sin) return false;

  for(unsigned i(0);i<2;i++) {
    sin >> _cmbTemperatures[i];
    if(!sin) return false;
  }

  for(unsigned i(0);i<9;i++) {
    sin >> _cmbVoltages[i];
    if(!sin) return false;
  }

  sin >> std::hex >> _ledSetting >> std::dec;
  if(!sin) return false;

  for(unsigned i(0);i<2;i++) {
    sin >> _cmbSizes[i];
    if(!sin) return false;
  }

  sin >> _hbabVoltage;
  if(!sin) return false;

  sin >> _hbabCurrent;
  if(!sin) return false;

  return true;
}

time_t SceSlowReadoutData::timeStamp() const {
  return _timeStamp;
}

void SceSlowReadoutData::timeStamp(time_t t) {
  _timeStamp=t;
}

unsigned SceSlowReadoutData::moduleNumber() const {
  return _moduleNumber;
}
std::vector<double> SceSlowReadoutData::getCmbTemperatures() const {
  std::vector<double> v;

  for(unsigned i(0);i<2;i++) {
    v.push_back(_cmbTemperatures[i]);
  }
  
  return v;
}

std::vector<double> SceSlowReadoutData::getCmbVoltages() const {
  std::vector<double> v;

  for(unsigned i(0);i<9;i++) {
    v.push_back(_cmbVoltages[i]);
  }
  
  return v;
}

unsigned SceSlowReadoutData::getLedSetting() const{
 return _ledSetting;
}

std::vector<unsigned> SceSlowReadoutData::getCmbSizes() const{
  std::vector<unsigned> v;

  for(unsigned i(0);i<2;i++) {
    v.push_back(_cmbSizes[i]);
  }
  
  return v;
}

double SceSlowReadoutData::getHbabVoltage() const{
  return _hbabVoltage;
}

double SceSlowReadoutData::getHbabCurrent() const{
  return _hbabCurrent;
}



std::ostream& SceSlowReadoutData::print(std::ostream &o, std::string s) const {
  o << s << "SceSlowReadoutData::print()" << std::endl;
  o << s << " Timestamp = " << _timeStamp << " = " << ctime(&_timeStamp);
  o << s << " Module number = " << _moduleNumber << std::endl;

  o << s << " CMB temperature lower = " << std::setw(5)
    << _cmbTemperatures[0] << " C" << std::endl;
  o << s << " CMB temperature upper = " << std::setw(5)
    << _cmbTemperatures[1] << " C" << std::endl;

  o << s << " CMB calib U041 voltage = " << _cmbVoltages[0] << " V" << std::endl;
  o << s << " CMB 12V power  voltage = " << _cmbVoltages[1] << " V" << std::endl;
  o << s << " CMB 1.235V     voltage = " << _cmbVoltages[2] << " V" << std::endl;
  o << s << " CMB VLDA upper voltage = " << _cmbVoltages[3] << " V" << std::endl;
  o << s << " CMB VLDB upper voltage = " << _cmbVoltages[4] << " V" << std::endl;
  o << s << " CMB VLDC upper voltage = " << _cmbVoltages[5] << " V" << std::endl;
  o << s << " CMB VLDD lower voltage = " << _cmbVoltages[6] << " V" << std::endl;
  o << s << " CMB 10V bias   voltage = " << _cmbVoltages[7] << " V" << std::endl;
  o << s << " CMB calib U051 voltage = " << _cmbVoltages[8] << " V" << std::endl;

  o << s << " LED setting = " << printHex(_ledSetting) << std::endl;

  o << s << " LED pulser width  = " << std::setw(3) << _cmbSizes[0] << std::endl;
  o << s << " LED pulser height = " << std::setw(3) << _cmbSizes[1] << std::endl;

  o << s << " HBAB HV voltage  = " << _hbabVoltage << " V" << std::endl;
  o << s << " HBAB HV current  = " << _hbabCurrent << " A" << std::endl;

  return o;
}

#endif
#endif
