#ifndef DaqRunType_HH
#define DaqRunType_HH

/**********************************************************************
 * DaqRunType - defines run types available
 *
 * Whenever you add or change the run types, you must also edit the
 * DaqConfiguration.hh and corresponding XxxConfiguration.hh file
 *
 **********************************************************************/

#include <string>
#include <iostream>

#include "UtlPack.hh"


class DaqRunType {

public:
  enum MajorType {
    daq,      // DAQ tests only; no readout
    crc,      // CRC crates using internal software triggers only
    trg,      // CRC trigger board triggers
    emc,      // ECAL crate only
    ahc,      // AHCAL crate only
    ahc1,     // AHCAL crate only continued
    dhc,      // DHCAL crate only
    tcm,      // TCMT CRC only
    bml,      // Beam line equipment only
    slow,     // Slow controls and readout only
    beam,     // Full system
    cosmics,  // Full system
    endOfMajorTypeEnum
  };

  enum Type {
    // daq
    daqTest=0x10*daq,
    endOfDaqTypeEnum,

    // crc
    crcTest=0x10*crc,
    crcNoise,
    crcBeParameters,
    crcFeParameters,
    crcIntDac,
    crcIntDacScan,
    crcExtDac,
    crcExtDacScan,
    crcFakeEvent,
    crcModeTest,

    // trg
    trgTest=0x10*trg,
    trgReadout,
    trgParameters,
    trgNoise,
    trgSpill,
    trgBeam,
    trgCosmics,

    // emc
    emcTest=0x10*emc,
    emcNoise,
    emcFeParameters,
    emcVfeDac,
    emcVfeDacScan,
    emcVfeHoldScan,
    emcTrgTiming,
    emcTrgTimingScan,
    emcBeam,
    emcBeamHoldScan,
    emcCosmics,
    emcCosmicsHoldScan,

    // ahc
    ahcTest=0x10*ahc,
    ahcDacScan,
    ahcCmNoise,
    ahcPmNoise,
    ahcAnalogOut,
    ahcCmAsic,
    ahcCmAsicVcalibScan,
    ahcCmAsicHoldScan,
    ahcPmAsic,
    ahcPmAsicVcalibScan,
    ahcPmAsicHoldScan,
    ahcCmLed,
    ahcCmLedVcalibScan,
    ahcCmLedHoldScan,
    ahcPmLed,
    ahcPmLedVcalibScan,
    ahcPmLedHoldScan,
    ahcScintillatorHoldScan,
    ahcBeam,
    ahcBeamHoldScan,
    ahcBeamStage,
    ahcBeamStageScan,
    ahcCosmics,
    ahcCosmicsHoldScan,
    ahcExpert,
    ahcGain,

    // dhc
    dhcTest=0x10*dhc,
    dhcNoise,
    dhcBeam,
    dhcCosmics,

    // tcm
    tcmTest=0x10*tcm,
    tcmNoise,
    tcmBeam,
    tcmBeamHoldScan,
    tcmCalLed,
    tcmPhysLed,
    tcmCalPedestal,
    tcmPhysPedestal,
    tcmCosmics,
    tcmCosmicsHoldScan,

    // bml
    bmlTest=0x10*bml,
    bmlNoise,
    bmlInternalTest,
    bmlBeam,

    // slow
    slowTest=0x10*slow,
    slowMonitor,

    // beam
    beamTest=0x10*beam,
    beamNoise,
    beamData,
    beamHoldScan,
    beamStage,
    beamStageScan,

    // cosmics
    cosmicsTest=0x10*cosmics,
    cosmicsNoise,
    cosmicsData,
    cosmicsHoldScan,

    endOfTypeEnum
  };


  DaqRunType();
  DaqRunType(Type t, unsigned char m=0, unsigned char v=0);
  DaqRunType(std::string s, unsigned char m=0, unsigned char v=0);

  Type type() const;
  void type(Type s);

  MajorType majorType() const;

  std::string        typeName() const;
  static std::string typeName(Type t);

  std::string        majorTypeName() const;
  static std::string majorTypeName(MajorType t);

  bool        knownType() const;
  static bool knownType(Type t);

  static Type typeNumber(std::string s);

  unsigned char printLevel() const;
  void          printLevel(unsigned char m);

  unsigned char switches() const;
  void          switches(unsigned char m);

  bool bufferRun() const;
  void bufferRun(bool b);

  bool spillRun() const;
  void spillRun(bool b);

  bool transferRun() const;
  void transferRun(bool b);

  bool histogramRun() const;
  void histogramRun(bool b);

  bool ascWriteRun() const;
  void ascWriteRun(bool b);

  bool writeRun() const;
  void writeRun(bool b);

  unsigned char version() const;
  void          version(unsigned char v);

  unsigned char        defaultVersion() const;
  static unsigned char defaultVersion(Type t);

  UtlPack data() const;
  void    data(UtlPack n);

  bool beamType() const;
  bool cosmicsType() const;

  std::ostream& print(std::ostream &o, std::string s="") const;


private:
  static const std::string _majorTypeName[endOfMajorTypeEnum];
  static const std::string _typeName[endOfTypeEnum];
  static const unsigned char _defaultVersion[endOfTypeEnum];

  // Byte 0 = version number
  // Byte 1 = bit switches (bits 8-10, 13-15)
  // Byte 2 = print level
  // Byte 3 = type (major type = bits 28-31)
  UtlPack _data;
};


#ifdef CALICE_DAQ_ICC


DaqRunType::DaqRunType() : _data(0) {
}

DaqRunType::DaqRunType(Type t, unsigned char m, unsigned char v) :
  _data(0) {
  type(t);
  switches(m);
  version(v);
}

DaqRunType::DaqRunType(std::string s, unsigned char m, unsigned char v) :
  _data(0) {
  type(typeNumber(s));
  switches(m);
  version(v);
}

DaqRunType::Type DaqRunType::type() const {
  return (Type)_data.byte(3);
}

void DaqRunType::type(Type n) {
  _data.byte(3,(unsigned char)n);
  _data.byte(0,defaultVersion(n));
}

DaqRunType::MajorType DaqRunType::majorType() const {
  MajorType m((MajorType)_data.halfByte(7));
  if(m==ahc1) m=ahc; // Catch second range of AHCAL
  return m;
}

std::string DaqRunType::typeName() const {
  return typeName(type());
}
 
std::string DaqRunType::typeName(Type t) {
  if(t<endOfTypeEnum) return _typeName[t];
  return "unknown";
}

std::string DaqRunType::majorTypeName() const {
  return majorTypeName(majorType());
}
 
std::string DaqRunType::majorTypeName(MajorType t) {
  if(t<endOfMajorTypeEnum) return _majorTypeName[t];
  return "unknown";
}

bool DaqRunType::knownType() const {
  return knownType(type());
}

bool DaqRunType::knownType(Type t) {
  return typeName(t)!="unknown";
}

DaqRunType::Type DaqRunType::typeNumber(std::string s) {
  for(unsigned t(0);t<endOfTypeEnum;t++) {
    if(s==typeName((Type)t)) return (Type)t;
  }
  return (Type)255;
}

unsigned char DaqRunType::printLevel() const {
  return _data.byte(2);
}

void DaqRunType::printLevel(unsigned char m) {
  _data.byte(2,m);
}

unsigned char DaqRunType::switches() const {
  return _data.byte(1);
}

void DaqRunType::switches(unsigned char m) {
  _data.byte(1,m);
}

bool DaqRunType::bufferRun() const {
  return _data.bit(8);
}

void DaqRunType::bufferRun(bool b) {
  return _data.bit(8,b);
}

bool DaqRunType::spillRun() const {
  return _data.bit(9);
}

void DaqRunType::spillRun(bool b) {
  return _data.bit(9,b);
}

bool DaqRunType::transferRun() const {
  return _data.bit(10);
}

void DaqRunType::transferRun(bool b) {
  return _data.bit(10,b);
}

bool DaqRunType::histogramRun() const {
  return _data.bit(13);
}

void DaqRunType::histogramRun(bool b) {
  return _data.bit(13,b);
}

bool DaqRunType::ascWriteRun() const {
  return _data.bit(14);
}

void DaqRunType::ascWriteRun(bool b) {
  return _data.bit(14,b);
}

bool DaqRunType::writeRun() const {
  return _data.bit(15);
}

void DaqRunType::writeRun(bool b) {
  return _data.bit(15,b);
}

unsigned char DaqRunType::version() const {
  return _data.byte(0);
}

void DaqRunType::version(unsigned char v) {
  _data.byte(0,v);
}

unsigned char DaqRunType::defaultVersion() const {
  return defaultVersion(type());
}
 
unsigned char DaqRunType::defaultVersion(Type t) {
  if(t<endOfTypeEnum) return _defaultVersion[t];
  return 0;
}

UtlPack DaqRunType::data() const {
  return _data;
}
 
void DaqRunType::data(UtlPack d) {
  _data=d;
}

bool DaqRunType::beamType() const {
  Type t(type());
  return
    t==DaqRunType::trgBeam ||
    
    t==DaqRunType::emcBeam ||
    t==DaqRunType::emcBeamHoldScan ||
    
    t==DaqRunType::ahcBeam ||
    t==DaqRunType::ahcBeamHoldScan ||
    t==DaqRunType::ahcBeamStage ||
    t==DaqRunType::ahcBeamStageScan ||
    
    t==DaqRunType::dhcBeam ||
    
    t==DaqRunType::tcmBeam ||
    t==DaqRunType::tcmBeamHoldScan ||
    
    t==DaqRunType::bmlBeam ||
    
    t==DaqRunType::beamTest ||
    t==DaqRunType::beamNoise ||
    t==DaqRunType::beamData ||
    t==DaqRunType::beamHoldScan ||
    t==DaqRunType::beamStage ||
    t==DaqRunType::beamStageScan;
}

bool DaqRunType::cosmicsType() const {
  Type t(type());
  return
    t==DaqRunType::trgCosmics ||
    
    t==DaqRunType::emcCosmics ||
    t==DaqRunType::emcCosmicsHoldScan ||
    
    t==DaqRunType::ahcCosmics ||
    t==DaqRunType::ahcCosmicsHoldScan ||
    
    t==DaqRunType::dhcCosmics ||
    
    t==DaqRunType::tcmCosmics ||
    t==DaqRunType::tcmCosmicsHoldScan ||
    
    t==DaqRunType::cosmicsTest ||
    t==DaqRunType::cosmicsNoise ||
    t==DaqRunType::cosmicsData ||
    t==DaqRunType::cosmicsHoldScan;
}

std::ostream& DaqRunType::print(std::ostream &o, std::string s) const {
  o << s << "DaqRunType::print()" << std::endl;

  o << s << " Data = " << printHex(_data) << std::endl;

  o << s << "  Type    = " << std::setw(3) << type()
    << " = " << typeName();
  if(beamType()) o << " = beam type";
  if(cosmicsType()) o << " = cosmics type";
  o << ", Major type = " << std::setw(3) << majorType()
    << " = " << majorTypeName() << std::endl;

  o << s << "  Version = " << std::setw(5) << (unsigned)version()
    << " (default = " << std::setw(5) << (unsigned)defaultVersion() << ")";
  o << ", Print level = " << std::setw(3) << (unsigned)printLevel() << std::endl;

  o << s << "  Switches  = " << printHex(switches()) << std::endl;

  if(majorType()!=slow) {
    //o << s << "   Data run" << std::endl;
    if(bufferRun()) {
      o << s << "   Buffering data run" << std::endl;
      if(spillRun())    o << s << "   Multi-event spill run" << std::endl;
      else              o << s << "   Single-event spill run" << std::endl;
      if(transferRun()) o << s << "   Multi-event transfer run" << std::endl;
      else              o << s << "   Single-event transfer run" << std::endl;
    } else {
      o << s << "   Single event data run" << std::endl;
    }
  } else {
    o << s << "   Slow monitoring run" << std::endl;
  }

  if(writeRun()) {
    //o << s << "   Write run" << std::endl;
    if(ascWriteRun()) o << s << "   Ascii write run" << std::endl;
    else              o << s << "   Binary write run" << std::endl;
  } else {
    o << s << "   Dummy write run" << std::endl;
  }

  if(histogramRun()) o << s << "   Histogram filling enabled" << std::endl;
  else               o << s << "   Histogram filling disabled" << std::endl;

  return o;
}

const std::string DaqRunType::_majorTypeName[]={
  "daq",
  "crc",
  "trg",
  "emc",
  "ahc",
  "ahc",
  "dhc",
  "tcm",
  "bml",
  "slow",
  "beam",
  "cosmics"
};

const std::string DaqRunType::_typeName[]={
  "daqTest",
  "unknown","unknown","unknown","unknown","unknown",
  "unknown","unknown","unknown","unknown","unknown",
  "unknown","unknown","unknown","unknown","unknown",

  "crcTest",
  "crcNoise",
  "crcBeParameters",
  "crcFeParameters",
  "crcIntDac",
  "crcIntDacScan",
  "crcExtDac",
  "crcExtDacScan",
  "crcFakeEvent",
  "crcModeTest",
  "unknown",
  "unknown","unknown","unknown","unknown","unknown",

  "trgTest",
  "trgReadout",
  "trgParameters",
  "trgNoise",
  "trgSpill",
  "trgBeam",
  "trgCosmics",
  "unknown","unknown","unknown","unknown",
  "unknown","unknown","unknown","unknown","unknown",
  
  "emcTest",
  "emcNoise",
  "emcFeParameters",
  "emcVfeDac",
  "emcVfeDacScan",
  "emcVfeHoldScan",
  "emcTrgTiming",
  "emcTrgTimingScan",
  "emcBeam",
  "emcBeamHoldScan",
  "emcCosmics",
  "emcCosmicsHoldScan",
  "unknown","unknown","unknown","unknown",

  "ahcTest",
  "ahcDacScan",
  "ahcCmNoise",
  "ahcPmNoise",
  "ahcAnalogOut",
  "ahcCmAsic",
  "ahcCmAsicVcalibScan",
  "ahcCmAsicHoldScan",
  "ahcPmAsic",
  "ahcPmAsicVcalibScan",
  "ahcPmAsicHoldScan",
  "ahcCmLed",
  "ahcCmLedVcalibScan",
  "ahcCmLedHoldScan",
  "ahcPmLed",
  "ahcPmLedVcalibScan",

  "ahcPmLedHoldScan",
  "ahcScintillatorHoldScan",
  "ahcBeam",
  "ahcBeamHoldScan",
  "ahcBeamStage",
  "ahcBeamStageScan",
  "ahcCosmics",
  "ahcCosmicsHoldScan",
  "ahcExpert",
  "ahcGain",
  "unknown",
  "unknown","unknown","unknown","unknown","unknown",

  "dhcTest",
  "dhcNoise",
  "dhcBeam",
  "dhcCosmics",
  "unknown","unknown",
  "unknown","unknown","unknown","unknown","unknown",
  "unknown","unknown","unknown","unknown","unknown",

  "tcmTest",
  "tcmNoise",
  "tcmBeam",
  "tcmBeamHoldScan",
  "tcmCalLed",
  "tcmPhysLed",
  "tcmCalPedestal",
  "tcmPhysPedestal",
  "tcmCosmics",
  "tcmCosmicsHoldScan",
  "unknown",
  "unknown","unknown","unknown","unknown","unknown",

  "bmlTest",
  "bmlNoise",
  "bmlInternalTest",
  "bmlBeam",
  "unknown","unknown",
  "unknown","unknown","unknown","unknown","unknown",
  "unknown","unknown","unknown","unknown","unknown",

  "slowTest",
  "slowMonitor",
  "unknown","unknown","unknown","unknown",
  "unknown","unknown","unknown","unknown","unknown",
  "unknown","unknown","unknown","unknown","unknown",

  "beamTest",
  "beamNoise",
  "beamData",
  "beamHoldScan",
  "beamStage",
  "beamStageScan",
  "unknown","unknown","unknown","unknown","unknown",
  "unknown","unknown","unknown","unknown","unknown",

  "cosmicsTest",
  "cosmicsNoise",
  "cosmicsData",
  "cosmicsHoldScan"
};

const unsigned char DaqRunType::_defaultVersion[]={
  // daq
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  // crc
    0,  5,  4,  2, 16, 63, 32, 63,  0,  0,  0,  0,  0,  0,  0,  0,
  // trg
    0, 12,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  // emc
    0,  0,  0,  4, 31, 31,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  // ahc
    0,  0,  0,  0,  1, 32, 10,255, 32, 10, 63, 38,  0, 30, 50, 30,
   65,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  // dhc
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  // tcm
    0,  0, 13, 13,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  // bml
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  // slow
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  // beam
    0,  0, 13, 13, 13, 13,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  // cosmics
    0,  0, 13, 13
};

#endif
#endif
