-- hds header_start
--
-- VHDL Architecture FED_DATA_FORMAT.data_format.fed_data_format
--
-- Created:
--          by - Emlyn Corrin
--          at - 12:16:52 01/23/03
--
-- Generated by Mentor Graphics' HDL Designer(TM) 2002.1a (Build 22)
--
-- Performs CMS DAQ Common Data Format Header and Trailer formatting
-- on the FED events.
-- The code implements the Common Data Format as specified at the
-- Readout Unit Working Group (RUWG) meeting of the 4 June 2003.
-- Some fields are still undefined, and will be discussed at future
-- RUWG meetings.
-- The optional second header and second trailer words are not
-- generated, as the contents of these have not yet been decided.
--
-- hds header_end
LIBRARY ieee;
USE ieee.std_logic_1164.ALL;


ENTITY data_format IS
   GENERIC( 
      LENGTH_START : boolean := false
   );
   PORT( 
      clk            : IN     std_logic;
      reset          : IN     std_logic;
      data_bx_id     : IN     std_logic_vector (11 DOWNTO 0);
      data_evt_lgth  : IN     std_logic_vector (23 DOWNTO 0);
      data_evt_stat  : IN     std_logic_vector (7 DOWNTO 0);
      data_evt_ty    : IN     std_logic_vector (3 DOWNTO 0);
      data_in        : IN     std_logic_vector (63 DOWNTO 0);
      data_lv1_id    : IN     std_logic_vector (23 DOWNTO 0);
      data_source_id : IN     std_logic_vector (11 DOWNTO 0);
      data_in_en     : IN     std_logic;
      start          : IN     std_logic;
      data_out       : OUT    std_logic_vector (63 DOWNTO 0);
      data_out_en    : OUT    std_logic;
      ctrl           : OUT    std_logic;
      ready          : OUT    std_logic;
      stop           : IN     std_logic
   );

-- Declarations

END data_format ;

-- hds interface_end

LIBRARY ieee;
USE ieee.std_logic_arith.ALL;
USE ieee.std_logic_unsigned.ALL;

-- Exactly ONE of the following USE lines should be uncommented.
-- The final implementation uses the CCITT (also known as USB) version.
-- The X25 version was a possible candidate, but is not being used.
-- The SIMPLE and NULL versions are for use during testing and should NOT
--     be used for any production code.

-- modified to use the back end fpga library

LIBRARY back_end_fpga;

USE back_end_fpga.PCK_CRC16_D64_CCITT.ALL;
-- USE FED_DATA_FORMAT.PCK_CRC16_D64_CCITT.ALL;
-- USE FED_DATA_FORMAT.PCK_CRC16_D64_X25.ALL;
-- USE FED_DATA_FORMAT.PCK_CRC16_D64_SIMPLE.ALL;
-- USE FED_DATA_FORMAT.PCK_CRC16_D64_NULL.ALL;

ARCHITECTURE data_format OF data_format IS

  -- Constants that define regions within the header and trailer words.
  -- The actual values have no meaning and are not used.
  CONSTANT r_boe:    std_logic_vector(63 DOWNTO 60) := (OTHERS => 'X');
  CONSTANT r_evt_ty: std_logic_vector(59 DOWNTO 56) := (OTHERS => 'X');
  CONSTANT r_lv1_id: std_logic_vector(55 DOWNTO 32) := (OTHERS => 'X');
  CONSTANT r_bx_id:  std_logic_vector(31 DOWNTO 20) := (OTHERS => 'X');
  CONSTANT r_src_id: std_logic_vector(19 DOWNTO  8) := (OTHERS => 'X');
  CONSTANT r_fov:    std_logic_vector( 7 DOWNTO  4) := (OTHERS => 'X');
  CONSTANT r_last:   std_logic_vector( 3 DOWNTO  3) := (OTHERS => 'X');

  CONSTANT r_eoe:        std_logic_vector(63 DOWNTO 60) := (OTHERS => 'X');
  CONSTANT r_evt_lgth:   std_logic_vector(55 DOWNTO 32) := (OTHERS => 'X');
  CONSTANT r_crc:        std_logic_vector(31 DOWNTO 16) := (OTHERS => 'X');
  CONSTANT r_evt_stat:   std_logic_vector(11 DOWNTO  4) := (OTHERS => 'X');

  -- Constants that define the value of parts of the header and trailer words.
--  CONSTANT c_boe1:   std_logic_vector := "0001";  -- Beginning of event, first word
  CONSTANT c_boe1:   std_logic_vector := "0101";  -- set to 0101 by Saeed on 05-02-04 requested by IC
  CONSTANT c_boe2:   std_logic_vector := "0010";  -- Beginning of event, second word
  CONSTANT c_fov:    std_logic_vector := "0001";  -- Format version
  CONSTANT c_last:   std_logic_vector := "1";     -- Last header/trailer word
  CONSTANT c_nlast:  std_logic_vector := "0";     -- Not last header/trailer word
--  CONSTANT c_eoe1:   std_logic_vector := "1001";  -- End of event, last word
  CONSTANT c_eoe1:   std_logic_vector := "1010";  -- set to 0101 by Saeed on 05-02-04 requested by IC
  CONSTANT c_eoe2:   std_logic_vector := "1010";  -- End of event, second-to-last word

  -- The initial value for the CRC calculation.
  -- Any value is equally good (except perhaps all zeroes)
  --     but it must be fixed once it has been chosen.
  CONSTANT CRC_INIT: std_logic_vector := "1111111111111111";
  
  -- registers to store the current calculated CRC value
  -- and other information needed for the trailer.
  SIGNAL crc: std_logic_vector(15 DOWNTO 0);
  SIGNAL save_evt_stat: std_logic_vector(data_evt_stat'range);
  SIGNAL save_evt_lgth: std_logic_vector(data_evt_lgth'range);
  
  -- The possible states of the state-machine
  -- and a register to hold the current state.
  TYPE t_state IS (idle, writing, trailer, trailer_crc);
  SIGNAL state: t_state;

BEGIN

  builder: PROCESS(clk, reset)
    -- If the length is provided at the start of the event, then this
	-- counts down to zero to detect the end of the event.
	-- Otherwise it is not used and should get optimised away.
    VARIABLE data_counter: unsigned(data_evt_lgth'range);
	-- Used to build up the header/trailer words, and to buffer
	-- the data for one clock-cycle so that there is no gap between
	-- the last data word and the trailer while the CRC is being calculated.
    VARIABLE temp_data: std_logic_vector(63 DOWNTO 0);
	-- Used to store the control bit associated with the above data word
	VARIABLE temp_ctrl: std_logic;
	-- Indicates whether the above two registers contain valid data.
	VARIABLE data_available: boolean;
  BEGIN    
    IF (reset = '1') THEN
      -- reset signals to default:
      data_counter   := conv_unsigned(0, data_counter'length);
	  data_available := false;
      state       <= idle AFTER 1 ns;
      crc         <= CRC_INIT AFTER 1 ns;
      ready       <= '0' AFTER 1 ns;
      ctrl        <= '0' AFTER 1 ns;
      data_out_en <= '0' AFTER 1 ns;
      data_out    <= (OTHERS => 'Z') AFTER 1 ns;
      save_evt_stat <= (OTHERS => '0') AFTER 1 ns;
      save_evt_lgth <= (OTHERS => '0') AFTER 1 ns;
    ELSIF (clk = '1' AND clk'event) THEN
      -- default assignments:
      ready       <= '0' AFTER 1 ns;
      ctrl        <= '0' AFTER 1 ns;
      data_out_en <= '0' AFTER 1 ns;
      data_out    <= (OTHERS => 'Z') AFTER 1 ns;
      CASE state IS
      WHEN idle =>
	    -- The default state, waiting for a start signal.
		-- Check that no data arrives before a start signal is detected.
       	ASSERT (data_in_en /= '1') REPORT "(Common Data Format) Unexpected DATA_IN_EN" SEVERITY error;
        IF (start = '1') THEN
          -- build first word (header)
          temp_data := (OTHERS => '0');      -- Initialise all bits to zero first
          temp_data(r_boe'range)    := c_boe1;
          temp_data(r_evt_ty'range) := data_evt_ty;
          temp_data(r_lv1_id'range) := data_lv1_id;
          temp_data(r_bx_id'range)  := data_bx_id;
          temp_data(r_src_id'range) := data_source_id;
          temp_data(r_fov'range)    := c_fov;
          temp_data(r_last'range)   := c_last;  -- will be c_nlast if there is another header word
          temp_ctrl := '1';
		  data_available := true;
          crc <= nextCRC16_D64(temp_data, crc) AFTER 1 ns;
          -- Save data needed for trailer word
          IF (LENGTH_START) THEN
            save_evt_stat <= data_evt_stat AFTER 1 ns;
            save_evt_lgth <= data_evt_lgth + 2 AFTER 1 ns;
            data_counter := unsigned(data_evt_lgth);
            IF (data_counter = 0) THEN
              state <= trailer AFTER 1 ns;
            ELSE
              ASSERT (stop /= '1') REPORT "(Common Data Format) Unexpected STOP" SEVERITY error;
              state <= writing AFTER 1 ns;
            END IF;
          ELSE
            IF (stop = '1') THEN
              save_evt_stat <= data_evt_stat AFTER 1 ns;
              save_evt_lgth <= data_evt_lgth + 2 AFTER 1 ns;
              state <= trailer AFTER 1 ns;
            ELSE
              state <= writing AFTER 1 ns;
            END IF;
          END IF;
        ELSE
          ASSERT (stop /= '1') REPORT "(Common Data Format) Unexpected STOP" SEVERITY error;
          ready <= '1' AFTER 1 ns;
        END IF;
      WHEN writing =>
	    -- an event is being sent, keep updating the CRC and checking for the end
       	ASSERT (start /= '1') REPORT "(Common Data Format) Unexpected START" SEVERITY error;
		IF (data_available) THEN
	      data_out    <= temp_data AFTER 1 ns;
		  ctrl        <= temp_ctrl AFTER 1 ns;
          data_out_en <= '1' AFTER 1 ns;
		  data_available := false;
		END IF;
        IF (data_in_en = '1') THEN
		  temp_data := data_in;
		  temp_ctrl := '0';
		  data_available := true;
          crc <= nextCRC16_D64(data_in, crc) AFTER 1 ns;
        END IF;
        IF (LENGTH_START) THEN
          IF (data_in_en = '1') THEN
            data_counter := data_counter - 1;
          END IF;
          IF (data_counter = 0) THEN
            state <= trailer AFTER 1 ns;
          ELSE
            ASSERT (stop /= '1') REPORT "(Common Data Format) Unexpected STOP" SEVERITY error;
          END IF;
        ELSE
          IF (stop = '1') THEN
            save_evt_stat <= data_evt_stat AFTER 1 ns;
            save_evt_lgth <= data_evt_lgth + 2 AFTER 1 ns;
            state <= trailer AFTER 1 ns;
          END IF;
        END IF;
      WHEN trailer =>
	    -- end of the event, build the trailer word (except CRC) and calculate the final CRC value.
       	ASSERT (data_in_en /= '1') REPORT "(Common Data Format) Unexpected DATA_IN_EN" SEVERITY error;
       	ASSERT (start /= '1') REPORT "(Common Data Format) Unexpected START" SEVERITY error;
       	ASSERT (stop /= '1') REPORT "(Common Data Format) Unexpected STOP" SEVERITY error;
		IF (data_available) THEN
		  data_out    <= temp_data after 1 ns;
		  ctrl        <= temp_ctrl after 1 ns;
		  data_out_en <= '1' after 1 ns;
		  data_available := false;
		END IF;
        -- build second word (trailer)
        temp_data := (OTHERS => '0');
        temp_data(r_eoe'range)      := c_eoe1;
        temp_data(r_evt_lgth'range) := save_evt_lgth;
        temp_data(r_evt_stat'range) := save_evt_stat;
        temp_data(r_last'range)     := c_last;  -- will be c_nlast in preceding trailer word
		temp_ctrl := '1';
        ready <= '1' AFTER 1 ns;
        crc <= nextCRC16_D64(temp_data, crc) AFTER 1 ns;
        state <= trailer_crc after 1 ns;
      WHEN trailer_crc =>
	    -- now the CRC is known, insert it into the trailer and send the trailer word to the output.
       	ASSERT (data_in_en /= '1') REPORT "(Common Data Format) Unexpected DATA_IN_EN" SEVERITY error;
       	ASSERT (start /= '1') REPORT "(Common Data Format) Unexpected START" SEVERITY error;
       	ASSERT (stop /= '1') REPORT "(Common Data Format) Unexpected STOP" SEVERITY error;
        temp_data(r_crc'range) := crc;
        data_out    <= temp_data AFTER 1 ns;
		ctrl        <= temp_ctrl AFTER 1 ns;
        data_out_en <= '1' AFTER 1 ns;
        state       <= idle AFTER 1 ns;
        ready       <= '1' AFTER 1 ns;
        crc         <= CRC_INIT AFTER 1 ns;
      END CASE;
    END IF;
  END PROCESS;

END data_format;
